/**
* @file    GdkParamPointSetRegion.h
* @brief   Declares the GdkPointSetRegion interface.
*
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef GDK_EXT_PARAM_POINT_SET_REGION_H
#define GDK_EXT_PARAM_POINT_SET_REGION_H

#include <Gdk/GdkDef.h>
#include <Gdk/Config/GdkParam.h>
#include <Gdk/Config/GdkParamInfo.h>

typedef kPointer GdkPointSetRegionParam;

/**
* Sets the constant Z value which will then be used for all points, and will not be modifiable by client.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @param    constantZ   Z value to use for all points.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamPointSetRegion_SetConstantZ(GdkPointSetRegionParam param, k64f constantZ);

/**
* Gets the current constant Z value.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @return               Currently used constant-z value.
*/
GdkFx(k64f) GdkParamPointSetRegion_ConstantZ(GdkPointSetRegionParam param);

/**
* Disables constant Z for the parameter. All points must have individual, valid Z-values
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamPointSetRegion_DisableConstantZ(GdkPointSetRegionParam param);

/**
* Sets the point shape mode for the parameter.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @param    shape       Shape to use when displaying points.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamPointSetRegion_SetPointShape(GdkPointSetRegionParam param, kMarkerShape shape);

/**
* Gets the point shape mode for the parameter.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @return               Current point-shape setting.
*/
GdkFx(kMarkerShape) GdkParamPointSetRegion_PointShape(GdkPointSetRegionParam param);

/**
* Sets the point size for the parameter.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @param    size        Size of points, in pixels, to display.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamPointSetRegion_SetPointSize(GdkPointSetRegionParam param, k16u size);

/**
* Gets the point size for the parameter.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @return               Current point-size of the parameter.
*/
GdkFx(k16u) GdkParamPointSetRegion_PointSize(GdkPointSetRegionParam param);

/**
* Sets the point colour for the parameter.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @param    color       Color to display points.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamPointSetRegion_SetPointColor(GdkPointSetRegionParam param, GdkPointSetRegionColor color);

/**
* Gets the point colour for the parameter.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @return               Current point color setting.
*/
GdkFx(GdkPointSetRegionColor) GdkParamPointSetRegion_PointColor(GdkPointSetRegionParam param);

/**
* Sets the line colour for the parameter.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @param    color       Color to display lines (if shown).
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamPointSetRegion_SetLineColor(GdkPointSetRegionParam param, GdkPointSetRegionColor color);

/**
* Gets the line colour for the parameter.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @return               Current line color setting.
*/
GdkFx(GdkPointSetRegionColor) GdkParamPointSetRegion_LineColor(GdkPointSetRegionParam param);

/**
* Sets the point connectedness mode for the parameter
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @param    mode        Mode with which to connect points (or not).
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamPointSetRegion_SetMode(GdkPointSetRegionParam param, GdkPointSetRegionMode mode);

/**
* Gets the point connectedness mode for the parameter
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @return               Mode with which to connect points (or not).
*/
GdkFx(GdkPointSetRegionMode) GdkParamPointSetRegion_Mode(GdkPointSetRegionParam param);

/**
* Sets the projection on or off for the parameter.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @param    showProjection  Show the projection of the point set or not.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamPointSetRegion_SetProjection(GdkPointSetRegionParam param, kBool showProjection);

/**
* Gets the projection on or off setting for the parameter.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @return               Current state of show-projection for the point set.
*/
GdkFx(kBool) GdkParamPointSetRegion_Projection(GdkPointSetRegionParam param);

/**
* Sets the maximum number of points supported by the parameter.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @param    maxPoints   Maximum number of points to be allowed.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamPointSetRegion_SetMaxPoints(GdkPointSetRegionParam param, k32u maxPoints);

/**
* Gets the maximum number of points supported by the parameter.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @return               Current max points setting.
*/
GdkFx(k32u) GdkParamPointSetRegion_MaxPoints(GdkPointSetRegionParam param);

/**
* Sets the minimum number of points supported by the parameter.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @param    minPoints   Minimum number of points to be allowed.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamPointSetRegion_SetMinPoints(GdkPointSetRegionParam param, k32u minPoints);

/**
* Gets the minimum number of points supported by the parameter.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @return               Current minimum points setting for the parameter.
*/
GdkFx(k32u) GdkParamPointSetRegion_MinPoints(GdkPointSetRegionParam param);


/**
* Sets the number of points for the parameter. Must be between (or equal to) the current min and max settings.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @param    count       Number of points.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamPointSetRegion_SetPointCount(GdkPointSetRegionParam param, kSize count);

/**
* Gets the number of points for the parameter.
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @return               Current point count.
*/
GdkFx(kSize) GdkParamPointSetRegion_PointCount(GdkPointSetRegionParam param);

/**
* Sets the value of a point within the parameter. Index must be in the range (0, GdkParamPointSetRegion_PointCount - 1)
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @param    index       Index of point to be set
* @param    point       Pointer to value to copy into point set.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamPointSetRegion_SetPointAt(GdkPointSetRegionParam param, kSize index, const kPoint3d64f* point);

/**
* Gets the value of a point within the parameter. Index must be in the range (0, GdkParamPointSetRegion_PointCount - 1)
*
* @public               @memberof GdkParamPointSetRegion
* @param    param       Param object.
* @param    index       Index of point to be retrieved.
* @return               Const pointer to point value.
*/
GdkFx(const kPoint3d64f*) GdkParamPointSetRegion_PointAt(GdkPointSetRegionParam param, kSize index);

#endif 
