/**
 * @file    GdkTool.h
 * @brief   Declares the GdkTool class.
 *
 * Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef GDK_TOOL_H
#define GDK_TOOL_H

#include <Gdk/GdkDef.h>
#include <Gdk/Tools/GdkToolInfo.h>
#include <Gdk/Config/GdkToolCfg.h>
#include <Gdk/Tools/GdkToolInput.h>
#include <Gdk/Tools/GdkToolOutput.h>
#include <Gdk/Utils/GdkSensorInfo.h>
#include <Gdk/Tools/GdkPointFeature.h>
#include <Gdk/Tools/GdkLineFeature.h>
#include <Gdk/Tools/GdkCircleFeature.h>
#include <Gdk/Tools/GdkPlaneFeature.h>

typedef struct GdkToolEnv
{
    GdkSensorInfo sensorInfo;
    kBool         isEmulator;       // DEPRECATED. Internal only -- do not use.
    kBool         isAccelerated;    // DEPRECATED. Internal only -- do not use.
} GdkToolEnv;

/**
* Base class for a tool implementation.
*
* @class       GdkTool
* @extends     kObject
* @ingroup     Gdk-Tools
*/
typedef kObject GdkTool;

/**
* Returns the tool configuration.
*
* @public               @memberof GdkTool
* @param   tool         Tool object.
* @return               Current configuration object.
*/
GdkFx(GdkToolCfg) GdkTool_Config(GdkTool tool);

/**
* Returns the message processing memory allocator. Use this allocator to
* allocate objects used while processing scan data messages 
* (ie. objects created within the VProcess() function).
* In cases where lots of memory need to be allocated/deallocated, the message
* allocator returned by this function can give better performance compared
* to using other allocators.
*
* @public               @memberof GdkTool
* @param   tool         Tool object.
* @return               Message processor allocator.
*/
GdkFx(kAlloc) GdkTool_MessageAlloc(GdkTool tool);

/**
* Returns the sensor information.
*
* @public               @memberof GdkTool
* @param   tool         Tool object.
* @return               Current sensor info.
*/
GdkFx(GdkSensorInfo) GdkTool_SensorInfo(GdkTool tool);

/**
* Returns the index for the named input.
*
* @public               @memberof GdkTool
* @param   tool         Tool object.
* @param   name         String name of input data set.
* @return               Index of input data with matching name;
*                       index 0 is reserved for the primary data inputs, including BOTTOM/RIGHT.
*                       (see #GDK_DATA_PRIMARY_INPUT_INDEX).
*                       If available, index >= #GDK_DATA_SECONDARY_INPUT_START_INDEX is reserved for the secondary input data (eg. parameters, features etc). 
*/
GdkFx(kSize) GdkTool_InputDataIndex(GdkTool tool, const kChar* name);

/**
* Find the index of a measurement by its type name.
*
* If there are multiple measurements of the same type, only the index of
* first instance will be returned. A full list of measurements can be found
* in the tool config (see GdkTool_Config).
*
* @public               @memberof GdkTool
* @param   tool         Tool object.
* @param   typeName     Type name of the measurement.
* @return               Index of measurement. kSIZE_NULL if not found.
*/
GdkFx(kSize) GdkTool_MeasurementIndexByType(GdkTool tool, const kChar* typeName);

/**
* Find the index of a feature by its type name.
*
* If there are multiple features of the same type, only the index of
* first instance will be returned. A full list of features can be found
* in the tool config. See GdkTool_Config.
*
* The type name of the feature should not be confused with its data type.
* See GdkToolInfo_AddFeature.
*
* @public               @memberof GdkTool
* @param   tool         Tool object.
* @param   typeName     Type name of the feature.
* @return               Index of feature. kSIZE_NULL if not found.
*/
GdkFx(kSize) GdkTool_FeatureIndexByType(GdkTool tool, const kChar* typeName);

/**
* Find the index of a tool data output by its type name.
*
* If there are multiple tool data outputs of the same type, only the index of
* first instance will be returned. A full list of tool data outputs can be found
* in the tool config. See GdkTool_Config.
*
* The type name of the tool data output should not be confused with its data type.
* See GdkToolInfo_AddToolDataOutput.
*
* @public               @memberof GdkTool
* @param   tool         Tool object.
* @param   typeName     Type name of the tool data output.
* @return               Index of tool data output. kSIZE_NULL if not found.
*/
GdkFx(kSize) GdkTool_DataOutputIndexByType(GdkTool tool, const kChar* typeName);

/**
* Applies the currently available anchoring parameters to a profile region.
*
* @public               @memberof GdkTool
* @param   tool         Tool object.
* @param   original     Pointer to original region.
* @param   anchored     Pointer to receive the anchor-adjusted region.
* @return               Operation status.
*/
GdkFx(kStatus) GdkTool_AnchorProfileRegion(GdkTool tool, const GdkRegionXZ64f* original, GdkRegionXZ64f* anchored);

/**
* Applies the currently available anchoring parameters to a surface region.
*
* @public               @memberof GdkTool
* @param   tool         Tool object.
* @param   original     Pointer to original region.
* @param   anchored     Pointer to receive the anchor-adjusted region.
* @return               Operation status.
*/
GdkFx(kStatus) GdkTool_AnchorSurfaceRegion(GdkTool tool, const GdkRegion3d64f* original, GdkRegion3d64f* anchored);

/**
* Applies the currently available anchoring parameters to a point set region.
*
* @public               @memberof GdkTool
* @param   tool         Tool object.
* @param   original     Pointer to original list of kPoint3d64f points.
* @param   anchored     Pointer to receive the anchor-adjusted list of points.
* @return               Operation status.
*/
GdkFx(kStatus) GdkTool_AnchorPointSetRegion(GdkTool tool, const kArrayList* original, kArrayList* anchored);

/**
* Set up the tool instance in preparation for sensor scanning.
*
* @public               @memberof GdkTool
* @param   tool         Tool object.
* @param   sensorInfo   Sensor info specific for this tool instance.
* @param   toolCfg      Tool configuration data for this instance.
* @return               Operation status.
*/
GdkFx(kStatus) GdkTool_SetupForScanning(GdkTool tool, GdkSensorInfo sensorInfo, GdkToolCfg toolCfg);

/**
* Determines whether or not this tools should show up in the list of tools.
*
* This function can be used to alter the availability of the sensor based
* on specific user needs.
*
* @public               @memberof GdkTool
* @param    env         Tool environment.
* @return               Operation status.
*/
GdkFx(kBool) GdkTool_VIsVisible(const GdkToolEnv* env);

/**
* Initializes newly added tool instance's configuration data.
*
* @public               @memberof GdkTool
* @param    tool        GDK tool instance
* @param    toolConfig  Configuration object.
* @return               Operation status.
*/
GdkFx(kStatus) GdkTool_VNewToolConfigInstanced(GdkTool tool, GdkToolCfg toolConfig);

/**
* Initializes newly added tool intance's measurement configuration data.
*
* @public                       @memberof GdkTool
* @param    tool                GDK tool instance
* @param    measurementConfig   Measurement configuration object (to be updated).
* @return                       Operation status.
*/
GdkFx(kStatus) GdkTool_VNewMeasurementConfigInstanced(GdkTool tool, GdkMeasurementCfg measurementConfig);

/**
* Initializes newly added tool instance's feature configuration data.
*
* @public                       @memberof GdkTool
* @param    tool                GDK tool instance
* @param    featureConfig       Feature configuration object (to be updated).
* @return                       Operation status.
*/
GdkFx(kStatus) GdkTool_VNewFeatureConfigInstanced(GdkTool tool, GdkFeatureCfg featureConfig);

/**
* Initializes newly added tool instance's data output configuration data.
*
* @public                           @memberof GdkTool
* @param    tool                    GDK tool instance
* @param    toolDataOutputConfig    Data output configuration object (to be updated).
* @return                           Operation status.
*/
GdkFx(kStatus) GdkTool_VNewToolDataOutputConfigInstanced(GdkTool tool, GdkToolDataOutputCfg toolDataOutputConfig);

/**
* Update the tool instance's configuration due to configuration file changes.
*
* The framework calls this function after the configuration has changed. The
* tool implementation has the opportunity to update configuration attributes
* that may need to be dynamically updated. For example the implementation
* may use the value of a parameter to update the min and max constraints of
* another parameter.
* This function can be called for configuration changes not related to the tool instance.
*
* @public                   @memberof GdkTool
* @param    tool            GDK tool instance
* @param    toolConfig      Tool configuration object.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkTool_VUpdateConfigInstanced(GdkTool tool, GdkToolCfg toolConfig);

/**
* Calculates the tool instance's region for a given data output.
*
* Override this function to implement customized calculation of data regions for specific
* outputs. If this function is unimplemented, or does not alter the input region, a default
* region is used by the framework.
* This function can be called for configuration changes not related to the tool instance.
*
* @public                   @memberof GdkTool
* @param    tool            GDK tool instance
* @param    outputConfig    Data output configuration object.
* @param    region          Pointer to receive new region. The value is initialized to the default region.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkTool_VCalcDataOutputRegionInstanced(GdkTool tool, GdkToolDataOutputCfg outputConfig, GdkRegion3d64f* region);


/**
* Called by the framework before data collection starts.
*
* @public               @memberof GdkTool
* @param    tool        Tool object.
* @return               Operation status.
*/
GdkFx(kStatus) GdkTool_VStart(GdkTool tool);

/**
* Called by the framework before data collection stop.
*
* @public               @memberof GdkTool
* @param    tool        Tool object.
* @return               Operation status.
*/
GdkFx(kStatus) GdkTool_VStop(GdkTool tool);

/**
* Process an incoming frame.
*
* The tool implementation should process the incoming data and send out results
* within this function.
*
* @public               @memberof GdkTool
* @param    tool        Tool object.
* @param    input       Collection of inputs.
* @param    output      Collection of outputs.
* @return               Operation status.
*/
GdkFx(kStatus) GdkTool_VProcess(GdkTool tool, GdkToolInput input, GdkToolOutput output);


//// Deprecated API functions

/**
* [Deprecated] Use GdkTool_InputDataIndex() instead.
*
* Returns the index for the named feature.
*
* @deprecated
* @public               @memberof GdkTool
* @param   tool         Tool object.
* @param   name         String name of feature.
* @return               Index of feature with matching name.
*/
GdkFx(kSize) GdkTool_InputFeatureIndex(GdkTool tool, const kChar* name);

/**
* [Deprecated] Use GdkTool_VNewToolConfigInstanced() instead.
*
* Initializes newly added tool configuration.
*
* @deprecated
* @public               @memberof GdkTool
* @param    env         Tool environment.
* @param    toolConfig  Configuration object.
* @return               Operation status.
*/
GdkFx(kStatus) GdkTool_VNewToolConfig(const GdkToolEnv* env, GdkToolCfg toolConfig);

/**
* [Deprecated] Use GdkTool_VNewMeasurementConfigInstanced() instead.
*
* Initializes newly added tool measurement configuration.
*
* @deprecated
* @public                       @memberof GdkTool
* @param    env                 Tool environment.
* @param    toolConfig          Parent tool configuration.
* @param    measurementConfig   Measurement configuration object (to be updated).
* @return                       Operation status.
*/
GdkFx(kStatus) GdkTool_VNewMeasurementConfig(const GdkToolEnv* env, GdkToolCfg toolConfig, GdkMeasurementCfg measurementConfig);

/**
* [Deprecated] Use GdkTool_VNewFeatureConfigInstanced() instead.
*
* Initializes newly added tool feature configuration.
*
* @deprecated
* @public                       @memberof GdkTool
* @param    env                 Tool environment.
* @param    toolConfig          Parent tool configuration.
* @param    featureConfig       Feature configuration object (to be updated).
* @return                       Operation status.
*/
GdkFx(kStatus) GdkTool_VNewFeatureConfig(const GdkToolEnv* env, GdkToolCfg toolConfig, GdkFeatureCfg featureConfig);

/**
* [Deprecated] Use GdkTool_VNewToolDataOutputConfigInstanced() instead.
*
* Initializes newly added tool data output configuration.
*
* @deprecated
* @public                           @memberof GdkTool
* @param    env                     Tool environment.
* @param    toolConfig              Parent tool configuration.
* @param    toolDataOutputConfig    Data output configuration object (to be updated).
* @return                           Operation status.
*/
GdkFx(kStatus) GdkTool_VNewToolDataOutputConfig(const GdkToolEnv* env, GdkToolCfg toolConfig, GdkToolDataOutputCfg toolDataOutputConfig);

/**
* [Deprecated] Use GdkTool_VUpdateConfigInstanced() instead.
*
* Update the configuration due to parameter changes.
*
* The framework calls this function after the configuration has changed. The
* tool implementation has the opportunity to update configuration attributes
* that may need to be dynamically updated. For example the implementation
* may use the value of a parameter to update the min and max constraints of
* another parameter.
* This function can be called for configuration changes not related to the tool type.
*
* @deprecated
* @public                   @memberof GdkTool
* @param    env             Tool environment.
* @param    toolConfig      Tool configuration object.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkTool_VUpdateConfig(const GdkToolEnv* env, GdkToolCfg toolConfig);

/**
* [Deprecated] Use GdkTool_VCalcDataOutputRegionInstanced() instead.
*
* Calculates the region for a given data output.
*
* Override this function to implement customized calculation of data regions for specific
* outputs. If this function is unimplemented, or does not alter the input region, a default
* region is used by the framework.
* This function can be called for configuration changes not related to the tool type.
*
* @deprecated
* @public                   @memberof GdkTool
* @param    env             Tool environment.
* @param    toolConfig      Tool configuration object.
* @param    outputConfig    Data output configuration object.
* @param    region          Pointer to receive new region. The value is initialized to the default region.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkTool_VCalcDataOutputRegion(const GdkToolEnv* env, GdkToolCfg toolConfig, GdkToolDataOutputCfg outputConfig, GdkRegion3d64f* region);

#include <Gdk/Tools/GdkTool.x.h>

#endif
