/////////////////////////////////////////////////////////////////////////////
// TestSurfaceDynamicAlignment
// 2017-02-01 (C) LMI Technologies
//
// Aligns the input surface to a surface defined by the centers of three regions
/////////////////////////////////////////////////////////////////////////////

#include <GdkAppSample/TestSurfaceDynamicAlignment.h>
#include <kVision/Common/kNullMath.h>
#include <math.h>

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

kBeginClassEx(Tool, TestSurfaceDynamicAlignment)
    kAddVMethod(TestSurfaceDynamicAlignment, kObject, VRelease)
    kAddVMethod(TestSurfaceDynamicAlignment, GdkTool, VInit)
    kAddVMethod(TestSurfaceDynamicAlignment, GdkTool, VName)
    kAddVMethod(TestSurfaceDynamicAlignment, GdkTool, VDescribe)
    kAddVMethod(TestSurfaceDynamicAlignment, GdkTool, VNewMeasurementConfig)
    kAddVMethod(TestSurfaceDynamicAlignment, GdkTool, VUpdateConfig)
    kAddVMethod(TestSurfaceDynamicAlignment, GdkTool, VNewToolConfig)
    kAddVMethod(TestSurfaceDynamicAlignment, GdkTool, VStart)
    kAddVMethod(TestSurfaceDynamicAlignment, GdkTool, VStop)
    kAddVMethod(TestSurfaceDynamicAlignment, GdkTool, VProcess)
kEndClassEx()

ToolFx(const kChar*) TestSurfaceDynamicAlignment_VName()
{
    return TEST_SURFACE_DYNAMIC_ALIGNMENT_TOOL_NAME;
}

ToolFx(kStatus) TestSurfaceDynamicAlignment_VDescribe(GdkToolInfo toolInfo)
{
    GdkMeasurementInfo mmtInfo;
    GdkToolDataOutputInfo toolDataInfo;
    GdkRegion3d64f refRegion1 = { -12, -20, -10, 3, 3, 20 };
    GdkRegion3d64f refRegion2 = { 8, -10, -10, 3, 3, 20 };
    GdkRegion3d64f refRegion3 = { -8, 10, -10, 3, 3, 20 };

    kCheck(GdkToolInfo_SetLabel(toolInfo, TEST_SURFACE_DYNAMIC_ALIGNMENT_TOOL_LABEL));
    kCheck(GdkToolInfo_SetTypeName(toolInfo, TEST_SURFACE_DYNAMIC_ALIGNMENT_TOOL_NAME));
    kCheck(GdkToolInfo_EnableAutoVersion(toolInfo, kFALSE));

    kCheck(GdkToolInfo_SetSourceType(toolInfo, GDK_DATA_TYPE_UNIFORM_SURFACE));
    kCheck(GdkToolInfo_AddSourceOption(toolInfo, GDK_DATA_SOURCE_TOP));

    kCheck(GdkToolInfo_EnableAnchoring(toolInfo, GDK_ANCHOR_PARAM_X, kTRUE));
    kCheck(GdkToolInfo_EnableAnchoring(toolInfo, GDK_ANCHOR_PARAM_Y, kTRUE));
    kCheck(GdkToolInfo_EnableAnchoring(toolInfo, GDK_ANCHOR_PARAM_Z, kTRUE));

    // Input Parameters
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_SURFACE_REGION, PARAM_REGION1_NAME, PARAM_REGION1_LABEL, &refRegion1, kNULL));
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_SURFACE_REGION, PARAM_REGION2_NAME, PARAM_REGION2_LABEL, &refRegion2, kNULL));
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_SURFACE_REGION, PARAM_REGION3_NAME, PARAM_REGION3_LABEL, &refRegion3, kNULL));

    // Output Measurements
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, MEAS_XANGLE_NAME, MEAS_XANGLE_LABEL, &mmtInfo)); // #0
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, MEAS_YANGLE_NAME, MEAS_YANGLE_LABEL, &mmtInfo)); // #1
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, MEAS_ZTRANS_NAME, MEAS_ZTRANS_LABEL, &mmtInfo)); // #2

    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_UNIFORM_SURFACE, OUTPUT_ALIGNED_SURFACE_NAME, OUTPUT_ALIGNED_SURFACE_LABEL, &toolDataInfo)); // #3

    return kOK;
}

ToolFx(kStatus) TestSurfaceDynamicAlignment_VInit(TestSurfaceDynamicAlignment tool, kType type, kAlloc alloc)
{
    kObjR(TestSurfaceDynamicAlignment, tool);

    kCheck(GdkTool_VInit(tool, type, alloc));
    kZero(obj->dataSource);
    kZero(obj->regions);

    return kOK;
}

ToolFx(kStatus) TestSurfaceDynamicAlignment_VRelease(TestSurfaceDynamicAlignment tool)
{
    kObj(TestSurfaceDynamicAlignment, tool);

    return GdkTool_VRelease(tool);
}

ToolFx(kStatus) TestSurfaceDynamicAlignment_VNewToolConfig(const GdkToolEnv* env, GdkToolCfg toolConfig)
{
    return kOK;
}

ToolFx(kStatus) TestSurfaceDynamicAlignment_VNewMeasurementConfig(const GdkToolEnv* env, GdkToolCfg toolConfig, GdkMeasurementCfg measurementConfig)
{
    return kOK;
}

ToolFx(kStatus) TestSurfaceDynamicAlignment_VUpdateConfig(const GdkToolEnv* env, GdkToolCfg toolConfig)
{
    return kOK;
}

ToolFx(kStatus) TestSurfaceDynamicAlignment_VStart(TestSurfaceDynamicAlignment tool)
{
    kObj(TestSurfaceDynamicAlignment, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    obj->dataSource = GdkToolCfg_Source(config);

    GdkParams params = GdkToolCfg_Parameters(config);

    obj->regions[0] = *GdkParam_AsSurfaceRegion(GdkParams_Find(params, PARAM_REGION1_NAME));
    obj->regions[1] = *GdkParam_AsSurfaceRegion(GdkParams_Find(params, PARAM_REGION2_NAME));
    obj->regions[2] = *GdkParam_AsSurfaceRegion(GdkParams_Find(params, PARAM_REGION3_NAME));

    return kOK;
}

ToolFx(kStatus) TestSurfaceDynamicAlignment_VStop(TestSurfaceDynamicAlignment tool)
{
    kObj(TestSurfaceDynamicAlignment, tool);
    return kOK;
}

ToolFx(kStatus) TestSurfaceDynamicAlignment_VProcess(TestSurfaceDynamicAlignment tool, GdkToolInput input, GdkToolOutput output)
{
    kObj(TestSurfaceDynamicAlignment, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    GdkSurfaceInput item = kNULL;
    GdkDataInfo itemInfo = kNULL;
    const kPoint3d64f* anchor = GdkToolInput_AnchorPosition(input);
    kSize i = 0;
    kPoint3d64f newOffset = { 0 };
    kPoint3d64f points[NUM_OF_REGIONS] = { { k64F_NULL, k64F_NULL, k64F_NULL }, { k64F_NULL, k64F_NULL, k64F_NULL }, { k64F_NULL, k64F_NULL, k64F_NULL } };
    k64f xAngle = 0, yAngle = 0, zIntersect = 0;
    kArray2 pointArray = kNULL;
    GvSurfaceMsg surfaceMsg = kNULL;

    item = GdkToolInput_Find(input, obj->dataSource);
    if (!item)
    {
        return kERROR_PARAMETER;
    }
    itemInfo = GdkInputItem_Info(item);

    kTry
    {
        // Adjust user region based on anchoring information. If anchoring is not enabled, anchor is zero.
        for(i = 0; i < NUM_OF_REGIONS; i++)
        {
            obj->regions[i].x += anchor->x;
            obj->regions[i].y += anchor->y;
            obj->regions[i].z += anchor->z;
        }

        // Find the average point of each region
        kTest(GetAvgPoints(tool, item, points));

        // Find a plane through the three points found
        kTest(FitPlane(points, &xAngle, &yAngle, &zIntersect));

        // Transform the surface so that the three points are aligned on the x-y plane
        kTest(TransformSurface(item, -xAngle, -yAngle, -zIntersect, &pointArray, &newOffset, kObject_Alloc(tool)));

        // Output Measurements
        kTest(TestSurfaceDynamicAlignment_OutputValue(output, MEAS_XANGLE_INDEX, xAngle, xAngle != k64F_NULL, config));
        kTest(TestSurfaceDynamicAlignment_OutputValue(output, MEAS_YANGLE_INDEX, yAngle, yAngle != k64F_NULL, config));
        kTest(TestSurfaceDynamicAlignment_OutputValue(output, MEAS_ZTRANS_INDEX, zIntersect, zIntersect != k64F_NULL, config));

        // Output the aligned Surface
        if (pointArray != kNULL)
        {
            kTest(GdkToolOutput_InitSurfaceAt(output, OUTPUT_ALIGNED_SURFACE_INDEX, kArray2_Length(pointArray, 1), kArray2_Length(pointArray, 0), &surfaceMsg));
            if (surfaceMsg != kNULL)
            {
                kTest(GvSurfaceMsg_SetOffset(surfaceMsg, &newOffset));
                kTest(GvSurfaceMsg_SetScale(surfaceMsg, GdkDataInfo_Scale(itemInfo)));
                kTest(GvSurfaceMsg_SetPointsArray(surfaceMsg, pointArray));
            }
        }
    }
    kFinally
    {
        kDestroyRef(&pointArray);
        kEndFinally();
    }

    return kOK;
}

ToolFx(kStatus) GetAvgPoints(TestSurfaceDynamicAlignment tool, GdkToolInput item, kPoint3d64f* regionAvg)
{
    kObj(TestSurfaceDynamicAlignment, tool);
    kSize i = 0, j = 0;
    k16s k = 0;
    kSize rows = 0, columns = 0;
    const kPoint3d64f* offset = kNULL, *scale = kNULL;
    k64f regionSum[NUM_OF_REGIONS] = { 0 };
    kSize regionCount[NUM_OF_REGIONS] = { 0 };
    GdkDataInfo itemInfo = GdkInputItem_Info(item);
    kSize regionIndex = 0;
    const GdkRegion3d64f* region = kNULL;

    columns = GdkSurfaceInput_ColumnCount(item);
    rows = GdkSurfaceInput_RowCount(item);
    offset = GdkInputItem_Offset(item);
    scale = GdkDataInfo_Scale(itemInfo);

    for (j = 0; j < rows; j++)
    {
        for (i = 0; i < columns; i++)
        {
            const k16s* pixel = GdkSurfaceInput_RangeRowAt(item, j) + i;
            k = *pixel;
            if (k != k16S_NULL)
            {
                kPoint3d64f pt = { 0, 0, 0 };

                // Translate the point to real-world coordinates
                pt.x = i*scale->x + offset->x;
                pt.y = j*scale->y + offset->y;
                pt.z = k*scale->z + offset->z;

                // Check if the point is in one of the regions
                for (regionIndex = 0; regionIndex < NUM_OF_REGIONS; regionIndex++)
                {
                    region = &obj->regions[regionIndex];
                    if (pt.x > region->x && pt.x < region->x + region->width &&
                        pt.y > region->y && pt.y < region->y + region->length &&
                        pt.z > region->z && pt.z < region->z + region->height)
                    {
                        regionSum[regionIndex] += pt.z;
                        regionCount[regionIndex]++;
                    }
                }
            }
        }
    }

    // Find the average point of each region
    for (regionIndex = 0; regionIndex < NUM_OF_REGIONS; regionIndex++)
    {
        if (regionCount[regionIndex] != 0)
        {
            regionAvg[regionIndex].x = obj->regions[regionIndex].x + obj->regions[regionIndex].width / 2;
            regionAvg[regionIndex].y = obj->regions[regionIndex].y + obj->regions[regionIndex].length / 2;
            regionAvg[regionIndex].z = regionSum[regionIndex] / regionCount[regionIndex];
        }
    }

    return kOK;
}

ToolFx(kStatus) FitPlane(kPoint3d64f* points, k64f* xAngle, k64f* yAngle, k64f* zIntersect)
{
    kPoint3d64f vec1 = { 0 }, vec2 = { 0 }, normal = { 0 };

    // Verify that all of the points are defined
    for (kSize i = 0; i < NUM_OF_REGIONS; i++)
    {
        if (points[i].x == k64F_NULL || points[i].y == k64F_NULL || points[i].z == k64F_NULL)
        {
            *xAngle = k64F_NULL;
            *yAngle = k64F_NULL;
            return kOK;
        }
    }

    // Find the normal to the plane by cross producting the vectors
    vec1 = Vector_Subtract(points[1], points[0]);
    vec2 = Vector_Subtract(points[2], points[0]);
    normal = Vector_CrossProduct(vec1, vec2);

    // Find the angles to the z axis, and the intersection with the z axis
    *yAngle = atan(normal.x / normal.z);
    *xAngle = -atan(normal.y / normal.z);
    *zIntersect = (normal.x*points[0].x + normal.y*points[0].y + normal.z*points[0].z) / normal.z;
    return kOK;
}

ToolFx(kStatus) TransformSurface(GdkToolInput item, k64f xAngle, k64f yAngle, k64f zTranslation, kArray2* outPointArray, kPoint3d64f* offset, kAlloc alloc)
{
    kSize columns = 0, rows = 0;
    kSize newWidth = 0, newHeight = 0;
    kSize  i = 0, j = 0;
    k16s k = 0;
    const kPoint3d64f* origOffset = kNULL, *scale = kNULL;
    kL3dTransform3d transform = { 0 };
    kArrayList listPoints = kNULL;
    kSize ptIdx = 0;
    kPoint3d64f pt = { 0, 0, 0 };
    kPoint3d16s pixelPt = { 0, 0, 0 };
    k64f maxX = k64F_MIN, maxY = k64F_MIN;
    GdkDataInfo itemInfo = GdkInputItem_Info(item);

    // Check that the inputs are defined
    if (xAngle == k64F_NULL || yAngle == k64F_NULL || zTranslation == k64F_NULL)
    {
        *outPointArray = kNULL;
        return kOK;
    }

    kTry
    {
        columns = GdkSurfaceInput_ColumnCount(item);
        rows = GdkSurfaceInput_RowCount(item);
        origOffset = GdkInputItem_Offset(item);
        scale = GdkDataInfo_Scale(itemInfo);

        offset->x = offset->y = k64F_MAX;
        offset->z = origOffset->z;

        // Create the transformation matrix
        kTest(kL3dTransform3d_Identity(&transform));
        kTest(kL3dTransform3d_Rotate(&transform, xAngle, yAngle, 0, 180 / M_PI, &transform));
        kTest(kL3dTransform3d_Translate(&transform, 0, 0, zTranslation, &transform));

        // Since the surface's dimensions might have changed, list all the points and
        // find the new height/width and x/y offsets

        kTest(kArrayList_Construct(&listPoints, kTypeOf(kPoint3d64f), rows*columns, alloc));
        for (j = 0; j < rows; j++)
        {
            for (i = 0; i < columns; i++)
            {
                const k16s* pixel = GdkSurfaceInput_RangeRowAt(item, j) + i;
                k = *pixel;
                if (k != k16S_NULL)
                {
                    // Get the original point
                    pt.x = i*scale->x + origOffset->x;
                    pt.y = j*scale->y + origOffset->y;
                    pt.z = k*scale->z + origOffset->z;

                    // Transform to the new point
                    kTest(kL3dTransform3d_Apply(&transform, pt.x, pt.y, pt.z, &pt.x, &pt.y, &pt.z));

                    kTest(kArrayList_AddT(listPoints, &pt));

                    // Find the new offset
                    offset->x = kMin_(offset->x, pt.x);
                    offset->y = kMin_(offset->y, pt.y);

                    // Find the new range
                    maxX = kMax_(maxX, pt.x);
                    maxY = kMax_(maxY, pt.y);
                }
            }
        }

        // Calculate the width and height from the offset and range
        newWidth = kMath_Round16s_((maxX - offset->x) / scale->x) + 1;
        newHeight = kMath_Round16s_((maxY - offset->y) / scale->y) + 1;

        // Place all the new points in a 2 dimensional array (make sure all pixels are initialized to NULL)
        kTest(kArray2_Construct(outPointArray, kTypeOf(k16s), newHeight, newWidth, alloc));
        for (j = 0; j < newHeight; j++)
        {
            for (i = 0; i < newWidth; i++)
            {
                k16s z = k16S_NULL;
                kTest(kArray2_SetItemT(*outPointArray, j, i, &z));
            }
        }
        for (ptIdx = 0; ptIdx < kArrayList_Count(listPoints); ptIdx++)
        {
            pt = *kArrayList_AtT(listPoints, ptIdx, kPoint3d64f);

            // Translate point to grid coordinate
            pixelPt.x = kMath_Round16s_((pt.x - offset->x) / scale->x);
            pixelPt.y = kMath_Round16s_((pt.y - offset->y) / scale->y);
            pixelPt.z = kMath_Round16s_((pt.z - offset->z) / scale->z);

            kTest(kArray2_SetItemT(*outPointArray, pixelPt.y, pixelPt.x, &pixelPt.z));
        }
    }
    kFinally
    {
        kDestroyRef(&listPoints);
        kEndFinally();
    }
    return kOK;
}

ToolFx(kPoint3d64f) Vector_Subtract(kPoint3d64f a, kPoint3d64f b)
{
    kPoint3d64f retVec = { 0 };

    retVec.x = a.x - b.x;
    retVec.y = a.y - b.y;
    retVec.z = a.z - b.z;

    return retVec;
}

ToolFx(kPoint3d64f) Vector_CrossProduct(kPoint3d64f a, kPoint3d64f b)
{
    kPoint3d64f retVec = { 0 };

    retVec.x = a.y*b.z - a.z*b.y;
    retVec.y = a.z*b.x - a.x*b.z;
    retVec.z = a.x*b.y - a.y*b.x;

    return retVec;
}

ToolFx(kStatus) TestSurfaceDynamicAlignment_OutputValue(GdkToolOutput output, kSize index, k64f value, kBool valid, GdkToolCfg config)
{
    GvMeasureMsg msg = kNULL;

    if (GdkMeasurementCfg_Enabled(GdkToolCfg_MeasurementAt(config, index)))
    {
        kCheck(GdkToolOutput_InitMeasurementAt(output, index, &msg));
        if (msg != kNULL)
        {
            if (valid)
            {
                kCheck(GvMeasureMsg_SetValue(msg, value));
                kCheck(GvMeasureMsg_SetStatus(msg, GV_MEASUREMENT_OK));
            }
            else
            {
                kCheck(GvMeasureMsg_SetStatus(msg, GV_MEASUREMENT_ERROR_INVALID_RESULT));
            }
        }
    }

    return kOK;
}
