/**
* @file    GvSurfaceMsg.h
* @brief   Declares the GvSurfaceMsg class.
*
* Copyright (C) 2017-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef GV_SURFACE_MSG_H
#define GV_SURFACE_MSG_H

#include <GoVision/Data/GvDataMsg.h>

/**
* Encapsulates a surface message.
*
* @class       GvSurfaceMsg
* @extends     GvDataMsg
* @ingroup     GoVision-Data
*/
typedef GvDataMsg GvSurfaceMsg;

//Takes ownership of ranges and intensities data
GvFx(kStatus) GvSurfaceMsg_Construct(GvSurfaceMsg* msg, const kStamp* stamp, k64u time, k64s encoder, kArray2 ranges, kArray2 intensities, 
                                     const kPoint3d64f* offset, const kPoint3d64f* scale,
                                     kAlloc allocator);

//Takes ownership of ranges and intensities data
GvFx(kStatus) GvSurfaceMsg_ConstructUnresampled(GvSurfaceMsg* msg, const kStamp* stamp, k64u time, k64s encoder, kArray2 ranges, kArray2 intensities,
                                                kAlloc allocator);

GvFx(kStatus) GvSurfaceMsg_ConstructSimple(GvSurfaceMsg* msg, kSize width, kSize height, kAlloc alloc);
GvFx(kStatus) GvSurfaceMsg_ConstructUnresampledSimple(GvSurfaceMsg* msg, kSize width, kSize height, kAlloc alloc);

/** 
 * Returns the width as number of columns.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @return              Column count.
 */
GvFx(kSize) GvSurfaceMsg_Width(GvSurfaceMsg msg);

/** 
 * Returns the length as number of rows.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @return              Row count.
 */
GvFx(kSize) GvSurfaceMsg_Length(GvSurfaceMsg msg);

/** 
 * Returns the x,y,z offset of the data message.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @return              Pointer to data offset.
 */
GvFx(const kPoint3d64f*) GvSurfaceMsg_Offset(GvSurfaceMsg msg);

/** 
 * Sets the x,y,z offset of the data message.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @param   offset      Pointer to data offset.
 * @return              Operation status.
 */
GvFx(kStatus) GvSurfaceMsg_SetOffset(GvSurfaceMsg msg, const kPoint3d64f* offset);

/** 
 * Returns the x,y,z scale of the data message.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @return              Pointer to data scale.
 */
GvFx(const kPoint3d64f*) GvSurfaceMsg_Scale(GvSurfaceMsg msg);

/** 
 * Sets the x,y,z scale of the data message.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @param   scale       Pointer to data scale.
 * @return              Operation status.
 */
GvFx(kStatus) GvSurfaceMsg_SetScale(GvSurfaceMsg msg, const kPoint3d64f* scale);

/** 
 * Returns the message point type.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @return              Point type.
 */
GvFx(kType) GvSurfaceMsg_PointType(GvSurfaceMsg msg);

/** 
 * Returns a pointer to the points buffer.
 *
 * The buffer can be both read and written. The buffer is organized such that
 * values in the same row are contiguous. In other words, the value at row i and
 * column j is at address i * width + j.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @return              Buffer pointer.
 */
GvFx(void*) GvSurfaceMsg_Points(GvSurfaceMsg msg);

/** 
 * Copies the points buffer with the data from an array.
 *
 * The item type and dimensions of the array must match those of the message.
 * Otherwise an error is returned.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @param   points      Points array.
 * @return              Operation status.
 */
GvFx(kStatus) GvSurfaceMsg_CopyPointsArray(GvSurfaceMsg msg, kArray2 points);

/** 
 * @deprecated. Use GvSurfaceMsg_CopyPointsArray() instead.
 * Updates the points buffer with the data from an array.
 *
 * The item type and dimensions of the array must match those of the message.
 * Otherwise an error is returned.
 * Note: this does not transfer ownership of the passed array but only copies it 
 * into the message object.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @param   points      Points array.
 * @return              Operation status.
 */
GvFx(kStatus) GvSurfaceMsg_SetPointsArray(GvSurfaceMsg msg, kArray2 points);

/** 
 * Returns an array object containing the points.
 *
 * The user must not free the returned array.
 *
 * The implementation reuses the points buffer so it is fairly efficient.
 * A shell object is still constructed however so it is not as efficient as
 * using the buffer directly.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @return              Buffer pointer.
 */
GvFx(kArray2) GvSurfaceMsg_PointsArray(GvSurfaceMsg msg);

/** 
 * Returns whether or not this message contains intensity data.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @return              Presence of intensity data.
 */
GvFx(kBool) GvSurfaceMsg_HasIntensity(GvSurfaceMsg msg);

/** 
 * Returns the message intensity type.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @return              Intensity type.
 */
GvFx(kType) GvSurfaceMsg_IntensityType(GvSurfaceMsg msg);

/** 
 * Allocates an intensity buffer for the message.
 *
 * If the buffer already exists and is of the same type as requested, success
 * is returned. If it exists but type does not match, kERROR_ALREADY_EXISTS is returned.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @param   type        Type of intensity values.
 * @return              Operation status.
 */
GvFx(kStatus) GvSurfaceMsg_AllocateIntensity(GvSurfaceMsg msg, kType type);

/** 
 * Returns a pointer to the intensity buffer.
 *
 * The buffer can be both read and written.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @return              Buffer pointer.
 */
GvFx(void*) GvSurfaceMsg_Intensity(GvSurfaceMsg msg);

/** 
 * Copies the intensity buffer with the data from an array into the message.
 *
 * The item type and size of the array must match those of the message.
 * Otherwise an error is returned.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @param   intensity   Intensity array.
 * @return              Operation status.
 */
GvFx(kStatus) GvSurfaceMsg_CopyIntensityArray(GvSurfaceMsg msg, kArray2 intensity);

/** 
 * @deprecated. Use GvSurfaceMsg_CopyIntensityArray() instead.
 * Updates the intensity buffer with the data from an array.
 * Note: this does not transfer ownership of the passed array but only copies it 
 * into the message object.
 *
 * The item type and size of the array must match those of the message.
 * Otherwise an error is returned.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @param   intensity   Intensity array.
 * @return              Operation status.
 */
GvFx(kStatus) GvSurfaceMsg_SetIntensityArray(GvSurfaceMsg msg, kArray2 intensity);

/** 
 * Returns an array object containing the intensity data.
 *
 * The user must not free the returned array.
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object. 
 * @return              Array object.
 */
GvFx(kArray2) GvSurfaceMsg_IntensityArray(GvSurfaceMsg msg);

/**
* Returns true if this is a resampled surface msg
*
* @public              @memberof GvSurfaceMsg
* @param   msg         Message object.
* @return              kBool.
*/
GvFx(kBool) GvSurfaceMsg_IsResampled(GvSurfaceMsg msg);

/**
* Sets if the data in a surface msg is adjacent (renderable)
* Default: kFALSE
*
* @public              @memberof GvSurfaceMsg
* @param   msg         Message object.
* @param   isAdjacent  Is the data adjacent?
* @return              kBool.
*/
GvFx(kStatus) GvSurfaceMsg_SetIsAdjacent(GvSurfaceMsg msg, kBool isAdjacent);

/**
* Returns true if the surface data in the message is adjacent
*
* @public              @memberof GvSurfaceMsg
* @param   msg         Message object.
* @return              kBool.
*/
GvFx(kBool) GvSurfaceMsg_IsAdjacent(GvSurfaceMsg msg);

/**
 * Sets the rendering object within the message.
 *
 * @public               @memberof GvSurfaceMsg
 * @param    msg         Message object.
 * @param    object      GdkGraphic object (ownership is transferred).
 * @return               Operation status.
 */
GvFx(kStatus) GvSurfaceMsg_SetRendering(GvSurfaceMsg msg, kObject object);

/**
 * Retrieves the rendering object from the message.
 *
 * @public               @memberof GvSurfaceMsg
 * @param    msg         Message object.
 * @return               GdkGraphic object (ownership is transferred).
 */
GvFx(kObject) GvSurfaceMsg_Rendering(GvSurfaceMsg msg);

/**
* Copies the surface start information from the provided table into the
* the surface message.
*
* @public              @memberof GvSurfaceMsg
* @param   msg         Message object.
* @param   startTable  Table containing the surface start information to copy
*                      into the surface message.
* @return              Operation status.
*/
GvFx(kStatus) GvSurfaceMsg_AddSurfaceStartInfo(GvSurfaceMsg msg, kArray2 startTable);

/**
* Returns a handle to the surface message's start info table.
*
* @public              @memberof GvSurfaceMsg
* @param   msg         Message object.
* @return              Handle to surface message's kArray2 start info table.
*/
GvFx(kArray2) GvSurfaceMsg_SurfaceStartInfo(GvSurfaceMsg msg);

/**
 * Returns the x,y,z offset with respect to frame of reference for the surface message .
 *
 * @public              @memberof GvSurfaceMsg
 * @param   msg         Message object.
 * @return              Surface offset.
 */
GvFx(kPoint3d64f) GvSurfaceMsg_OffsetWithRef(GvSurfaceMsg msg);

#include <GoVision/Data/GvSurfaceMsg.x.h>

#endif
