#ifndef K_FIRESYNC_NODEDEV_H
#define K_FIRESYNC_NODEDEV_H

#include <kFireSync/kNodeDef.h>

#if defined(K_FIRESYNC_INCLUDE_DEVICE_ACCESS)

/**
 * @class   kNodeDev
 * @extends kObject
 * @ingroup kFireSync-Client-Device
 * @brief   Collection of static utility methods for low-level device access.
 *
 * The utility functions in this class perform hardware configuration and should 
 * only be used in a controlled factory environment.  
 * 
 * The 'key' argument used in class methods refers to kFIRESYNC_DEVICE_ACCESS_KEY.
 */


 /** 
 * Interrogates hardware to determine the light model, revision, and device ID.
 *
 * Revision only has major and minor fields.
 *
 * When communicating with a legacy device that does not contain this information,
 * the revision and device ID fields will return unsigned "null" (e.g., k32U_NULL).
 *
 * @public              @memberof kNodeDev
 * @param   node        Node object.  
 * @param   key         Device access key.
 * @param   index       Light driver index.
 * @param   model       Light model identification (e.g. LD16-G).
 * @param   revision    Light model revision.
 * @param   deviceId    Light model device identification.
 * @return              Operation status.
 */
kFsFx(kStatus) kNodeDev_ReadLightDriverInfo(kNode node, k64u key, kSize index, kLightModel* model, kVersion* revision, k32u* deviceId); 

/** 
 * Stores the light model, revision, and device ID.
 * 
 * This method can be used to set the light model, revision, and device ID.
 * The light model, revision, and device ID should only be configured
 * under controlled factory conditions.
 * 
 * Light device information is committed to non-volatile storage within the light device. Storage capacity is 
 * extremely limited (~500 bits) and capacity is <em>permanently</em> consumed each time that either the 
 * device information or the light power is committed. Use with caution!
 * 
 * Setting the light device information may invalidate any stored light power setting. After setting, 
 * kNodeDev_WritePower can be used to commit a new power setting. 
 * 
 * @public              @memberof kNodeDev
 * @param   node        Node object.  
 * @param   key         Device access key.
 * @param   index       Light driver index.
 * @param   model       Light model identification (e.g. LD16-G).
 * @param   revision    Light model revision.
 * @param   deviceId    Light model device identification.
 * @return              Operation status.
 */
kFsFx(kStatus) kNodeDev_WriteLightDriverInfo(kNode node, k64u key, kSize index, kLightModel model, kVersion revision, k32u deviceId);

/** 
 * Enables laser power manual control mode.
 *
 * Laser power setting has two modes, manual and auto control:
 * Manual: power can be explicity set via kNodeDev_AdjustLightDriverPower(), in which uses the specified power 
 *         and ignore all other settings such as light intensity, high/low power...etc.
 * Auto:   power cannot be set via kNodeDev_AdjustLightDriverPower(), in which case power value is determined
 *         automatically using various settings (light intensity, enable high power, non-volatile diode power, 
 *         LD cal power). 
 * 
 * @public              @memberof kNodeDev
 * @param   node        Node object.  
 * @param   key         Device access key.
 * @param   index       Light driver index.
 * @param   enabled     Enables manual control mode.
 * @return              Operation status.
 */
kFsFx(kStatus) kNodeDev_EnableLightDriverControl(kNode node, k64u key, kSize index, kBool enabled);  

/** 
 * Reports whether laser power manual control mode is enabled.
 * 
 * @public              @memberof kNodeDev
 * @param   node        Node object.  
 * @param   key         Device access key.
 * @param   index       Light driver index.
 * @param   enabled     Is manual control mode enabled?
 * @return              Operation status
 */
kFsFx(kStatus) kNodeDev_LightDriverControlEnabled(kNode node, k64u key, kSize index, kBool* enabled);

/** 
 * Reports constraint and validity information for a laser power setting. 
 *
 * @public              @memberof kNodeDev
 * @param   model       Light driver model. 
 * @param   power       Power value (or k32U_NULL for constraints only).  
 * @param   info        Receives constraint and validity information. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kNodeDev_LightDriverPowerInfo(kLightModel model, k32u power, kInfo32u* info); 

/** 
 * Adjusts the light power level temporarily, without committing it to storage.
 *
 * This method will return an error if manual control mode is not enabled.
 * 
 * This method can be used to adjust the light power level. Adjusting light power 
 * may violate laser classification rules. Accordingly, light power should only be adjusted 
 * under controlled factory conditions. 
 * 
 * Do not mix the use of this low-level method with the use of kLight_SetIntensity; light intensity 
 * should be set to 100% (its default value) when directly manipulating driver power.
 *
 * @public              @memberof kNodeDev
 * @param   node        Node object.  
 * @param   key         Device access key.
 * @param   index       Light driver index.
 * @param   power       Light power.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNodeDev_AdjustLightDriverPower(kNode node, k64u key, kSize index, k32u power);

/** 
 * Sets the light power level and commits it to storage.
 * 
 * This method can be used to adjust the light power level. Adjusting light power 
 * may violate laser classification rules. Accordingly, light power should only be adjusted 
 * under controlled factory conditions. 
 *
 * Do not mix the use of this low-level method with the use of kLight_SetIntensity; light intensity 
 * should be set to 100% (its default value) when directly manipulating driver power.
 *
 * Light power is committed to non-volatile storage within the light device. Storage capacity is 
 * extremely small (~500 bits) and capacity is <em>permanently</em> consumed each time that 
 * either the light power or the light model is committed. Use with caution!
 * 
 * @public              @memberof kNodeDev
 * @param   node        Node object.  
 * @param   key         Device access key.
 * @param   index       Light driver index.
 * @param   power       Light power.
 * @return              Operation status. 
 */
kFsFx(kStatus) kNodeDev_WriteLightDriverPower(kNode node, k64u key, kSize index, k32u power);

/** 
 * Reads the current light power level.  
 *
 * Do not mix the use of this low-level method with the use of kLight_SetIntensity; light intensity 
 * should be set to 100% (its default value) when directly reading driver power.
 *
 * @public              @memberof kNodeDev
 * @param   node        Node object.
 * @param   key         Device access key.
 * @param   index       Light driver index.
 * @param   power       Receives light power.
 * @return              Operation status.
 */
kFsFx(kStatus) kNodeDev_ReadLightDriverPower(kNode node, k64u key, kSize index, k32u* power);

/**
 * Reports constraint and validity information for a light current limit setting.
 *
 * @public                  @memberof kNodeDev
 * @param   model           Light driver model.
 * @param   currentLimit    Light current limit value (or k32U_NULL for constraints only).
 * @param   revision        Light driver revision.
 * @param   info            Receives constraint and validity information.
 * @return                  Operation status.
 */
kFsFx(kStatus) kNodeDev_LightDriverCurrentLimitInfo(kLightModel model, k32u currentLimit, kVersion revision, kInfo32u* info);

/**
 * Adjusts the light current limit temporarily, without committing it to storage.
 *
 * This method will return an error if manual control mode is not enabled.
 *
 * This method can be used to adjust the light current limit. Adjusting light current limit
 * may violate laser classification rules. Accordingly, light current limit should only be adjusted
 * under controlled factory conditions.
 *
 * In factory environment, light power will be initially adjusted to a desired output. In a later stage,
 * light current limit will be lowered to a certain percentage up to a point where the light power starts clipping.
 * Light current limit ensures that the laser is within its class specification.
 *
 * @public                  @memberof kNodeDev
 * @param   node            Node object.
 * @param   key             Device access key.
 * @param   index           Light driver index.
 * @param   currentLimit    Light current limit.
 * @return                  Operation status.
 */
kFsFx(kStatus) kNodeDev_AdjustLightDriverCurrentLimit(kNode node, k64u key, kSize index, k32u currentLimit);

/**
 * Sets the light current limit and commits it to storage.
 *
 * This method can be used to adjust the light current limit. Adjusting light current limit
 * may violate laser classification rules. Accordingly, light current limit should only be adjusted
 * under controlled factory conditions.
 *
 * In factory environment, light power will be initially adjusted to a desired output. In a later stage,
 * light current limit will be lowered to a certain percentage up to a point where the light power starts clipping.
 * Light current limit ensures that the laser is within its class specification.
 *
 * Light current limit is committed to non-volatile storage within the light device. 
 *
 * @public                  @memberof kNodeDev
 * @param   node            Node object.
 * @param   key             Device access key.
 * @param   index           Light driver index.
 * @param   currentLimit    Light current limit.
 * @return                  Operation status.
 */
kFsFx(kStatus) kNodeDev_WriteLightDriverCurrentLimit(kNode node, k64u key, kSize index, k32u currentLimit);

/**
 * Reads the light current limit.
 *
 * @public                  @memberof kNodeDev
 * @param   node            Node object.
 * @param   key             Device access key.
 * @param   index           Light driver index.
 * @param   currentLimit    Receives light current limit.
 * @return                  Operation status.
 */
kFsFx(kStatus) kNodeDev_ReadLightDriverCurrentLimit(kNode node, k64u key, kSize index, k32u* currentLimit);

/**
 * Reads the remaining light memory slot count.
 *
 * @public                  @memberof kNodeDev
 * @param   node            Node object.
 * @param   key             Device access key.
 * @param   index           Light driver index.
 * @param   remainingCount  Receives light remaining slot count.
 * @return                  Operation status.
 */
kFsFx(kStatus) kNodeDev_ReadLightDriverRemainingSlotCount(kNode node, k64u key, kSize index, kSize* remainingCount);

/**
 * Sets the light driver calibration data.
 *
 * Calls this function also trigger light driver to load kLdModel.xml from temporary storage if exists.
 *
 * @public                  @memberof kNodeDev
 * @param   node            Node object.
 * @param   key             Device access key.
 * @param   index           Light driver index.
 * @param   ldCal           Light driver calibration object.
 * @return                  Operation status.
 */
kFsFx(kStatus) kNodeDev_LoadLightDriverCalibration(kNode node, k64u key, kSize index, kLdCal ldCal);

/**
 * Sets and commits the light driver calibration data.
 *
 * Calls this function also trigger light driver to load kLdModel.xml from temporary storage if exists.
 *
 * @public                  @memberof kNodeDev
 * @param   node            Node object.
 * @param   key             Device access key.
 * @param   index           Light driver index.
 * @param   ldCal           Light driver calibration object.
 * @return                  Operation status.
 */
kFsFx(kStatus) kNodeDev_WriteLightDriverCalibration(kNode node, k64u key, kSize index, kLdCal ldCal);

/**
 * Reads the light driver calibration data.
 *
 * @public                  @memberof kNodeDev
 * @param   node            Node object.
 * @param   key             Device access key.
 * @param   index           Light driver index.
 * @param   ldCal           Receives light driver calibration object.
 * @return                  Operation status.
 */
kFsFx(kStatus) kNodeDev_ReadLightDriverCalibration(kNode node, k64u key, kSize index, kLdCal* ldCal, kAlloc alloc);

/**
 * Reads the light driver temperature.
 *
 * @public                  @memberof kNodeDev
 * @param   node            Node object.
 * @param   key             Device access key.
 * @param   index           Light driver index.
 * @param   temperature     Receives light driver temperature.
 * @return                  Operation status.
 */
kFsFx(kStatus) kNodeDev_ReadLightDriverTemperature(kNode node, k64u key, kSize index, k32s* temperature);

#include <kFireSync/Client/Device/kNodeDev.x.h>

#endif

#endif
