/** 
 * @file    kLightSteps.h
 * @brief   Declares the kLightSteps type. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_LIGHT_STEPS_H
#define K_FIRESYNC_LIGHT_STEPS_H

#include <kFireSync/Client/Light/Extensions/kLightExt.h>

/**
 * @class   kLightSteps
 * @extends kLightExt
 * @ingroup kFireSync-Client-Light-Extensions
 * @brief   Represents parameters for a light step extension. 
 */
//typedef kLightExt kLightSteps;            --forward-declared in kFsDef.x.h

/** 
 * Sets the number of steps (state/count pairs) defined by this extension. 
 *
 * @public              @memberof kLightSteps
 * @param   extension    Sequence object. 
 * @param   count       Step count. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kLightSteps_SetCount(kLightSteps extension, kSize count); 

/** 
 * Gets the number of steps (state/count pairs) defined by this extension. 
 *
 * @public              @memberof kLightSteps
 * @param   extension    Sequence object. 
 * @return              Step count. 
 */
kFsFx(kSize) kLightSteps_Count(kLightSteps extension); 

/** 
 * Reports constraint and validity information for the Count setting. 
 *
 * @public              @memberof kLightSteps
 * @param   extension    Sequence object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLightSteps_CountInfo(kLightSteps extension, kInfoSize* info); 

/** 
 * Sets the light state id (index) associated with the specified step index. 
 *
 * @public              @memberof kLightSteps
 * @param   extension    Sequence object. 
 * @param   stepIndex   Step index. 
 * @param   stateIndex  Light state id/index. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kLightSteps_SetState(kLightSteps extension, kSize stepIndex, kSize stateIndex); 

/** 
 * Gets the light state id (index) associated with the specified step index. 
 *
 * @public              @memberof kLightSteps
 * @param   extension    Sequence object. 
 * @param   stepIndex   Step index. 
 * @return              Light state id/index.
 */
kFsFx(kSize) kLightSteps_State(kLightSteps extension, kSize stepIndex); 

/** 
 * Sets the repetition count associated with the specified step index. 
 *
 * @public              @memberof kLightSteps
 * @param   extension    Sequence object. 
 * @param   stepIndex   Step index. 
 * @param   count       Repetition count. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kLightSteps_SetRepetitionCount(kLightSteps extension, kSize stepIndex, kSize count); 

/** 
 * Gets the repetition count associated with the specified step index. 
 *
 * @public              @memberof kLightSteps
 * @param   extension    Sequence object. 
 * @param   stepIndex   Step index. 
 * @return              Repetition count. 
 */
kFsFx(kSize) kLightSteps_RepetitionCount(kLightSteps extension, kSize stepIndex); 

/**
* Sets the steps style.
*
* @public              @memberof kLightSteps
* @param   extension   Sequence object.
* @param   style       Steps style.
* @return              Operation status.
*/
kFsFx(kStatus) kLightSteps_SetStyle(kLightSteps extension, kLightStepsStyle style);

/**
* Gets the steps style.
*
* @public              @memberof kLightSteps
* @param   extension   Extension object.
* @return              Steps style.
*/
kFsFx(kLightStepsStyle) kLightSteps_Style(kLightSteps extension);

/**
* Reports constraint and validity information for the Steps style setting.
*
* @public              @memberof kLightSteps
* @param   extension   Extension object.
* @param   info        Receives the info structure.
* @return              Operation status.
*/
kFsFx(kStatus) kLightSteps_StyleInfo(kLightSteps extension, kInfoBits* info);

#include <kFireSync/Client/Light/Extensions/kLightSteps.x.h>

#endif
