/** 
 * @file    kProjectorInfo.h
 * @brief   Declares the kProjectorInfo class and related types. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_PROJECTOR_INFO_H
#define K_FIRESYNC_PROJECTOR_INFO_H

#include <kFireSync/kNodeDef.h>

/**
 * @internal
 * @class   kProjectorInfo
 * @extends kObject
 * @ingroup kFireSync-Client-Projector
 * @brief   Abstract base class for projector model classes. 
 */
//typedef kObject kProjectorInfo;         --forward-declared in kFsDef.x.h

//DP03
typedef struct kDp03FrameRateControl
{
    k8u hsyncWidth;
    k8u hsyncBackPorch;
    k16u hsyncFrontPorch;
    k8u vsyncWidth;
    k8u vsyncBackPorch;
    k16u vsyncFrontPorch;
} kDp03FrameRateControl;

//kDp03PatternConfiguration is written to DLPC, cannot have structure padding.
#pragma pack(push, 1)

typedef struct kDp03PatternConfiguration
{
    k8u type;
    k8u imagesPerFrame;
    k8u ledSelect;
    k32u exposureUs;
    k32u preExpDarkTimeUs;
    k32u postExpDarkTimeUs;
} kDp03PatternConfiguration;

#pragma pack(pop)

typedef struct kDp03Timing
{
    kDp03FrameRateControl frameControl;
    kDp03PatternConfiguration patternConfig;
    k64u frameTime;
    k64u patternDuration;
    k64u patternTime;
    k64u imageTime;
} kDp03Timing;

//DP01
typedef struct kDlpc300ExternalSequence
{
    kSize frameRate;             //Frame rate.
    kSize imagesPerFrame;        //Number of images per frame.
    kSize sequenceNumber;        //DLPC300 sequence number.
    kSize startVector;           //DLPC300 start vector.
    kSize subVector;             //DLPC300 sub-vector.
} kDlpc300ExternalSequence;

typedef struct kDlpc300InternalSequence
{
    k64u exposure;              //Exposure time (real ns).
    kSize sequenceNumber;        //DLPC300 sequence number.
    kSize startVector;           //DLPC300 start vector.
    kSize subVector;             //DLPC300 sub-vector.
} kDlpc300InternalSequence;

typedef union kDlpc300VideoSequence
{
    kDlpc300InternalSequence internal;
    kDlpc300ExternalSequence external;
} kDlpc300VideoSequence;

kDeclareValueEx(kFs, kDlpc300VideoSequence, kValue)

typedef struct kDp01FrameRateControl
{
    k32u engineRegister;
    k8u hsyncWidth;
    k8u hsyncBackPorch;
    k8u hsyncFrontPorch;
    k8u vsyncWidth;
    k8u vsyncBackPorch;
    k8u vsyncFrontPorch;
} kDp01FrameRateControl;

kDeclareValueEx(kFs, kDp01FrameRateControl, kValue)

typedef struct kDp01Timing
{
    kProjectorPatternSource patternSource;
    kDp01FrameRateControl frameControl;
    kDlpc300VideoSequence videoSequence;
    kSize lastFrameImageNumber;
    kSize lastExposureImageNumber;
    k64u patternDuration;
    k64u patternTime;
    k64u imageTime;
    kBool exposureOverride;
} kDp01Timing;

//LP01
typedef struct kLp01Timing
{
    k32u colorWeightRed;
    k32u colorWeightGreen;
    k32u colorWeightBlue;
    k64u ledTime;
} kLp01Timing;

typedef union kPanelTiming
{
    kLp01Timing lp01;
    kDp01Timing dp01;
    kDp03Timing dp03;
} kPanelTiming;

typedef struct kProjectorTiming
{
    kSize repeatCount;
    k64f accurateExposure;
    kPanelTiming panel;
} kProjectorTiming;

kDeclareValueEx(kFs, kProjectorTiming, kValue)

/** 
 * Constructs a kProjectorInfo object.
 *
 * @public              @memberof kProjectorInfo
 * @param   info        Destination for the constructed object handle. 
 * @param   model       Projector model identifier.
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kProjectorInfo_Construct(kProjectorInfo* info, kProjectorModel model, k64f pixelClockFrequency, kAlloc allocator);

kFsFx(kProjectorModel) kProjectorInfo_Model(kProjectorInfo info);

kFsFx(kSize) kProjectorInfo_Width(kProjectorInfo info);
kFsFx(kSize) kProjectorInfo_Height(kProjectorInfo info);

kFsFx(k64u) kProjectorInfo_MinExposure(kProjectorInfo info, k32u intensity, k32u minIntensity);

kFsFx(k32u) kProjectorInfo_MinPatternCount(kProjectorInfo info);
kFsFx(k32u) kProjectorInfo_MaxPatternCount(kProjectorInfo info);

kFsFx(k32u) kProjectorInfo_MinBitDepth(kProjectorInfo info);
kFsFx(k32u) kProjectorInfo_MaxBitDepth(kProjectorInfo info);

kFsFx(k32u) kProjectorInfo_MinIntensity(kProjectorInfo info);
kFsFx(k32u) kProjectorInfo_MaxIntensity(kProjectorInfo info);

kFsFx(k32u) kProjectorInfo_MinPaletteColor(kProjectorInfo info);
kFsFx(k32u) kProjectorInfo_MaxPaletteColor(kProjectorInfo info);

kFsFx(kProjectorPatternMode) kProjectorInfo_PatternModes(kProjectorInfo info);

kFsFx(kBool) kProjectorInfo_HasDynamicFrameRate(kProjectorInfo info);
kFsFx(kBool) kProjectorInfo_IsPilotRunNeeded(kProjectorInfo info);
kFsFx(kBool) kProjectorInfo_IsCalibrationNeeded(kProjectorInfo info);
kFsFx(kBool) kProjectorInfo_HasTempProbe(kProjectorInfo info);
kFsFx(kBool) kProjectorInfo_CanAutoShutdown(kProjector projector);

kFsFx(k64u) kProjectorInfo_ImageTime(kProjectorInfo info, kSize bitDepth, k64u exposure, kBool dynFrameRate);
kFsFx(k64u) kProjectorInfo_FrameTime(kProjectorInfo info, kSize bitDepth, k64u exposure, kBool dynFrameRate);
kFsFx(kStatus) kProjectorInfo_CalculateExposure(kProjectorInfo info, kSize bitDepth, k64u exposure, k64u cameraDuration, kBool dynFrameRate, kProjectorTiming* timing);
kFsFx(k64u) kProjectorInfo_SequenceDuration(kProjectorInfo info, kSize bitDepth, k64u exposure, k64u cameraDuration, kBool dynFrameRate, kSize patternCount, kBool sequenceEnabled);
kFsFx(k64u) kProjectorInfo_MinimumGap(kProjectorInfo info, kSize bitDepth, k64u exposure, k64u cameraDuration, kBool dynFrameRate, kSize patternCount, kBool sequenceEnabled, k64u cameraPlProcessingTime);
kFsFx(k64u) kProjectorInfo_NominateCameraExposure(kProjectorInfo info, kSize bitDepth, k64u stateExposure, k64u cameraDuration, kBool dynFrameRate, k64u desiredExposure);

kFsFx(kProjectorExtension) kProjectorInfo_ExtensionTypes(kProjectorInfo info);

kFsFx(k32u) kProjectorInfo_DefaultPatternCount(kProjectorInfo info);
kFsFx(kSize) kProjectorInfo_DefaultBitDepth(kProjectorInfo info);
kFsFx(kProjectorPatternMode) kProjectorInfo_DefaultPatternMode(kProjectorInfo info);
kFsFx(k32u) kProjectorInfo_DefaultPalette(kProjectorInfo info, kSize index);
kFsFx(k32u) kProjectorInfo_DefaultIntensity(kProjectorInfo info);
kFsFx(k64u) kProjectorInfo_DefaultExposure(kProjectorInfo info);
kFsFx(k64u) kProjectorInfo_DefaultGap(kProjectorInfo info);

kFsFx(k64u) kProjectorInfo_PilotRunExposure(kProjectorInfo info);
kFsFx(k64u) kProjectorInfo_PilotRunPeriod(kProjectorInfo info);
kFsFx(k32u) kProjectorInfo_PilotRunIntensity(kProjectorInfo info);

#include <kFireSync/Client/Projector/kProjectorInfo.x.h>

#endif
