/** 
 * @file    kCamera.h
 * @brief   Declares the kCamera type. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_CAMERA_H
#define K_FIRESYNC_CAMERA_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kCamera
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents camera configuration.
 */
//typedef kObject kCamera;        --forward-declared in kFsDef.x.h

/** 
 * Gets the model of the camera.
 *
 * The model supported by a camera module is determined by the current 
 * hardware configuration. If a different camera model is required, load a hardware 
 * configuration that supports the desired model. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Returns the model of the camera.
 */
kFsFx(kCameraModel) kCamera_Model(kCamera camera);

/** 
 * Gets the id of the camera module.
 *
 * The id of a camera module is the same as the index of the camera module in the 
 * node's camera list.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Returns the id of the camera module.
 */
kFsFx(kSize) kCamera_Id(kCamera camera);

/** 
 * Gets the absolute maximum width of the camera.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Maximum width, in columns.
 */
kFsFx(k32u) kCamera_MaxWidth(kCamera camera);

/** 
 * Gets the absolute maximum height of the camera.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Maximum height, in rows.
 */
kFsFx(k32u) kCamera_MaxHeight(kCamera camera);

/** 
 * Gets the hardware-accelerated algorithm supported by this camera. 
 *
 * @public              @memberof kCamera
 * @param   camera      Block object.  
 * @return              Algorithm identifier. 
 */
kFsFx(kCameraAlgorithm) kCamera_Algorithm(kCamera camera);

/** 
 * Resets camera mode settings to defaults.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   stateCount  Number of camera states to create.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_Clear(kCamera camera, kSize stateCount);

/** 
 * Resets camera mode and device settings to defaults.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   stateCount  Number of camera states to create.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_ClearAll(kCamera camera, kSize stateCount);

/**
 * Reports the current camera video buffer size.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @return              Video buffer size, in bytes.
 */
kFsFx(kSize) kCamera_VideoBufferSize(kCamera camera);

/**
 * Sets the video buffer size that will take effect after reset.
 *
 * This setting represents the requested video buffer size; the actual size may be reduced
 * due to memory limitations or page granularity requirements.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @param   size        Video buffer size.
 * @return              Operation status.
 */
kFsFx(kStatus) kCamera_SetNextVideoBufferSize(kCamera camera, kSize size);

/**
 * Reports the video buffer size that will take effect after reset.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @return              Video buffer size, in bytes.
 */
kFsFx(kSize) kCamera_NextVideoBufferSize(kCamera camera);

/**
 * Reports the current camera algorithm buffer size.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @return              Algorithm buffer size, in bytes.
 */
kFsFx(kSize) kCamera_AlgorithmBufferSize(kCamera camera);

/**
 * Sets the algorithm buffer size that will take effect after reset.
 *
 * This setting represents the requested algorithm buffer size; the actual size may be reduced
 * due to memory limitations or page granularity requirements.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @param   size        Algorithm buffer size.
 * @return              Operation status.
 */
kFsFx(kStatus) kCamera_SetNextAlgorithmBufferSize(kCamera camera, kSize size);

/**
 * Reports the algorithm buffer size that will take effect after reset.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @return              Algorithm buffer size, in bytes.
 */
kFsFx(kSize) kCamera_NextAlgorithmBufferSize(kCamera camera);

/** 
 * Enables or disables the camera. 
 *
 * Disabled cameras are not checked during verification, and do not respond to 
 * Start/Pause/Resume/Stop commands. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   enable      Specifies whether to enable or disable the camera.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_Enable(kCamera camera, kBool enable);

/** 
 * Reports whether the camera is currently enabled. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Returns whether the camera is currently enabled. 
 */
kFsFx(kBool) kCamera_IsEnabled(kCamera camera);

/** 
 * Reports whether a camera device is connected and communicating. 
 *
 * Cameras are detected immediately after a node is reset; this value is not updated dynamically. 
 * 
 * Physical connection has no effect on configuration; a camera module without a connected 
 * camera can still be configured and verified. At runtime, a disconnected camera module will 
 * not produce images, but the disconnected status will not be treated as an error that 
 * prevents the system from running.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Returns whether a physical camera is connected to the camera module.
 */
kFsFx(kBool) kCamera_IsConnected(kCamera camera);

/** 
 * Sets the control mode of the camera.
 *
 * For control modes that support multiple sources (e.g. kCAMERA_CONTROL_TRIGGERED_BY_EVENT), 
 * use the kCamera_SetControlId function to specify a specific source.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   type        Camera control mode.
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_SetControl(kCamera camera, kCameraControl type);

/** 
 * Reports the control mode of the camera. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Returns the control mode of the camera. 
 */
kFsFx(kCameraControl) kCamera_Control(kCamera camera);

/** 
 * Reports constraint and validity information for the Control setting. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_ControlInfo(kCamera camera, kInfoBits* info);

/** 
 * Sets the control source id for a camera. 
 *
 * For control modes that support multiple sources (e.g. kCAMERA_CONTROL_TRIGGERED_BY_EVENT), 
 * this function cam be used to specify a specific source id.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   id          Specifies the control source id.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_SetControlId(kCamera camera, k32u id);

/** 
 * Gets the control source id for a camera.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Control source id.
 */
kFsFx(k32u) kCamera_ControlId(kCamera camera);

/** 
 * Reports constraint and validity information for the ControlId setting. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_ControlIdInfo(kCamera camera, kInfo* info);

/** 
 * Determines whether estimated imager post-exposure is incorporated into nominal exposure.
 *
 * When enabled, the underlying imager post-exposure (the duration for which an imager is sensitive to 
 * light after its physical exposure signal is deasserted, determined via characterization) will be 
 * included in its nominal exposure duration. In this case, a lead-out (offset between end of physical 
 * imager exposure and end of nominal camera exposure) is used. In turn, this lead-out typically 
 * determines the minimum exposure time constraint. Use of this option, which is enabled by default, 
 * provides some automatic protection against light multiplexing hazards by ensuring that the camera's 
 * nominal exposure time reflects all of the time in which the camera is actually sensitive to light. 
 * When this option is enabled, the kCameraState_LeadOutNs property will typically report a positive 
 * value, and the kCameraState_PostExposureNs property will report a zero value. 
 *
 * However, some cameras can produce exposure artifacts if significant illumination is present 
 * during camera post-exposure (e.g., "black sun" effects on LC2000/5000). In these cases, reliance on 
 * camera lead-outs can result in artifacts when light exposure is coupled to (or manually aligned with)
 * camera exposure. Additionally, if lights are coupled to cameras and very low exposures are required, 
 * disabling the camera lead-out allows the nominal camera exposure to be reduced down to zero, which in 
 * turn allows the coupled light exposure to be reduced down to zero. This can be used to simulate exposure 
 * times that are lower than what the camera itself can support, an advantage in some contexts. 
 * 
 * Due to the problems and limitations associated with camera lead-out, it is recommended that lead-out 
 * should be disabled in new applications. However, please note that doing so will change the effective 
 * exposure time for the same nominal exposure setting; as such, this choice must be carefully weighed 
 * if backwards compability with older platform versions is important. When this option is disabled, the 
 * kCameraState_LeadOutNs property will report zero, and the kCameraState_PostExposureNs property will 
 * likely report a positive value. The post-exposure time should be considered when determining multiplexed 
 * camera/light time slots. 
 * 
 * This option is enabled by default. This option is not configurable for DM647 (M2, M50) controllers.
 * 
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   enable      kTRUE to include camera lead-out.  
 * @return              Operation status. 
 * @see                 kCameraState_LeadOutNs, kCameraState_PostExposureNs 
 */
kFsFx(kStatus) kCamera_EnableLeadOut(kCamera camera, kBool enable);

/** 
 * Reports whether camera lead-out is enabled.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              kTRUE if lead-out is enabled.
 */
kFsFx(kBool) kCamera_LeadOutEnabled(kCamera camera);

/** 
 * Reports constraint and validity information for the LeadOutEnabled setting. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_LeadOutEnabledInfo(kCamera camera, kInfoBool* info);

/** 
 * Creates and adds a new state with default values.
 *
 * The new state is added to the end of the camera's state list. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   state       Receives the new state object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_AddState(kCamera camera, kCameraState* state);

/** 
 * Destroys the specified state and removes it from the camera's state list.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   state       State object to be deleted.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_DeleteState(kCamera camera, kCameraState state);

/** 
 * Swaps the order of two states in the camera's state list.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   index0      Index of the first state object to be swapped.  
 * @param   index1      Index of the second state object to be swapped.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_SwapStates(kCamera camera, kSize index0, kSize index1);

/** 
 * Gets the number of states belonging to the camera. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Count of camera states. 
 */
kFsFx(kSize) kCamera_StateCount(kCamera camera);

/** 
 * Reports constraint and validity information for the StateCount setting. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_StateCountInfo(kCamera camera, kInfoSize* info);

/** 
 * Gets the state at the specified index.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   index       Index of the state.  
 * @return              State at the specified index.
 */
kFsFx(kCameraState) kCamera_StateAt(kCamera camera, kSize index);

/** 
 * Creates and adds a new camera control extension of the specified type.
 *
 * The new extension is added to the end of the camera's extension list. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   type        Type of extension to add. 
 * @param   extension   Receives the new extension object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_AddExtension(kCamera camera, kType type, kCameraExt* extension);

/** 
 * Destroys the specified camera control extension and removes it from the camera's extension list.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   extension   Extension object to be deleted.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_DeleteExtension(kCamera camera, kCameraExt extension);

/** 
 * Gets the number of control extensions that have been configured.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Count of camera control extensions. 
 */
kFsFx(kSize) kCamera_ExtensionCount(kCamera camera);

/** 
 * Gets the control extension at the specified index.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   index       Index of the extension.  
 * @return              Extension at the specified index.
 */
kFsFx(kCameraExt) kCamera_ExtensionAt(kCamera camera, kSize index);

/** 
 * Reports the time required by this camera to respond to an event. 
 *
 * Minimum period values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Minimum period, in nanoseconds. 
 */
kFsFx(k64u) kCamera_MinimumPeriod(kCamera camera);

/** 
 * Sets the readout mode for the camera.  
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   readoutMode Readout mode identifier.   
 * @see                 @ref kCameraReadout
 * @return              Operation status. 
 * 
 * This function sets the readout mode for the camera. This setting may be ignored by the 
 * camera, based on the model or other settings. See @ref kCameraReadout for more information.
 */
kFsFx(kStatus) kCamera_SetReadout(kCamera camera, kCameraReadout readoutMode);

/** 
 * Gets the readout mode for the camera.  
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Returns the readout mode for the camera. 
 * @see                 @ref kCameraReadout
 * 
 * This function gets the readout mode for the camera. This setting may be ignored by the 
 * camera, based on the model or other settings. See @ref kCameraReadout for more information.
 */
kFsFx(kCameraReadout) kCamera_Readout(kCamera camera);

/** 
 * Reports constraint and validity information for the Readout setting. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   info        Receives the info structure.  
 * @see                 @ref kCameraReadout
 * @return              Operation status. 
 * 
 * This function gets the readout mode options available in kFireSync. See @ref kCameraReadout
 * for more information.
 */
kFsFx(kStatus) kCamera_ReadoutInfo(kCamera camera, kInfoBits* info);

/** 
 * Specifies whether the FPGA RAM Image feature is enabled.
 *
 * The FPGA RAM image feature is used for testing/simulation.  Essentially, a set 
 * of pre-recorded images can be loaded for each camera state. At runtime, the 
 * the FPGA cycles through these images, rather than using real images captured by the 
 * camera.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   enabled     Specifies whether FPGA Ram Image is enabled.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_EnableRamImage(kCamera camera, kBool enabled);

/** 
 * Reports whether the FPGA RAM Image feature is currently enabled. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Returns whether the FPGA RAM Image feature is currently enabled. 
 */
kFsFx(kBool) kCamera_RamImageEnabled(kCamera camera);

/** 
 * Reports constraint and validity information for the RamImageEnabled setting. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   info        Receives the info structure.  
 * @return              Returns constraint and validity information for the RamImageEnabled setting. 
 */
kFsFx(kBool) kCamera_RamImageEnabledInfo(kCamera camera, kInfoBool* info);

/** 
 * Specifies whether PRNU is enabled.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   enabled     Specifies whether PRNU is enabled.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_EnablePrnu(kCamera camera, kBool enabled);

/** 
 * Reports whether PRNU is currently enabled. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Returns whether PRNU is currently enabled. 
 */
kFsFx(kBool) kCamera_PrnuEnabled(kCamera camera);

/** 
 * Reports constraint and validity information for the PrnuEnabled setting. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   info        Receives the info structure.  
 * @return              Returns constraint and validity information for the PrnuEnabled setting. 
 */
kFsFx(kBool) kCamera_PrnuEnabledInfo(kCamera camera, kInfoBool* info);

/** 
 * Specifies whether FPN is enabled.
 *
 * This setting enables static FPN.  Do not enable this feature when using the 
 * the dynamic FPN camera extension.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   enabled     Specifies whether FPN is enabled.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_EnableFpn(kCamera camera, kBool enabled);

/** 
 * Reports whether FPN is currently enabled. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Returns whether FPN is currently enabled. 
 */
kFsFx(kBool) kCamera_FpnEnabled(kCamera camera);

/** 
 * Reports constraint and validity information for the FpnEnabled setting. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   info        Receives the info structure.  
 * @return              Returns constraint and validity information for the FpnEnabled setting. 
 */
kFsFx(kBool) kCamera_FpnEnabledInfo(kCamera camera, kInfoBool* info);

/** 
 * Writes PRNU table.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   white       white offsets.  
 * @param   black       black offsets.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_WritePrnu(kCamera camera, kImage white, kImage black);

/** 
 * Writes FPN table.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   offsets     FPN array. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_WriteFpn(kCamera camera, kImage offsets);

/** 
 * Writes FPGA Range LUT.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   params      LUT parameters.  
 * @param   xArray      X LUT.  
 * @param   zArray      Z LUT.  
 * @param   validArray  Valid LUT.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_WriteRangeLut(kCamera camera, const kRangeLutParams* params, const kRangeLutArray* xArray,
                                                            const kRangeLutArray* zArray, const kRangeLutArray* validArray);

/** 
 * Specifies whether FPN calibration is enabled.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   enabled     Specifies whether FPN calibration is enabled.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_EnableFpnCalibration(kCamera camera, kBool enabled);

/** 
 * Reports whether FPN calibration is currently enabled. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Returns whether FPN calibration is currently enabled. 
 */
kFsFx(kBool) kCamera_FpnCalibrationEnabled(kCamera camera);

/** 
 * Reports constraint and validity information for the FpnCalibrationEnabled setting. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   info        Receives the info structure.  
 * @return              Returns constraint and validity information for the FpnCalibrationEnabled setting. 
 */
kFsFx(kBool) kCamera_FpnCalibrationEnabledInfo(kCamera camera, kInfoBool* info);

/** 
 * Specifies whether Test pattern is enabled.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   enabled     Specifies whether test pattern is enabled.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_EnableTestPattern(kCamera camera, kBool enabled);

/** 
 * Reports whether test pattern is currently enabled. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Returns whether test pattern is currently enabled. 
 */
kFsFx(kBool) kCamera_TestPatternEnabled(kCamera camera);

/** 
 * Reports constraint and validity information for the TestPatternEnabled setting. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   info        Receives the info structure.  
 * @return              Returns constraint and validity information for the TestPatternEnabled setting. 
 */
kFsFx(kBool) kCamera_TestPatternEnabledInfo(kCamera camera, kInfoBool* info);

/** 
 * Retrieve statistics.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   stats       Receives the stats structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_Stats(kCamera camera, kCameraStats* stats);

/** 
 * Sets the input counter source for a camera. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   source      Input counter source.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_SetInputCounterSource(kCamera camera, kCameraInputCounterSource source);

/** 
 * Retrieves the input counter source.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.   
 * @return              Input counter source. 
 */
kFsFx(kCameraInputCounterSource) kCamera_InputCounterSource(kCamera camera);

/** 
 * Reports constraint and validity information for the Input Counter Source setting. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_InputCounterSourceInfo(kCamera camera, kInfoBits* info);

/** 
 * Sets the input counter source id for a camera. 
 *
 * For input counter source modes that support multiple sources, 
 * this function cam be used to specify a specific source id.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   id          Specifies the input counter source id.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_SetInputCounterSourceId(kCamera camera, k32u id);

/** 
 * Gets the input counter source id for a camera.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Input counter source id.
 */
kFsFx(k32u) kCamera_InputCounterSourceId(kCamera camera);

/** 
 * Reports constraint and validity information for the InputCounterSourceId setting. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_InputCounterSourceIdInfo(kCamera camera, kInfo* info);

/** 
 * Provides a software trigger to a camera. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_Trigger(kCamera camera);

/**
 * Provides a list of configuration overrides to modify live camera behaviour. 
 * 
 * This function is used in conjunction with the "Override" extension (kCameraOverride). When 
 * the override extension is enabled, and node acquisition is started, this function can be 
 * used to modify specific camera settings without having to stop/restart the node. This feature 
 * can be used to implement low-speed dynamic exposure control or window tracking in software. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @param   overrides   Override items.
 * @param   count       Count of the override items.
 * @return              Operation status.
 */
kFsFx(kStatus) kCamera_Override(kCamera camera, const kCameraOverrideItem* overrides, kSize count);

/**
 * Sets the output mode of the camera.
 *
 * FireSync supports receiving both video and algorithm data from sensor at the same time.
 * User can change camera output mode to select what type of data to output from camera,
 * could be either video or algorithm, or both.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @param   mode        Camera output mode.
 * @return              Operation status.
 */
kFsFx(kStatus) kCamera_SetOutputMode(kCamera camera, kCameraOutputMode mode);

/**
 * Gets the output mode of the camera.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @return              Returns the output mode of the camera.
 */
kFsFx(kCameraOutputMode) kCamera_OutputMode(kCamera camera);

/**
 * Reports whether video output is enabled, based on the current OutputMode setting.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @return              kTRUE if video (or video+algorithm) output is enabled.
 */
kInlineFx(kBool) kCamera_VideoEnabled(kCamera camera);

/**
 * Reports whether algorithm output is enabled, based on the current OutputMode setting.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @return              kTRUE if algorithm (or video+algorithm) output is enabled.
 */
kInlineFx(kBool) kCamera_AlgorithmEnabled(kCamera camera);

/** 
 * Reports constraint and validity information for the OutputMode setting. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_OutputModeInfo(kCamera camera, kInfoBits* info);

/**
 * Sets the port mode of the camera.
 *
 * There are two camera port mode, combined and separate.
 *
 * With combined port mode, there is only one port per camera state which is "State x",
 * where "x" is state ID. All the messages generated from this state are sent from
 * the same port. When there are both video and algorithm data from camera, camera
 * port will output both data type alternatively.
 *
 * With separate port mode, there are two ports, "Video x" and "Algorithm x", per 
 * camera state. Messages are sent from dedicated port based on data type.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @param   mode        Camera port mode.
 * @return              Operation status.
 */
kFsFx(kStatus) kCamera_SetPortMode(kCamera camera, kCameraPortMode mode);

/**
 * Gets the port mode of the camera.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @return              Returns the port mode of the camera.
 */
kFsFx(kCameraPortMode) kCamera_PortMode(kCamera camera);

/** 
 * Reports constraint and validity information for the PortMode setting. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_PortModeInfo(kCamera camera, kInfoBits* info);

/** 
 * Gets required video buffer size that is enough to buffer the desired number of frames
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   frameCount  Frame count.
 * @return              Operation status. 
 */
kFsFx(kSize) kCamera_NominalVideoBufferSize(kCamera camera, kSize frameCount);

/** 
 * Gets required algorithm buffer size that is enough to buffer the desired number of frames
 *
 * NOTE: Frame count indicates how many frames from PL can be stored in the buffer. 
 *       For phase decoder, stripe+phase pair is consider a frame. It reports required
 *       buffer size for whole sequence.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   frameCount  Frame count.
 * @return              Operation status. 
 */
kFsFx(kSize) kCamera_NominalAlgorithmBufferSizeEx(kCamera camera, kSize frameCount);

/**
 * Specifies the bandwidth reduction factor.
 *
 * NOTE: Reduction factor represents the nth frame sequence to keep.
 *
 * Examples:
 * window = 1 and factor = 5
 * We will keep every fifth frame (frames to keep: 1, 6, 11, etc.).
 *
 * window = 3 and factor = 4
 * We will keep every 4th 3-frame sequence (frames to keep: 1, 2, 3, 13, 14, 15, etc.).
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @param   factor      Reduction factor.
 * @return              Operation status.
 */
kFsFx(kStatus) kCamera_SetReductionFactor(kCamera camera, k32u factor);

/**
 * Reports the bandwidth reduction factor.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @return              Bandwidth reduction factor.
 */
kFsFx(k32u) kCamera_ReductionFactor(kCamera camera);

/**
 * Reports constraint and validity information for the bandwidth reduction factor setting.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @param   info        Receives the info structure.
 * @return              Operation status.
 */
kFsFx(kStatus) kCamera_ReductionFactorInfo(kCamera camera, kInfo32u* info);

/**
 * Specifies the window size to be used for bandwidth reduction.
 *
 * NOTE: Reduction window represents the number of frames in a sequence.
 *
 * Examples:
 * window = 1 and factor = 5
 * We will keep every fifth frame (frames to keep: 1, 6, 11, etc.).
 *
 * window = 3 and factor = 4
 * We will keep every 4th 3-frame sequence (frames to keep: 1, 2, 3, 13, 14, 15, etc.).
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @param   window      Window size.
 * @return              Operation status.
 */
kFsFx(kStatus) kCamera_SetReductionWindow(kCamera camera, k32u window);

/**
 * Reports the window size used for bandwidth reduction.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @return              Bandwidth reduction window.
 */
kFsFx(k32u) kCamera_ReductionWindow(kCamera camera);

/**
 * Reports constraint and validity information for the bandwidth reduction window setting.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @param   info        Receives the info structure.
 * @return              Operation status.
 */
kFsFx(kStatus) kCamera_ReductionWindowInfo(kCamera camera, kInfo32u* info);

/**
 * Specifies the imager LVDS power setting.
 *
 * NOTE: LVDS power is a volatile setting, it won't be saved to sensor configuration.
 *       That said, the change is only temporary for current session and will back to firmware default after reset.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @param   power       LVDS power.
 * @return              Operation status.
 */
kFsFx(kStatus) kCamera_SetLvdsPower(kCamera camera, k32u power);

/**
 * Reports the imager LVDS power setting.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @return              Bandwidth reduction window.
 */
kFsFx(k32u) kCamera_LvdsPower(kCamera camera);

/**
 * Reports constraint and validity information for imager LVDS power setting.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @param   info        Receives the info structure.
 * @return              Operation status.
 */
kFsFx(kStatus) kCamera_LvdsPowerInfo(kCamera camera, kInfo32u* info);

/**
 * Set the imager LVDS power using kFactory default setting.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @return              Operation status.
 */
kFsFx(kStatus) kCamera_SetFactoryLvdsPower(kCamera camera);

/** 
 * Sets the acceleration test mode of the camera.
 *
 * Use kCamera_EnableAccelerationTest to enable test mode.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   mode        Camera acceleration test mode.
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_SetAccelerationTestMode(kCamera camera, kCameraAccelerationTestMode mode);

/** 
 * Reports acceleration test mode of the camera. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @return              Returns the control acceleration test mode of the camera. 
 */
kFsFx(kCameraAccelerationTestMode) kCamera_AccelerationTestMode(kCamera camera);

/** 
 * Reports constraint and validity information for the Acceleration test mode setting. 
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_AccelerationTestModeInfo(kCamera camera, kInfoBits* info);

/** 
 * Writes acceleration test input data.
 *
 * This function transfers a test dataset to the camera, for use in PL acceleration testing. The dataset 
 * is stored in sensor RAM until it is needed, then transferred to PL memory on demand.  The dataset can be 
 * cleared (freeing sensor resources) by passing null as the 'data' argument to this method. 
 * 
 * The format of the dataset is configuration-specific. E.g.,
 * -kCAMERA_ACCELERATION_TEST_MODE_PHASE_COMPRESSION: 
 *  kMsg containing kArray2<kPhasePixel2> or kArray1<kByte> (pre-formatted binary data)
 * 
 * Use the kCamera_SetAccelerationTestMode method to enable acceleration testing. kCamera_SetAccelerationTestMode
 * can be called before or after uploading data.
 * 
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @param   data        Acceleration test data.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_WriteAccelerationTestData(kCamera camera, kObject data);

/** 
 * Retrieves algorithm (e.g. EPS) acceleration test result, for use in algorithm testing. 
 * Use this function with caution as it reads PL memory used when calculating algorithm result. 
 * Sensor has to be in running state. See kCameraAccelerationTestMode enum for details on result format.
 *
 * @public              @memberof kCamera
 * @param   camera      Camera object.
 * @param   data        Receives test result. 
 * @param   allocator   Optional allocator to receive result, if applicable (can be null). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCamera_ReadAccelerationTestResult(kCamera camera, kObject* data, kAlloc allocator);

#include <kFireSync/Client/kCamera.x.h>

#endif
