/** 
 * @file    kCameraState.h
 * @brief   Declares the kCameraState type. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_CAMERA_STATE_H
#define K_FIRESYNC_CAMERA_STATE_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kCameraState
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents a camera state configuration.
 */
//typedef kObject kCameraState;        --forward-declared in kFsDef.x.h

/** 
 * Gets the id of the state. 
 *
 * The id of a state is the same as the index of the state in the camera's state list.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the unique id of this state. 
 */
kFsFx(kSize) kCameraState_Id(kCameraState state);

/** 
 * Sets the horizontal origin of the camera region-of-interest (ROI).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   left        Horizontal origin of the camera ROI.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetLeft(kCameraState state, k32u left, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the horizontal origin of the camera region-of-interest (ROI).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the horizontal origin of the camera region-of-interest (ROI).
 */
kFsFx(k32u) kCameraState_Left(kCameraState state);

/** 
 * Reports constraint and validity information for the Left setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_LeftInfo(kCameraState state, kInfo32u* info);

/** 
 * Sets the vertical origin of the camera region-of-interest (ROI).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   top         Horizontal origin of the camera ROI.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetTop(kCameraState state, k32u top, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the vertical origin of the camera region-of-interest (ROI).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the vertical origin of the camera region-of-interest (ROI).
 */
kFsFx(k32u) kCameraState_Top(kCameraState state);

/** 
 * Reports constraint and validity information for the Top setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_TopInfo(kCameraState state, kInfo32u* info);

/** 
 * Sets the width of the camera region-of-interest (ROI).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   width       Width of the camera ROI.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetWidth(kCameraState state, k32u width, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the width of the camera region-of-interest (ROI).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the width of the camera region-of-interest (ROI).
 */
kFsFx(k32u) kCameraState_Width(kCameraState state);

/** 
 * Reports constraint and validity information for the Width setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_WidthInfo(kCameraState state, kInfo32u* info);

/** 
 * Sets the height of the camera region-of-interest (ROI).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   height      Height of the camera ROI.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetHeight(kCameraState state, k32u height, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the height of the camera region-of-interest (ROI).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the height of the camera region-of-interest (ROI).
 */
kFsFx(k32u) kCameraState_Height(kCameraState state);

/** 
 * Reports constraint and validity information for the Height setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_HeightInfo(kCameraState state, kInfo32u* info);

/** 
 * Sets the column subsampling amount for the camera state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   xSubampling Column subsampling amount for the camera state (e.g. 1, 2, 4). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetXSubsampling(kCameraState state, k32u xSubampling);

/** 
 * Gets the column subsampling amount for the camera state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the column subsampling amount for the camera state (e.g. 1, 2, 4). 
 */
kFsFx(k32u) kCameraState_XSubsampling(kCameraState state);

/** 
 * Reports constraint and validity information for the XSubsampling setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_XSubsamplingInfo(kCameraState state, kInfoBits* info);

/** 
 * Sets the column subsampling mode for the camera state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   mode        Column subsampling mode for the camera state.
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetXSubsamplingMode(kCameraState state, kCameraSubsampling mode);

/** 
 * Gets the column subsampling mode for the camera state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the column subsampling mode for the camera state. 
 */
kFsFx(kCameraSubsampling) kCameraState_XSubsamplingMode(kCameraState state);

/** 
 * Reports constraint and validity information for the XSubsamplingMode setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_XSubsamplingModeInfo(kCameraState state, kInfoBits* info);

/** 
 * Sets the row subsampling amount for the camera state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   ySubampling Row subsampling amount for the camera state (e.g. 1, 2, 4).   
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetYSubsampling(kCameraState state, k32u ySubampling);

/** 
 * Gets the row subsampling amount for the camera state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the row subsampling for the camera state (e.g. 1, 2, 4).
 */
kFsFx(k32u) kCameraState_YSubsampling(kCameraState state);

/** 
 * Reports constraint and validity information for the YSubsampling setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_YSubsamplingInfo(kCameraState state, kInfoBits* info);

/** 
 * Sets the row subsampling mode for the camera state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   mode        Row subsampling mode for the camera state.
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetYSubsamplingMode(kCameraState state, kCameraSubsampling mode);

/** 
 * Gets the row subsampling mode for the camera state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the row subsampling mode for the camera state. 
 */
kFsFx(kCameraSubsampling) kCameraState_YSubsamplingMode(kCameraState state);

/** 
 * Reports constraint and validity information for the YSubsamplingMode setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_YSubsamplingModeInfo(kCameraState state, kInfoBits* info);

/** 
 * Sets the horizontal orientation for the camera state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   invert      Invert horizontal orientation for the camera state.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_InvertX(kCameraState state, kBool invert);

/** 
 * Gets the horizontal orientation for the camera state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the horizontal orientation for the camera state.
 */
kFsFx(kBool) kCameraState_XInverted(kCameraState state);

/** 
 * Reports constraint and validity information for the XInverted setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_XInvertInfo(kCameraState state, kInfoBool* info);

/** 
 * Sets the vertical orientation for the camera state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   invert      Invert vertical orientation for the camera state.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_InvertY(kCameraState state, kBool invert);

/** 
 * Gets the vertical orientation for the camera state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the vertical orientation for the camera state.
 */
kFsFx(kBool) kCameraState_YInverted(kCameraState state);

/** 
 * Reports constraint and validity information for the YInverted setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_YInvertInfo(kCameraState state, kInfoBool* info);

/** 
 * Enables automatic black-level adjustment for the camera state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   enabled     Specifies whether automatic black-level adjustment is enabled.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_EnableAutoBlack(kCameraState state, kBool enabled);

/** 
 * Reports whether automatic black-level adjustment is enabled for the camera state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns whether automatic black-level adjustment is enabled for the camera state.
 */
kFsFx(kBool) kCameraState_AutoBlackEnabled(kCameraState state);

/** 
 * Reports constraint and validity information for the AutoBlackEnabled setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_AutoBlackEnabledInfo(kCameraState state, kInfoBool* info);

/** 
 * Set the black-level for the camera state (used when AutoBlack is disabled).  
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   black       Black-level value.  
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetBlack(kCameraState state, k32s black, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the black-level for the camera state (used when AutoBlack is disabled).  
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the black-level for the camera state (used when AutoBlack is disabled).  
 */
kFsFx(k32s) kCameraState_Black(kCameraState state);

/** 
 * Reports constraint and validity information for the Black setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_BlackInfo(kCameraState state, kInfo32s* info);

/** 
 * Sets the gain for the specified gain channel.  
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   channel     Gain channel identifier.   
 * @param   gain        Gain value.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetGain(kCameraState state, kCameraGain channel, k64f gain);

/** 
 * Gets the gain for the specified gain channel.  
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   channel     Gain channel identifier.   
 * @return              Returns the gain for the specified gain channel.  
 */
kFsFx(k64f) kCameraState_Gain(kCameraState state, kCameraGain channel);

/** 
 * Reports constraint and validity information for a Gain setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   channel     Channel.  
 * @param   info        Receives the info structure.    
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_GainInfo(kCameraState state, kCameraGain channel, kInfo64f* info);

/** 
 * Sets the number of knee points. This feature should be used to alter the camera's gamma curve (multi slope).
 * For each knee point use @ref kCameraState_SetKneeTime and @ref kCameraState_SetKneeCap to change the actual curve.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   count       Number of Knee Points.   
 * @return              Operation status. 
 * @see                 @ref kCameraState_SetKneeTime, @ref kCameraState_SetKneeCap
 */
kFsFx(kStatus) kCameraState_SetKneeCount(kCameraState state, kSize count);

/** 
 * Gets the number of knee points in the gamma curve (multi slope).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the number of knee points.  
 */
kFsFx(kSize) kCameraState_KneeCount(kCameraState state);

/** 
 * Reports constraint and validity information for the KneeCount setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_KneeCountInfo(kCameraState state, kInfoSize* info);

/** 
 * Sets the knee point's normalized exposure time as a fraction of the total exposure time.
 *
 * @public                              @memberof kCameraState
 * @param   state                       State object.  
 * @param   index                       Knee Point index.   
 * @param   normalizedExposureTime      Knee Point normalized exposure time.
 * @return                              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetKneeTime(kCameraState state, kSize index, k64f normalizedExposureTime);

/** 
 * Gets the knee point's normalized exposure time as a fraction of the total exposure time.  
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   index       Knee Point index.   
 * @return              Returns the knee point exposure.  
 */
kFsFx(k64f) kCameraState_KneeTime(kCameraState state, kSize index);

/** 
 * Reports constraint and validity information for the KneeTime setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   index       Knee Point index.   
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_KneeTimeInfo(kCameraState state, kSize index, kInfo64f* info);

/** 
 * Sets the knee point's normalized reset point as a fraction of the maximum output.
 *
 * @public                              @memberof kCameraState
 * @param   state                       State object.  
 * @param   index                       Knee Point index.   
 * @param   normalizedOutputCap         Knee Point reset point.   
 * @return                              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetKneeCap(kCameraState state, kSize index, k64f normalizedOutputCap);

/** 
 * Gets the knee point's normalized reset point as a fraction of the maximum output.  
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   index       Knee Point index.   
 * @return              Returns the knee point reset point.  
 */
kFsFx(k64f) kCameraState_KneeCap(kCameraState state, kSize index);

/** 
 * Reports constraint and validity information for the KneeCap setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   index       Knee Point index.   
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_KneeCapInfo(kCameraState state, kSize index, kInfo64f* info);

/** 
 * Sets the exposure mode for the camera state.  
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   mode        Exposure mode identifier.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetExposureMode(kCameraState state, kCameraExposure mode);

/** 
 * Gets the exposure mode for the camera state.  
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the exposure mode for the camera state.  
 */
kFsFx(kCameraExposure) kCameraState_ExposureMode(kCameraState state);

/** 
 * Reports constraint and validity information for the ExposureMode setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_ExposureModeInfo(kCameraState state, kInfoBits* info);

/** 
 * Sets the maximum message batching size. 
 *
 * Message batching is used to combine multiple camera frames into one message, to reduce 
 * CPU utilization.
 *
 * By default the maximum batching size is 0, which implies no limit.
 *
 * @public                  @memberof kCameraState
 * @param   state           State object.  
 * @param   maxBatchSize    Maximum batch size, in camera frames. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kCameraState_SetMaxBatchSize(kCameraState state, kSize maxBatchSize);

/** 
 * Gets the maximum message batching size for the camera state.  
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Maximum batch size, in camera frames. 
 */
kFsFx(kSize) kCameraState_MaxBatchSize(kCameraState state);

/** 
 * Reports constraint and validity information for the MaxBatchSize setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_MaxBatchSizeInfo(kCameraState state, kInfoSize* info);

/** 
 * Sets the delay for the camera state, in nanoseconds.  
 *
 * Delay values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   delay       Delay value.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetDelayNs(kCameraState state, k64u delay, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the delay for the camera state, in nanoseconds.  
 *
 * Delay values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the delay for the camera state, in nanoseconds.  
 */
kFsFx(k64u) kCameraState_DelayNs(kCameraState state);

/** 
 * Reports constraint and validity information for the DelayNs setting, in nanoseconds.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_DelayNsInfo(kCameraState state, kInfo64u* info);

/** 
 * Sets the exposure for the camera state, in nanoseconds.
 *
 * Exposure values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   exposure    Exposure value.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetExposureNs(kCameraState state, k64u exposure, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the exposure for the camera state, in nanoseconds.  
 *
 * Exposure values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the exposure for the camera state, in nanoseconds.  
 */
kFsFx(k64u) kCameraState_ExposureNs(kCameraState state);

/** 
 * Reports constraint and validity information for the ExposureNs setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_ExposureNsInfo(kCameraState state, kInfo64u* info);

/** 
 * Sets the gap for the camera state, in nanoseconds.
 *
 * Gap values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   gap         Gap value.
 * @param   adjustment  Automatic adjustment style (optional).   
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetGapNs(kCameraState state, k64u gap, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the gap for the camera state, in nanoseconds.  
 *
 * Gap values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the gap for the camera state, in nanoseconds.  
 */
kFsFx(k64u) kCameraState_GapNs(kCameraState state);

/** 
 * Reports constraint and validity information for the GapNs setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_GapNsInfo(kCameraState state, kInfo64u* info);

/** 
 * Enables automatic gap calculation for the camera state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   enabled     Specifies whether automatic gap is enabled.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_EnableAutoGap(kCameraState state, kBool enabled);

/** 
 * Reports whether automatic gap is enabled for the camera state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns whether automatic gap is enabled for the camera state.
 */
kFsFx(kBool) kCameraState_AutoGapEnabled(kCameraState state);

/** 
 * Reports constraint and validity information for the AutoGapEnabled setting. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_AutoGapEnabledInfo(kCameraState state, kInfoBool* info);

/** 
 * Reports the total time required to execute this camera state, in nanoseconds. 
 *
 * Duration values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              State duration, in nanoseconds. 
 */
kFsFx(k64u) kCameraState_DurationNs(kCameraState state);

/**
 * Reports the lead-in time for the camera state.
 *
 * Lead-in is the time offset between the assertion of the physical camera exposure signal
 * and the start of nominal camera exposure (defined by DEG parameters).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.
 * @return              Lead-in time, in FireSync nanoseconds.
 */
kFsFx(k64u) kCameraState_LeadInNs(kCameraState state);

/**
 * Reports the total lead-in time for the camera state.
 *
 * Total lead-in includes the local camera state lead-in as well as the lead-ins of
 * any modules coupled to the camera (i.e., lights).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.
 * @return              Total lead-in time, in FireSync nanoseconds.
 */
kFsFx(k64u) kCameraState_TotalLeadInNs(kCameraState state);

/**
 * Reports the lead-out time for the camera state.
 *
 * Lead-out is the time offset between the deassertion of the physical camera exposure signal
 * and the end of nominal camera exposure (defined by DEG parameters).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.
 * @return              Lead-out time, in FireSync nanoseconds.
 */

kFsFx(k64u) kCameraState_LeadOutNs(kCameraState state);

/**
 * Reports the total lead-out time for the camera state.
 *
 * Total lead-out includes the local camera state lead-out as well as the lead-outs of
 * any module coupled to the camera (i.e., lights).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.
 * @return              Total lead-out time, in FireSync nanoseconds.
 */
kFsFx(k64u) kCameraState_TotalLeadOutNs(kCameraState state);

/**
 * Estimates the time for which the camera might be sensitive to light prior to nominal exposure.
 * 
 * This value should be used when determining multiplexed camera/light time slots. In practice, 
 * this value will usually be zero.
 * 
 * @public              @memberof kCameraState
 * @param   state       State object.
 * @return              Pre-exposure time, in FireSync nanoseconds.
 */
kFsFx(k64u) kCameraState_PreExposureNs(kCameraState state);

/**
 * Estimates the time for which the camera might be sensitive to light after nominal exposure.
 * 
 * This value should be used when determining multiplexed camera/light time slots. It will 
 * typically be zero if lead-out is enabled, and non-zero if lead-out is disabled.  
 *
 * @public              @memberof kCameraState
 * @param   state       State object.
 * @return              Post-exposure time, in FireSync nanoseconds.
 * @see                 kCamera_EnableLeadOut
 */
kFsFx(k64u) kCameraState_PostExposureNs(kCameraState state);

/** 
 * Gets current configuration type for a hardware-accelerated camera algorithm. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Algorithm type. 
 */
kFsFx(kCameraAlgorithm) kCameraState_AlgorithmType(kCameraState state);

/** 
 * Gets configuration for a hardware-accelerated camera algorithm. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Configuration object. 
 */
kFsFx(kAlgInfo) kCameraState_AlgorithmConfig(kCameraState state);

/** 
 * Sets the number of images that will be stored in FPGA RAM for this camera state 
 * (used for testing/simulation).
 *
 * The Ram Image feature must be enabled with the kCamera_EnableRamImage function; 
 * setting the RAM image count does not automatically enable the feature.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   count       Image count.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_SetRamImageCount(kCameraState state, k32u count);

/** 
 * Gets the number of images stored in FPGA RAM for this camera state 
 * (used for testing/simulation).
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Returns the number of images stored in FPGA RAM for this camera state 
 */
kFsFx(k32u) kCameraState_RamImageCount(kCameraState state);

/** 
 * Reports constraint and validity information for the RamImageCount setting.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_RamImageCountInfo(kCameraState state, kInfo32u* info);

/** 
 * Writes an image into FPGA RAM (used for testing/simulation).
 *
 * This function writes an image into FPGA RAM. The location of the image in FPGA RAM is determined 
 * by the size and count of images defined for <em>every</em> camera state used with this sensor. Accordingly,
 * when image size (based on Width/Height/Subsmapling properties) or count (based on RamImageCount property)
 * change for <em>any</em> camera state, the contents of FPGA RAM may be inconsistent with the updated settings.
 * As a result, image sizes and counts should be set <em>before</em> writing RAM images, to ensure that images
 * are written/read in the correct locations. If settings are changed after writing RAM images, the content
 * of the images emitted by each camera state may be different than expected.
 * 
 * Note: If settings have been changed, kNode_Apply should be called prior to calling kCameraState_WriteRamImage, to 
 * ensure that uncommitted settings are successfully applied before writing the image to the sensor. This ensures that 
 * the sensor has updated camera state properites, which are required to determine the correct location at which to 
 * write the image.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   index       Index of image within simulated image sequence.   
 * @param   image       Image to be written; should be same size as image produced by camera state.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraState_WriteRamImage(kCameraState state, kSize index, kImage image);

/** 
 * Reports validity of state transitions.
 *
 * @public              @memberof kCameraState
 * @param   previous    Previous state object.  
 * @param   current     Current state object.  
 * @return              kTRUE if transition is valid, otherwise, kFALSE. 
 */
kFsFx(kBool) kCameraState_IsTransitionValid(kCameraState previous, kCameraState current);

/** 
 * Gets the frame size of the state.
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @param   frameSize   Receive frame size of the state.
 * @return              Operation status.
 */
kFsFx(kStatus) kCameraState_FrameSize(kCameraState state, kSize* frameSize);

/**
 * Gets the camera object associated with this state object. 
 *
 * @public              @memberof kCameraState
 * @param   state       State object.  
 * @return              Camera object (parent).
 */
kFsFx(kCamera) kCameraState_Camera(kCameraState state); 

#include <kFireSync/Client/kCameraState.x.h>

#endif
