/** 
 * @file    kLight.h
 * @brief   Declares the kLight type. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_LIGHT_H
#define K_FIRESYNC_LIGHT_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kLight
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents light configuration.
 */
//typedef kObject kLight;        --forward-declared in kFsDef.x.h

/** 
 * Gets the id of the light module.
 *
 * The id of a light module is the same as the index of the light module in the 
 * node's light list.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @return              Returns the id of the light module.
 */
kFsFx(kSize) kLight_Id(kLight light);

/** 
 * Resets light mode settings to defaults.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   stateCount  Number of light states to create.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_Clear(kLight light, kSize stateCount);

/** 
 * Resets light mode and device settings to defaults.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   stateCount  Number of light states to create.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_ClearAll(kLight light, kSize stateCount);

/** 
 * Enables or disables the light. 
 *
 * Disabled lights are not checked during verification, and do not respond to 
 * Start/Pause/Resume/Stop commands. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   enable      Specifies whether to enable or disable the light.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_Enable(kLight light, kBool enable);

/** 
 * Reports whether the light is currently enabled. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @return              Returns whether the light is currently enabled. 
 */
kFsFx(kBool) kLight_IsEnabled(kLight light);

/** 
 * Sets the light control mode.
 *
 * For control modes that support multiple sources (e.g. kLIGHT_CONTROL_TRIGGERED_BY_EVENT), 
 * use the kLight_SetControlId function to specify a source.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   type        Light control mode.
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_SetControl(kLight light, kLightControl type);

/** 
 * Reports the light control mode. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @return              Returns the control mode of the light. 
 */
kFsFx(kLightControl) kLight_Control(kLight light);

/** 
 * Reports constraint and validity information for the Control setting. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_ControlInfo(kLight light, kInfoBits* info);

/** 
 * Sets the control source id for a light. 
 *
 * For control modes that support multiple sources (e.g. kLIGHT_CONTROL_TRIGGERED_BY_EVENT), 
 * this function cam be used to specify a specific source id.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   id          Specifies the control source id.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_SetControlId(kLight light, k32u id);

/** 
 * Gets the control source id for a light.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @return              Control source id.
 */
kFsFx(k32u) kLight_ControlId(kLight light);

/** 
 * Reports constraint and validity information for the ControlId setting. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_ControlIdInfo(kLight light, kInfo* info);

/** 
 * Sets the light priming mode.
 *
 * Some light devices rely on stored charge to provide a near-instantaneous response when the 
 * physical light exposure signal is asserted. Unfortunately, this charge can dissipate when the light 
 * is not in use. Light priming can be used to combat this effect, improving the consistency of 
 * illumination during nominal exposure time. 
 * 
 * Three priming modes are supported. The kLIGHT_PRIMING_MODE_ALWAYS mode uses a lead-in offset 
 * to allocate time for light priming in every DEG (Delay, Exposure, Gap) cycle. This mode can 
 * reduce maximum frame rates for exposure-limited configurations but will produce uniform exposures 
 * even with sparse event triggers (e.g., continuous encoder-based triggering or software triggering, 
 * where there may be gaps of arbitrary duration between exposures). Note that if light timing is coupled 
 * to camera timing (kLIGHT_CONTROL_COUPLED_TO_CAMERA) this setting will only be honored if the light 
 * coupling mode is set to one of kLIGHT_COUPLING_MODE_LEAD_IN or kLIGHT_COUPLING_MODE_FULL.
 * This is the recommended mode for new applications and may become the default in a future FireSync 
 * platform version.
 * 
 * In contrast, the kLIGHT_PRIMING_MODE_ONCE mode performs light priming during acquisition startup. 
 * This will typically result in a brief light exposure before the start of scanning. This mode is suitable 
 * for continuous time-based event triggering, where a steady sequence of triggers is guaranteed after 
 * acquisition startup (effectively, keeping the light primed). Use of this mode does not impact 
 * frame rates.
 * 
 * Finally, the kLIGHT_PRIMING_MODE_NONE mode disables light priming. This is the default behavior, 
 * for compatibility with older clients. However, it is recommended to use one of the two priming 
 * modes described above in newer applications.
 * 
 * This option is not configurable for DM647 (M2, M50) controllers.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   mode        Light priming mode.
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_SetPrimingMode(kLight light, kLightPrimingMode mode);

/** 
 * Reports the light priming mode. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @return              Returns the priming mode of the light. 
 */
kFsFx(kLightPrimingMode) kLight_PrimingMode(kLight light);

/** 
 * Reports constraint and validity information for the PrimingMode setting. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_PrimingModeInfo(kLight light, kInfoBits* info);

/** 
 * Sets the light coupling mode.
 *
 * The light coupling mode affects exposure timing when the light control mode is set to 
 * kLIGHT_CONTROL_COUPLED_TO_CAMERA. Three coupling modes are supported. 
 *
 * In kLIGHT_COUPLING_MODE_FULL mode, light and camera timing are coordinated to take all lead-in/lead-out 
 * timing offsets into account. For example, if the light priming mode is set to kLIGHT_PRIMING_MODE_ALWAYS, 
 * then the light will often have a non-zero lead-in time, and this offset will be factored into coupled 
 * camera+light exposure timing. Similarly, camera lead-out is taken into account such that the light exposes 
 * for the full extent of the camera's nominal exposure time rather than only the physical imager exposure time. 
 * The result is that exposures are optimally synchronized between cameras and lights. This is the recommended 
 * mode for new applications and may become the default in a future FireSync platform version.
 *
 * In kLIGHT_COUPLING_MODE_NONE mode, light timing is coupled to physical imager exposure timing. 
 * Light lead-in/lead-out are not taken into account in camera or light timing, and camera lead-in/lead-out 
 * are not taken into account in light timing. This was the original FireSync platform behavior and (as a 
 * result) is still the default behavior. However, this mode is supported for backwards compatibility only.

 * In kLIGHT_COUPLING_MODE_LEAD_IN mode, light and camera lead-ins are coordinated, but leads-outs are not.
 * This mode was introduced as an intermediate migration step for existing applications. The benefits of this mode 
 * are that a) per-frame light priming (kLIGHT_PRIMING_MODE_ALWAYS) can be factored into coupled camera timing, 
 * and b) the effective exposure for coupled configurations remains very similar to the original FireSync 
 * platform behavior (in contrast with kLIGHT_COUPLING_MODE_FULL, which can increase effective exposure for 
 * the same nominal camera exposure value due to lead-out coordination). 
 *  
 * This option is not configurable for DM647 (M2, M50) controllers, which only support kLIGHT_COUPLING_MODE_NONE.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   mode        Light priming mode.
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_SetCouplingMode(kLight light, kLightCouplingMode mode);

/** 
 * Reports the light coupling mode. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @return              Returns the priming mode of the light. 
 */
kFsFx(kLightCouplingMode) kLight_CouplingMode(kLight light);

/** 
 * Reports constraint and validity information for the CouplingMode setting. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_CouplingModeInfo(kLight light, kInfoBits* info);

/** 
 * Sets intensity associated with this light.  
 *
 * By default, light intensity is set to 1.0 (100%). In this case, intensity is set to the maximum 
 * safe value (consistent with manufacturing-time calibration, if any). 
 * 
 * This feature requires intensity adjustment support in underlying light hardware, which 
 * may not be available for all light devices.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   intensity   Desired light intensity (normalized).
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_SetIntensity(kLight light, k64f intensity);

/** 
 * Gets the intensity associated with this light. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @return              Light intensity (normalized). 
 */
kFsFx(k64f) kLight_Intensity(kLight light);

/** 
 * Reports constraint and validity information for the Intensity setting. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_IntensityInfo(kLight light, kInfo64f* info);

/** 
 * Sets the light mask source type.
 *
 * The light mask setting can optionally be used to specify a signal to mask light output. 
 * Light output will be active only when the mask signal is asserted (subject to polarity setting).  
 * 
 * For mask source types that support multiple source instances (e.g. kLIGHT_MASK_NODE_INPUT), 
 * use the kLight_SetMaskId function to specify the instance id.
 * 
 * Light masking is asynchronous to and independent from exposure sequence timing. Accordingly, 
 * masking can start or stop in the middle of an exposure. Exposure sequences continue to be 
 * scheduled while masking is in effect. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   mask        Light mask source type.
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_SetMask(kLight light, kLightMask mask);

/** 
 * Reports the light mask source type.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @return              Light mask source type. 
 */
kFsFx(kLightMask) kLight_Mask(kLight light);

/** 
 * Reports constraint and validity information for the Mask setting. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_MaskInfo(kLight light, kInfoBits* info);

/** 
 * Sets the light mask source id. 
 *
 * For mask modes that support multiple source instances (e.g. kLIGHT_MASK_MASTER_INPUT), 
 * this function cam be used to specify a specific source id.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   id          Specifies the mask source id.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_SetMaskId(kLight light, k32u id);

/** 
 * Gets the light mask source id.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @return              Mask source id.
 */
kFsFx(k32u) kLight_MaskId(kLight light);

/** 
 * Reports constraint and validity information for the MaskId setting. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_MaskIdInfo(kLight light, kInfo* info);

/** 
 * Sets the light mask polarity.  
 * 
 * Light output is active only when the light mask source signal is asserted. The 
 * mask polarity setting determines whether the mask signal is interpreted as 
 * active high (default) or active low. 
 *
 * @public               @memberof kLight
 * @param   light        Light object.  
 * @param   isActiveHigh Specifies the polarity.  
 * @return               Operation status. 
 */
kFsFx(kStatus) kLight_SetMaskPolarity(kLight light, kBool isActiveHigh);

/** 
 * Gets the light mask polarity.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @return              kTRUE if active high. 
 */
kFsFx(kBool) kLight_MaskPolarity(kLight light);

/** 
 * Reports constraint and validity information for the MaskPolarity setting. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_MaskPolarityInfo(kLight light, kInfoBool* info);

/**
* Enables light power-saver mode.
*
* This option can be used to reduce power and increase light lifespan by turning lights off 
* automatically after a period of encoder inactivity.
* 
* Light masking and lockout have priority over the power saver setting. 
*
* @see  kNode_SetPowerSaverTimeout, kNode_SetPowerSaverThreshold
*
* @public              @memberof kLight
* @param   light       Light object.
* @param   enabled     Power saver setting.
* @return              Operation status.
*/
kFsFx(kStatus) kLight_EnablePowerSaver(kLight light, kBool enabled);

/**
* Reports whether power saver is currently enabled.
*
* @public              @memberof kLight
* @param   light       Light object.
* @return              kTRUE if power saver is enabled; kFALSE otherwise.
*/
kFsFx(kBool) kLight_PowerSaverEnabled(kLight light);

/**
* Reports constraint and validity information for the PowerSaverEnabled property.
*
* @public              @memberof kLight
* @param   light       Light object.
* @param   info        Receives the info structure.
* @return              Operation status.
*/
kFsFx(kStatus) kLight_PowerEnabledSaverInfo(kLight light, kInfoBool* info);

/** 
 * Sets the maximum light duty cycle. 
 * 
 * The light duty cycle imposes an additional constraint that affects the light 
 * minimum gap setting, as well as the minimum gap of any devices to which the 
 * light might be slaved (e.g., camera). 
 *
 * @public               @memberof kLight
 * @param   light        Light object.  
 * @param   dutyCycle    Maximum duty cycle (percentage).   
 * @return               Operation status. 
 */
kFsFx(kStatus) kLight_SetMaxDutyCycle(kLight light, k64f dutyCycle);

/** 
 * Gets the maximum light duty cycle setting.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @return              Duty cycle (percentage). 
 */
kFsFx(k64f) kLight_MaxDutyCycle(kLight light);

/** 
 * Reports constraint and validity information for the MaxDutyCycle setting. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_MaxDutyCycleInfo(kLight light, kInfo64f* info);

/**
* Controls the light lockout state. 
* 
* The light lockout feature can be used to suppress light output under software 
* control. The light lockout feature can be used in any light control mode, including SetBySoftware. 
* 
* The effect of the LockOut method is immediate. Lockout state is not recorded in node 
* configuration settings and is therefore not persistent. 
* 
* Light lockout is asynchronous to exposure sequences. Accordingly, lockout can 
* occur in the middle of an exposure. Exposure sequences continue to be scheduled 
* while light lockout is in effect. 
*
* @public              @memberof kLight
* @param   light       Light object.
* @param   shouldLock  Forces the output to be turned on or off.
* @return              Operation status.
*/
kFsFx(kStatus) kLight_LockOut(kLight light, kBool shouldLock);

/**
* Reports whether light lockout is currently in effect.
*
* @public              @memberof kLight
* @param   light       Light object.
* @return              kTRUE if output is supressed; kFALSE otherwise.
*/
kFsFx(kBool) kLight_IsLockedOut(kLight light);

/**
* Reports constraint and validity information for the LockOut property.
*
* @public              @memberof kLight
* @param   light       Light object.
* @param   info        Receives the info structure.
* @return              Operation status.
*/
kFsFx(kStatus) kLight_LockedOutInfo(kLight light, kInfoBool* info);

/** 
 * Creates and adds a new state with default values.
 *
 * The new state is added to the end of the light's state list. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   state       Receives the new state object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_AddState(kLight light, kLightState* state);

/** 
 * Destroys the specified state and removes it from the light's state list.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   state       State object to be deleted.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_DeleteState(kLight light, kLightState state);

/** 
 * Swaps the order of two states in the light's state list.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   index0      Index of the first state object to be swapped.  
 * @param   index1      Index of the second state object to be swapped.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_SwapStates(kLight light, kSize index0, kSize index1);

/** 
 * Gets the number of states belonging to the light. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @return              Returns the number of states belonging to the light. 
 */
kFsFx(kSize) kLight_StateCount(kLight light);

/** 
 * Reports constraint and validity information for the StateCount setting. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_StateCountInfo(kLight light, kInfoSize* info);

/** 
 * Gets the state at the specified index.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   index       Index of the state.  
 * @return              Returns the state at the specified index.
 */
kFsFx(kLightState) kLight_StateAt(kLight light, kSize index);

/** 
 * Creates and adds a new light control extension of the specified type.
 *
 * The new extension is added to the end of the light's extension list. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   type        Type of extension to add. 
 * @param   extension   Receives the new extension object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_AddExtension(kLight light, kType type, kLightExt* extension);

/** 
 * Destroys the specified light control extension and removes it from the light's extension list.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   extension   Extension object to be deleted.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_DeleteExtension(kLight light, kLightExt extension);

/** 
 * Gets the number of control extensions that have been configured.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @return              Count of light control extensions. 
 */
kFsFx(kSize) kLight_ExtensionCount(kLight light);

/** 
 * Gets the control extension at the specified index.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   index       Index of the extension.  
 * @return              Extension at the specified index.
 */
kFsFx(kLightExt) kLight_ExtensionAt(kLight light, kSize index);

/** 
 * Reports the time required by this light to respond to an event. 
 *
 * Minimum period values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public             @memberof kLight
 * @param   light      Light object.  
 * @return             Minimum period, in nanoseconds. 
 */
kFsFx(k64u) kLight_MinimumPeriod(kLight light);

/** 
 * Sets output state of the light in SetBySoftware mode.
 *
 * In SetBySoftware control mode, the SetOutputState function can be used at any time 
 * (e.g., node Ready or Running states) to turn the light on or off. 
 * 
 * Light masking and lockout features have greater priority than the output state 
 * specified via the SetOutputState function.  
 *  
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   state       New light output state (kTRUE: on, kFALSE: off). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_SetOutputState(kLight light, kBool state);

/** 
 * Gets light statistics.
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @param   stats       Receives the stats structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_Stats(kLight light, kLightStats* stats);

/** 
 * Provides a software trigger to a light. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kLight_Trigger(kLight light);

/**
 * Provides a list of configuration overrides to modify live light behaviour. 
 * 
 * This function is used in conjunction with the "Override" extension (kLightOverride). When 
 * the override extension is enabled, and node acquisition is started, this function can be 
 * used to modify specific light settings without having to stop/restart the node. This feature 
 * can be used to implement low-speed dynamic exposure control in software. 
 *
 * @public              @memberof kLight
 * @param   light       Light object.
 * @param   overrides   Override items.
 * @param   count       Count of the override items.
 * @return              Operation status.
 */
kFsFx(kStatus) kLight_Override(kLight light, const kLightOverrideItem* overrides, kSize count);

#include <kFireSync/Client/kLight.x.h>

#endif
