/** 
 * @file    kNet.x.h
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_NET_X_H
#define K_FIRESYNC_NET_X_H

#define kNET_DEFAULT_IP_ADDRESS_BASE             (0x0A000000)    ///< Base for default node IP address (add node id for default address). 
#define kNET_RESCUE_IP_ADDRESS                   (0xC0A8010A)    ///< Default node Rescue IP address. 
#define kNET_RESCUE_PREFIX_LENGTH                (16)            ///< Default node Rescue Prefix length. 

#define kNET_DATA_OUT_DEFAULT_PORT_COUNT         (1)             ///< Default number of output block ports. 
#define kNET_DATA_OUT_DEFAULT_MAX_QUEUE_SIZE     (1 << 23)       ///< Default maximum per-channel queue size (bytes). 
#define kNET_DATA_OUT_DEFAULT_MAX_QUEUE_COUNT    (1 << 12)       ///< Default maximum per-channel queue item count. 
#define kNET_DATA_OUT_DEFAULT_SOCKET_BUFFER_SIZE (1 << 13)       ///< Default maximum per-output-channel socket buffer size (bytes).

#define kNET_DATA_IN_DEFAULT_SOCKET_BUFFER_SIZE  (1 << 13)       ///< Default maximum per-input-channel socket buffer size (bytes). 

#define kNET_DEFAULT_WEB_PORT                   (80)            ///< Web server port. 

typedef struct kNetClass
{
    kObjectClass base; 

    kNode node;                                 //Node (hierarchy ancestor).
    kPipe pipe;                                 //Pipe configuration object. 

    kArrayList interfaces;                      //Network interfaces (kArrayList<kNetInterface>)
    kNetInterface dummyInterface;               //Dummy network interface, simplifying compatibility logic.

    kSyncSource syncSource;                     //Node synchronization source. 

    k32u discoveryPort;                         //Current discovery UDP port. 
    k32u nextDiscoveryPort;                     //Discovery UDP port after reboot. 
    k32u controlPort;                           //Current control TCP port. 
    k32u nextControlPort;                       //Control TCP port after reboot. 
    k32u healthPort;                            //Current health TCP port. 
    k32u nextHealthPort;                        //Health TCP port after reboot. 
    k32u dataPort;                              //Current data TCP port. 
    k32u nextDataPort;                          //Data TCP port after reboot. 
    k32u webPort;                               //Current web TCP port. 
    k32u nextWebPort;                           //Web TCP port after reboot. 

    kSize dataOutPortCount;                     //Number of output block ports. 
    kSize dataOutMaxQueueSize;                  //Maximum total size of per-channel message queues (bytes). 
    kSize dataOutMaxQueueCount;                 //Maximum item count of per-channel message queues. 
    kSize dataOutSocketBufferSize;              //Output socket buffer size, in bytes. 
    kSize dataOutDefaultSocketBufferSize;       //Default output socket buffer size, in bytes.
    kThreadPriorityClass dataOutThreadPriorityClass;   //Desired thread priority class (high, normal, low).
    k32s dataOutThreadPriorityOffset;           //Desired thread priority offset within class.
    kBitArray dataOutThreadAffinity;            //Desired thread affinity.

    kBlock dataOutBlock;                        //Network output system block.  

    kSize dataInSocketBufferSize;               //Input socket buffer size, in bytes. 
    kSize dataInDefaultSocketBufferSize;        //Default input socket buffer size, in bytes. 
    kThreadPriorityClass dataInThreadPriorityClass;   //Desired thread priority class (high, normal, low).
    k32s dataInThreadPriorityOffset;            //Desired thread priority offset within class.
    kBitArray dataInThreadAffinity;             //Desired thread affinity.

    kArrayList dataInSources;                   //Data input port information -- kArrayList<kDataInSource>. 
    kBlock dataInBlock;                         //Network input system block.  

} kNetClass; 

kDeclareClassEx(kFs, kNet, kObject)
        
kFsFx(kStatus) kNet_Construct(kNet* net, kNode node, kAlloc allocator); 

kFsFx(kStatus) kNet_Init(kNet net, kType type, kNode node, kAlloc alloc); 
kFsFx(kStatus) kNet_VRelease(kNet net); 

//deprecated -- use kNetInterface instead
kFsFx(kNetworkSpeed) kNet_Speed(kNet net);
kFsFx(kNetworkSpeed) kNet_NextSpeed(kNet net);
kFsFx(kStatus) kNet_SetNextSpeed(kNet net, kNetworkSpeed speed);
kFsFx(kStatus) kNet_NextSpeedInfo(kNet net, kInfoBits* info);
kFsFx(k32u) kNet_TransmitRate(kNet net);
kFsFx(kStatus) kNet_SetTransmitRate(kNet net, k32u rate);
kFsFx(kStatus) kNet_TransmitRateInfo(kNet net, kInfo32u* info);
kFsFx(const kIpConfig*) kNet_IpConfig(kNet net);
kFsFx(const kIpConfig*) kNet_NextIpConfig(kNet net);
kFsFx(kStatus) kNet_SetNextIpConfig(kNet net, const kIpConfig* config);
kFsFx(kMacAddress) kNet_MacAddress(kNet net);

kFsFx(kStatus) kNet_ParseDeviceInterface(kNet net, kXml xml, kXmlItem item);
kFsFx(kStatus) kNet_ParseDeviceInterfaces(kNet net, kXml xml, kXmlItem item);
kFsFx(kStatus) kNet_ParseDevice(kNet net, kXml xml, kXmlItem item); 

kFsFx(kStatus) kNet_FormatDeviceInterface(kNet net, kXml xml, kXmlItem item);
kFsFx(kStatus) kNet_FormatDeviceInterfaces(kNet net, kXml xml, kXmlItem item);
kFsFx(kStatus) kNet_FormatDevice(kNet net, kXml xml, kXmlItem item); 

kFsFx(kStatus) kNet_ParseInterface(kNet net, kXml xml, kXmlItem item);
kFsFx(kStatus) kNet_ParseInterfaces(kNet net, kXml xml, kXmlItem item);
kFsFx(kStatus) kNet_Parse(kNet net, kXml xml, kXmlItem item); 
kFsFx(kStatus) kNet_ParseDataOut(kNet net, kXml xml, kXmlItem item); 
kFsFx(kStatus) kNet_ParseDataIn(kNet net, kXml xml, kXmlItem item); 
kFsFx(kStatus) kNet_ParseHealth(kNet net, kXml xml, kXmlItem item); 

kFsFx(kStatus) kNet_FormatInterface(kNet net, kXml xml, kXmlItem item);
kFsFx(kStatus) kNet_FormatInterfaces(kNet net, kXml xml, kXmlItem item);
kFsFx(kStatus) kNet_Format(kNet net, kXml xml, kXmlItem item); 
kFsFx(kStatus) kNet_FormatDataOut(kNet net, kXml xml, kXmlItem item); 
kFsFx(kStatus) kNet_FormatDataIn(kNet net, kXml xml, kXmlItem item); 
kFsFx(kStatus) kNet_FormatHealth(kNet net, kXml xml, kXmlItem item); 

kFsFx(kStatus) kNet_ClearEx(kNet net, kNodeClearOption options); 
kFsFx(kStatus) kNet_ClearAffinity(kNet net, kBitArray affinity);

kFsFx(kStatus) kNet_Verify(kNet net); 

kFsFx(kStatus) kNet_RegisterDataOutBlock(kNet net); 
kFsFx(kStatus) kNet_UnregisterDataOutBlock(kNet net); 
kFsFx(kStatus) kNet_UpdateDataOutBlock(kNet net); 

kFsFx(kStatus) kNet_RegisterDataInBlock(kNet net); 
kFsFx(kStatus) kNet_UnregisterDataInBlock(kNet net); 
kFsFx(kStatus) kNet_UpdateDataInBlock(kNet net); 

kFsFx(k32u) kNet_DiscoveryPort(kNet net);
kFsFx(kStatus) kNet_SetNextDiscoveryPort(kNet net, k32u port);
kFsFx(k32u) kNet_NextDiscoveryPort(kNet net);

kFsFx(k32u) kNet_ControlPort(kNet net);
kFsFx(kStatus) kNet_SetNextControlPort(kNet net, k32u port);
kFsFx(k32u) kNet_NextControlPort(kNet net);

kFsFx(k32u) kNet_HealthPort(kNet net);
kFsFx(kStatus) kNet_SetNextHealthPort(kNet net, k32u port);
kFsFx(k32u) kNet_NextHealthPort(kNet net);

kFsFx(k32u) kNet_DataPort(kNet net);
kFsFx(kStatus) kNet_SetNextDataPort(kNet net, k32u port);
kFsFx(k32u) kNet_NextDataPort(kNet net);

kFsFx(k32u) kNet_WebPort(kNet net);
kFsFx(kStatus) kNet_SetNextWebPort(kNet net, k32u port);
kFsFx(k32u) kNet_NextWebPort(kNet net);

kFsFx(kNode) kNet_Node(kNet net);

#endif

