/** 
 * @file    kNetInterface.h
 * @brief   Declares the kNetInterface class. 
 *
 * @internal
 * Copyright (C) 2018-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_NET_INTERFACE_H
#define K_FIRESYNC_NET_INTERFACE_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kNetInterface
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents configuration for network interfaces. 
 */
//typedef kObject kNetInterface;        --forward-declared in kFsDef.x.h

/** 
 * Resets network interface mode settings to defaults.
 *
 * @public                  @memberof kNetInterface
 * @param   netInterface    NetInterface object.  
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNetInterface_Clear(kNetInterface netInterface);

/** 
 * Resets network interface mode and device settings to defaults.
 *
 * @public                  @memberof kNetInterface
 * @param   netInterface    NetInterface object.  
 * @param   clearIpConfig   Should the node's IP configuration be reset?
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNetInterface_ClearAll(kNetInterface netInterface, kBool clearIpConfig);

/** 
 * Gets the current network interface speed configuration. 
 *
 * @public                  @memberof kNetInterface
 * @param   netInterface    NetInterface object.  
 * @return                  Current network interface speed configuration.
 */
kFsFx(kNetworkSpeed) kNetInterface_Speed(kNetInterface netInterface);

/** 
 * Reports the network interface speed configuration that will take effect after reset. 
 *
 * @public                  @memberof kNetInterface
 * @param   netInterface    NetInterface object.  
 * @return                  Future network interface speed configuration.
 */
kFsFx(kNetworkSpeed) kNetInterface_NextSpeed(kNetInterface netInterface);

/** 
 * Sets the network interface speed configuration that will take effect after reset. 
 * 
 * After changing a node's IP address and resetting, use kSystem_HasChanges/kSystem_Refresh 
 * to re-detect the node.
 *
 * @public                  @memberof kNetInterface
 * @param   netInterface    NetInterface object.  
 * @param   speed           Future netInterfacework speed configuration. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNetInterface_SetNextSpeed(kNetInterface netInterface, kNetworkSpeed speed);

/** 
 * Reports constraint and validity information for the NextSpeed setting. 
 *
 * @public                  @memberof kNetInterface
 * @param   netInterface    NetInterface object. 
 * @param   info            Receives the info structure.  
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNetInterface_NextSpeedInfo(kNetInterface netInterface, kInfoBits* info);

/**
* Reports the configured maximum network interface transmission rate. 
*
* This setting can be used to scale down the network interface transmission rate. This feature requires 
* hardware support for inter-packet or inter-word gap stretching, which may not be available
* on all devices (e.g., Zynq-7000: Yes; Dm647: No). 
* 
* @public                   @memberof kNetInterface
* @param   netInterface     NetInterface object.
* @return                   Network interface transmission rate (percentage).
*/
kFsFx(k32u) kNetInterface_TransmitRate(kNetInterface netInterface);

/**
* Sets the maximum network interface transmission rate. 
*
* @public                   @memberof kNetInterface
* @param   netInterface     NetInterface object.
* @param   rate             network interface transmission rate (percentage).
* @return                   Operation status.
*/
kFsFx(kStatus) kNetInterface_SetTransmitRate(kNetInterface netInterface, k32u rate);

/**
* Reports constraint and validity information for the maximum transmit rate setting.
*
* @public                   @memberof kNetInterface
* @param   netInterface     NetInterface object.
* @param   info             Receives the info structure.
* @return                   Operation status.
*/
kFsFx(kStatus) kNetInterface_TransmitRateInfo(kNetInterface netInterface, kInfo32u* info);

/** 
 * Gets the current node IP configuration. 
 *
 * @public                  @memberof kNetInterface
 * @param   netInterface    NetInterface object.  
 * @return                  Current IP configuration.
 */
kFsFx(const kIpConfig*) kNetInterface_IpConfig(kNetInterface netInterface);

/** 
 * Reports the IP configuration that will take effect after reset. 
 *
 * @public                  @memberof kNetInterface
 * @param   netInterface    NetInterface object.  
 * @return                  Future IP configuration.
 */
kFsFx(const kIpConfig*) kNetInterface_NextIpConfig(kNetInterface netInterface);

/** 
 * Sets the IP configuration that will take effect after reset. 
 * 
 * After changing a node's IP address and resetting, use kSystem_HasChanges/kSystem_Refresh 
 * to re-detect the node.
 *
 * @public                  @memberof kNetInterface
 * @param   netInterface    NetInterface object.  
 * @param   config          Future IP configuration. 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNetInterface_SetNextIpConfig(kNetInterface netInterface, const kIpConfig* config);

/** 
 * Gets the current node MAC address. 
 *
 * @public                  @memberof kNetInterface
 * @param   netInterface    NetInterface object.  
 * @return                  Current MAC address.
 */
kFsFx(kMacAddress) kNetInterface_MacAddress(kNetInterface netInterface);

#include <kFireSync/Client/kNetInterface.x.h>

#endif
