/** 
 * @file    kStorage.h
 * @brief   Declares the kStorage class. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_STORAGE_H
#define K_FIRESYNC_STORAGE_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kStorage
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents node file services.
 * 
 * This class can be used to work with files and directories on a remote node. Use of the storage module
 * requires that the node has a valid control connection (e.g., after successful kNode_Connect). Use of the 
 * storage module also requires that the remote node is running FireSync 6+ firmware; if these minimum
 * conditions are met, storage functionality should be available. 
 * 
 * All methods accept relative or absolute paths. When using methods such as kStorage_ListFiles that expect 
 * a path to a directory, "." can be used to refer to the current working directory. By default, the currrent 
 * working directory is set to "/user", which corresponds to the standard non-volatile storage location for 
 * application data files.
 * 
 * Though the methods provided by this class allow for directories to be created or destroyed, this feature
 * isn't necessarily supported in all remote directories. For example, as of mid-2013, DM647-based sensors 
 * support the creation/deletion of directories only within the "/temp" folder (volatile, DDR2-based storage). 
 */
//typedef kObject kStorage;        --forward-declared in kFsDef.x.h

/** 
 * Checks that the specified file exists.
 *
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   filePath    File path.  
 * @return              Returns whether the specified file exists.
 */
kFsFx(kBool) kStorage_FileExists(kStorage storage, const kChar* filePath);

/** 
 * Gets the size of the specified file.
 *
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   filePath    File path.  
 * @return              Returns the size of the specified file.
 */
kFsFx(kSize) kStorage_FileSize(kStorage storage, const kChar* filePath);

/** 
 * Copies a file from node storage to the specified local destination.
 *
 * The destination file will be overwritten if it already exists. 
 *
 * If the remote node supports progress feedback, the specified callback will be invoked to provide updates 
 * on the progress of the operation. The callback 'args' parameter will receive a k32u value representing the 
 * percentage completed. 
 * 
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   sourcePath  Source file path.  
 * @param   destPath    Destination file path.  
 * @param   progress    Optional progress callback (can be kNULL). 
 * @param   context     Callback context.
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_DownloadToFile(kStorage storage, const kChar* sourcePath, const kChar* destPath, kCallbackFx progress, kPointer context);

/** 
 * Reads the contents of a storage file into a memory buffer. 
 * 
 * This function will fail if the capacity of the buffer does not match the size of the file. 
 *
 * If the remote node supports progress feedback, the specified callback will be invoked to provide updates 
 * on the progress of the operation. The callback 'args' parameter will receive a k32u value representing the 
 * percentage completed. 
 * 
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   filePath    File path.  
 * @param   buffer      Receives data read from the file.  
 * @param   capacity    Buffer capacity.   
 * @param   progress    Optional progress callback (can be kNULL). 
 * @param   context     Callback context.
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_DownloadToBuffer(kStorage storage, const kChar* filePath, void* buffer, kSize capacity, kCallbackFx progress, kPointer context);

/** 
 * Reads the contents of a storage file into a local stream. 
 * 
 * If the remote node supports progress feedback, the specified callback will be invoked to provide updates 
 * on the progress of the operation. The callback 'args' parameter will receive a k32u value representing the 
 * percentage completed. 
 * 
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   filePath    Source file path.  
 * @param   stream      Destination stream. 
 * @param   progress    Optional progress callback (can be kNULL). 
 * @param   context     Callback context.
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_DownloadToStream(kStorage storage, const kChar* filePath, kStream stream, kCallbackFx progress, kPointer context);

/** 
 * Copies the specified local file into node storage.
 *
 * The destination file will be overwritten if it already exists. 
 *
 * If the remote node supports progress feedback, the specified callback will be invoked to provide updates 
 * on the progress of the operation. The callback 'args' parameter will receive a k32u value representing the 
 * percentage completed. 
 * 
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   sourcePath  Source file path.  
 * @param   destPath    Destination file path.  
 * @param   progress    Optional progress callback (can be kNULL). 
 * @param   context     Callback context.
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_UploadFromFile(kStorage storage, const kChar* sourcePath, const kChar* destPath, kCallbackFx progress, kPointer context);

/** 
 * Write the contents of a memory buffer to a storage module file.
 *
 * The destination file will be overwritten if it already exists. 
 *
 * If the remote node supports progress feedback, the specified callback will be invoked to provide updates 
 * on the progress of the operation. The callback 'args' parameter will receive a k32u value representing the 
 * percentage completed. 
 * 
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   filePath    Destination file path.  
 * @param   buffer      Data buffer to write.  
 * @param   size        Buffer capacity.   
 * @param   progress    Optional progress callback (can be kNULL). 
 * @param   context     Callback context.
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_UploadFromBuffer(kStorage storage, const kChar* filePath, const void* buffer, kSize size, kCallbackFx progress, kPointer context);

/** 
 * Write the contents of a local stream to a storage file, with progress updates.
 * 
 * If the remote node supports progress feedback, the specified callback will be invoked to provide updates 
 * on the progress of the operation. The callback 'args' parameter will receive a k32u value representing the 
 * percentage completed. 
 * 
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   filePath    Destination file path.  
 * @param   stream      Source stream. 
 * @param   size        Count of bytes to write.    
 * @param   progress    Optional progress callback (can be kNULL). 
 * @param   context     Callback context.
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_UploadFromStream(kStorage storage, const kChar* filePath, kStream stream, kSize size, kCallbackFx progress, kPointer context);

/** 
 * Copies a file from one location to another within sensor storage.  
 *
 * If the remote node supports progress feedback, the specified callback will be invoked to provide updates 
 * on the progress of the operation. The callback 'args' parameter will receive a k32u value representing the 
 * percentage completed. 
 * 
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   sourcePath  Source file path.  
 * @param   destPath    Destination file path. 
 * @param   progress    Optional progress callback (can be kNULL). 
 * @param   context     Callback context.
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_Copy(kStorage storage, const kChar* sourcePath, const kChar* destPath, kCallbackFx progress, kPointer context);

/** 
 * Deletes the specified file.
 *
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   filePath    File path.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_Delete(kStorage storage, const kChar* filePath);

/** 
 * Lists the files in the specified directory. 
 * 
 * Use kArrayList_Purge to destroy the file names returned by this function.
 * 
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   directory   Full path of the directory. 
 * @param   files       Receives file names (kArrayList<kString>). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_ListFiles(kStorage storage, const kChar* directory, kArrayList files);

/** 
 * Lists the sub-directories in the specified directory. 
 * 
 * Use kArrayList_Purge to destroy the directory names returned by this function.
 * 
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   directory   Full path of the directory. 
 * @param   directories Receives directory names (kArrayList<kString>). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_ListDirectories(kStorage storage, const kChar* directory, kArrayList directories); 

/** 
 * Lists the file system entries in the specified directory. 
 * 
 * Use kArrayList_Purge to destroy the names returned by this function.
 * 
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   directory   Full path of the directory. 
 * @param   entries     Receives entry names (kArrayList<kString>).  
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_ListEntries(kStorage storage, const kChar* directory, kArrayList entries); 

/** 
 * Retrieves information about each file system entry within the specified directory. 
 * 
 * Use kArrayList_Purge to destroy the entries returned by this function.
 *
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   directory   Full path of the directory. 
 * @param   entries     Receives information about each directory entry (kArrayList<kStorageItem>).
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_ListEntriesEx(kStorage storage, const kChar* directory, kArrayList entries);

/** 
 * Creates a directory at the specified location. 
 *
 * This function will fail if the directory already exists or if the parent directory does 
 * not support sub-directories. 
 * 
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   directory   Full path of the directory. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_CreateDirectory(kStorage storage, const kChar* directory); 

/** 
 * Deletes the specified directory, including all of its contents. 
 *
 * This function will fail if the specified directory cannot be deleted. 
 * 
 * @public              @memberof kStorage
 * @param  storage      Storage object.  
 * @param  directory    Full path of the directory. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_DeleteDirectory(kStorage storage, const kChar* directory);

/** 
 * Deletes all user and temp files. 
 * 
 * This function deletes all files from the '/user' and '/temp' directories, except for a select group of 
 * files that are considered to be FireSync platform assets. This white list includes files such 
 * as the mode/device configuration files loaded at boot time. 
 *
 * @public              @memberof kStorage
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_Wipe(kStorage storage);

/** 
 * Gets the directory that should be used for application configuration/resource files. 
 *
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   directory   Returns the full path of the requested directory. 
 * @param   capacity    Maximum number of characters (including null terminator).   
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_UserDirectory(kStorage storage, kChar* directory, kSize capacity);

/** 
 * Gets the directory used for executables and system configuration files. 
 *
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   directory   Returns the full path of the requested directory. 
 * @param   capacity    Maximum number of characters (including null terminator).   
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_SystemDirectory(kStorage storage, kChar* directory, kSize capacity);

/** 
 * Gets the directory that should be used for temporary files. 
 *
 * @public              @memberof kStorage
 * @param   storage     Storage object.  
 * @param   directory   Returns the full path of the requested directory. 
 * @param   capacity    Maximum number of characters (including null terminator).   
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorage_TempDirectory(kStorage storage, kChar* directory, kSize capacity);

#include <kFireSync/Client/kStorage.x.h>

#endif
