/** 
 * @file    kTestJig.h
 * @brief   Declares the kTestJig type. 
 *
 * @internal
 * Copyright (C) 2014-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_TEST_JIG_H
#define K_FIRESYNC_TEST_JIG_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kTestJig
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents M2 test jig configuration.
 */
//typedef kObject kTestJig;        --forward-declared in kFsDef.x.h

/** 
 * Resets test output device settings to defaults. 
 *
 * @public              @memberof kTestJig
 * @param   test        Test jig object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTestJig_Clear(kTestJig test);

/** 
 * Reset the I/O test module.
 *
 * @public              @memberof kTestJig
 * @param   test        Test jig object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTestJig_Reset(kTestJig test);

/**
 * Set PL register data. Read from kTestJigStats to verify.
 *
 * @public              @memberof kTestJig
 * @param   test        Test jig object.
 * @param   data        Register data.
 * @return              Operation status.
*/
kFsFx(kStatus) kTestJig_SetPlRegister(kTestJig test, k32u data);

/**
 * Set PL output register. Read from kTestJigStats to verify.
 *
 * @public              @memberof kTestJig
 * @param   test        Test jig object.
 * @param   data        Output data.
 * @return              Operation status.
*/
kFsFx(kStatus) kTestJig_SetPlIo(kTestJig test, k32u data);

/**
 * Run PL DDR2 test.
 *
 * @public              @memberof kTestJig
 * @param   test        Test jig object.
 * @param   address     Start address.
 * @return              Operation status.
*/
kFsFx(kStatus) kTestJig_Ddr2Test(kTestJig test, k32u address);

/**
 * Run camera data test.
 *
 * @public              @memberof kTestJig
 * @param   test        Test jig object.
 * @param   data        Camera data.
 * @return              Operation status.
*/
kFsFx(kStatus) kTestJig_SetCameraData(kTestJig test, k64u data);

/**
 * Run video port test.
 *
 * @public              @memberof kTestJig
 * @param   test        Test jig object.
 * @param   channel     Video port channel.
 * @return              Operation status.
*/
kFsFx(kStatus) kTestJig_VideoPortTest(kTestJig test, k32u channel);

/**
* Generate a PL interrupt to sensor.
*
* @public              @memberof kTestJig
* @param   test        Test jig object.
* @return              Operation status.
*/
kFsFx(kStatus) kTestJig_GenerateInterrupt(kTestJig test);

/**
* Set camera power on/off.
*
* @public              @memberof kTestJig
* @param   test        Test jig object.
* @param   index       Camera index.
* @param   enable      Enable/Disable specified camera power.
* @return              Operation status.
*/
kFsFx(kStatus) kTestJig_SetCameraPower(kTestJig test, kSize index, kBool enable);

/**
* Set PL test pins output.
*
* @public              @memberof kTestJig
* @param   test        Test jig object.
* @param   plTestPin   PL test pin states.
* @return              Operation status.
*/
kFsFx(kStatus) kTestJig_SetPlTestPin(kTestJig test, k32u plTestPin);

/**
* Set DSP test pins output.
*
* @public              @memberof kTestJig
* @param   test        Test jig object.
* @param   psTestPin   PS test pin states.
* @return              Operation status.
*/
kFsFx(kStatus) kTestJig_SetPsTestPin(kTestJig test, k32u psTestPin);

/**
 * Retrieve statistics.
 *
 * @public              @memberof kTestJig
 * @param   test        Test jig object.  
 * @param   stats       Receives the stats structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kTestJig_Stats(kTestJig test, kTestJigStats* stats);

#include <kFireSync/Client/kTestJig.x.h>

#endif
