/** 
 * @file    kCudaRuntime.h
 * @brief   Collection of Cuda runtime management functions.
 *
 * @internal
 * Copyright (C) 2020-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_CUDA_RUNTIME_H
#define K_FIRESYNC_CUDA_RUNTIME_H

#include <kFireSync/Cuda/kCudaDef.h>
#include <kFireSync/Cuda/kCudaRuntime.x.h>

/**
 * @class   kCudaRuntime
 * @ingroup kFireSync-Cuda
 * @brief   Collection of static methods for Cuda runtime management and reporting.
 */

/** 
 * Reports whether Cuda integration was enabled when the kFireSync library was built.
 * 
 * @public      @memberof kCudaRuntime
 * @return      kTRUE if Cuda integration was enabled at build time.
 */
kFsFx(kBool) kCudaRuntime_IntegrationEnabled();

/** 
 * Reports the current Cuda runtime version.
 * 
 * Note, if there is no Cuda device present in the system, the Cuda runtime will not be loaded 
 * and the version cannot be determined. In this case, kERROR_NOT_FOUND may be returned. 
 * 
 * Similarly, if the Cuda driver version is incompatible, the runtime version cannot be determined. 
 * In this case kERROR_VERSION may be returned.
 * 
 * @public              @memberof kCudaRuntime
 * @param   version     Receives Cuda runtime version.
 * @return              Operation status (see notes).
 */
kFsFx(kStatus) kCudaRuntime_RuntimeVersion(kVersion* version);

/** 
 * Reports the maximum Cuda runtime version supported by the current driver.
 * 
 * @public              @memberof kCudaRuntime
 * @param   version     Receives maximum Cuda runtime version supported by driver.
 * @return              Operation status.
 */
kFsFx(kStatus) kCudaRuntime_MaximumDriverRuntimeVersion(kVersion* version);

/** 
 * Reports the minimum Cuda runtime version required by the FireSync platform.
 * 
 * @public      @memberof kCudaRuntime
 * @return      Minimum Cuda runtime version required by FireSync platform.  
 */
kFsFx(kVersion) kCudaRuntime_MinimumRuntimeVersion();

/** 
 * Reports whether Cuda runtime support is available. 
 * 
 * Note, the kCudaUtils_EnvironmentStatus function can be used to help determine the reason that Cuda runtime
 * support is not available.
 * 
 * @public      @memberof kCudaRuntime
 * @return      kTRUE if Cuda runtime support is available. 
 * @see         kCudaUtils_EnvironmentStatus
 */
kFsFx(kBool) kCudaRuntime_IsAvailable();

#endif
