/** 
 * @file    kBinCSum4Alg.h
 * @brief   Declares the kBinCSum4Alg type. 
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_BIN_CSUM4_ALG_H
#define K_FIRESYNC_BIN_CSUM4_ALG_H

#include <kFireSync/kFsDef.h>

/**
 * @class   kBinCSum4Alg
 * @extends kObject
 * @ingroup kFireSync-Data
 * @brief   Software implementation of the BinCSum4 (Binarize + Column-Sum, version 4) camera algorirthm. 
 */
//typedef kObject kBinCSum4Alg;            --forward-declared in kFsDef.x.h  

/** 
 * Constructs a kBinCSum4Alg object.
 *
 * @public              @memberof kBinCSum4Alg
 * @param   alg         Destination for the constructed object handle. 
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4Alg_Construct(kBinCSum4Alg* alg, kAlloc allocator); 

/** 
 * Sets the binarize threshold. 
 *
 * @public              @memberof kBinCSum4Alg
 * @param   alg         Algorithm object. 
 * @param   threshold   Binarize threshold. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4Alg_SetBinThreshold(kBinCSum4Alg alg, k32u threshold);

/** 
 * Reports the binarize threshold. 
 *
 * @public              @memberof kBinCSum4Alg
 * @param   alg         Algorithm object. 
 * @return              Binarize threshold. 
 */
kFsFx(k32u) kBinCSum4Alg_BinThreshold(kBinCSum4Alg alg);

/** 
 * Sets the column-sum threshold. 
 *
 * @public              @memberof kBinCSum4Alg
 * @param   alg         Algorithm object. 
 * @param   threshold   Column-sum threshold. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4Alg_SetCSumThreshold(kBinCSum4Alg alg, k32u threshold);

/** 
 * Reports the column-sum threshold. 
 *
 * @public              @memberof kBinCSum4Alg
 * @param   alg         Algorithm object. 
 * @return              Column-sum threshold. 
 */
kFsFx(k32u) kBinCSum4Alg_CSumThreshold(kBinCSum4Alg alg);

/** 
 * Sets the number of left-most columns used to calculate background noise.  
 *
 * @public              @memberof kBinCSum4Alg
 * @param   alg         Algorithm object. 
 * @param   width       Column count. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4Alg_SetBackgroundWidth(kBinCSum4Alg alg, kSize width);

/** 
 * Reports the number of left-most columns used to calculate background noise.  
 *
 * @public              @memberof kBinCSum4Alg
 * @param   alg         Algorithm object. 
 * @return              Column count.
 */
kFsFx(kSize) kBinCSum4Alg_BackgroundWidth(kBinCSum4Alg alg);

/** 
 * Sets the number of active camera algorithm "pipes" that should be simulated. 
 * 
 * "Pipes" are a PL implementation detail. Unfortunately, the number of pipes can subtley affect 
 * algorithm results. If you are attempting to <em>precisely</em> match the PL algorithm, you
 * will need to consult the PL camera module documentation for details.  If approximate results 
 * will suffice, then set this value to 1. 
 *
 * @public              @memberof kBinCSum4Alg
 * @param   alg         Algorithm object. 
 * @param   count       Active pipe count.
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4Alg_SetActivePipeCount(kBinCSum4Alg alg, kSize count);

/** 
 * Reports the number of active camera algorithm "pipes" that should be simulated. 
 *
 * @public              @memberof kBinCSum4Alg
 * @param   alg         Algorithm object. 
 * @return              Active pipe count.
 */
kFsFx(kSize) kBinCSum4Alg_ActivePipeCount(kBinCSum4Alg alg);

/** 
 * Prepares the algorithm for execution. 
 * 
 * This function should be called after changing any settings and before calling Execute 
 * for the first time on a new data set. 
 * 
 * @public              @memberof kBinCSum4Alg
 * @param   alg         kBinCSum4Alg object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4Alg_Start(kBinCSum4Alg alg); 

/** 
 * Executes the algorithm on the specified input image. 
 * 
 * This width of the image is expected to be divisible by 8. 
 *
 * @public              @memberof kBinCSum4Alg
 * @param   alg         kBinCSum4Alg object.  
 * @param   image       Input image -- kImage<k8u>. 
 * @param   bin         Receives binarize output -- kArray2<k8u>
 * @param   csum        Receives csum output -- kArray1<kCSum4>
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4Alg_Execute(kBinCSum4Alg alg, kImage image, kArray2 bin, kArray1 csum); 

#include <kFireSync/Data/kBinCSum4Alg.x.h>

#endif
