/** 
 * @file    kGraphic.h
 * @brief   Declares the kGraphic class. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_GRAPHIC_H
#define K_FIRESYNC_GRAPHIC_H

#include <kFireSync/Data/kColor.h>

/**
 * Represents a graphic brush.
 *
 * @relates kGraphic
 */
typedef kPointer kGraphicBrush;         

/**
 * Represents a graphic pen.
 *
 * @relates kGraphic
 */
typedef kPointer kGraphicPen;

/**
* Represents a graphic font.
*
* @relates kGraphic
*/
typedef kPointer kGraphicFont;

/**
* Represents a graphic marker.
*
* @relates kGraphic
*/
typedef kPointer kGraphicMarker;

/**
 * @class       kGraphic
 * @extends     kObject
 * @ingroup     kFireSync-Data
 * @brief       Represents a collection of vector graphics.
 */
//typedef kObject kGraphic;            --forward-declared in kFsDef.x.h

#include <kFireSync/Data/kGraphic.x.h>

/** 
 * Constructs a graphic object. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Destination for the constructed object handle.  
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kGraphic_Construct(kGraphic* graphic, kObject allocator);

/** 
 * Adds a solid brush that can be used for drawing operations. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   color       Brush color.   
 * @return              Returns a solid brush that can be used for drawing operations. 
 */
kFsFx(kGraphicBrush) kGraphic_AddSolidBrush(kGraphic graphic, kColor color);

/** 
 * Adds a pen that can be used for drawing operations. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   color       Pen color.   
 * @param   width       Stroke width.   
 * @return              Returns a pen that can be used for drawing operations. 
 */
kFsFx(kGraphicPen) kGraphic_AddPen(kGraphic graphic, kColor color, k32f width);

/** 
 * Adds a font that can be used for drawing operations. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   family      Font family.   
 * @param   style       Font style.   
 * @param   height      Font height.   
 * @return              Returns a font that can be used for drawing operations. 
 */
kFsFx(kGraphicFont) kGraphic_AddFont(kGraphic graphic, kFontFamily family, kFontStyle style, k32f height);

/** 
 * Adds a marker shape that can be used for drawing operations. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   shape       Marker shape.   
 * @param   size        Marker size.   
 * @return              Returns a marker shape that can be used for drawing operations. 
 */
kFsFx(kGraphicMarker) kGraphic_AddMarker(kGraphic graphic, kMarkerShape shape, k32f size);

/** 
 * Draws connected line segments.
 *
 * The internal buffer returned by this function can be modifed. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   pen         Pen to create strokes.   
 * @param   points      Array of points to be drawn (if null, will allocate a point buffer without copying points).    
 * @param   count       Count of points.    
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kPoint32s*) kGraphic_DrawPolyline32s(kGraphic graphic, kGraphicPen pen, const kPoint32s* points, kSize count);

/** 
 * Draws connected line segments.
 *
 * The internal buffer returned by this function can be modifed. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   pen         Pen to create strokes.   
 * @param   points      Array of line points (if null, will allocate a buffer without copying points).    
 * @param   count       Count of points.    
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kPoint32f*) kGraphic_DrawPolyline32f(kGraphic graphic, kGraphicPen pen, const kPoint32f* points, kSize count);

/** 
 * Draws a polygon.
 *
 * The internal buffer returned by this function can be modifed. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   pen         Pen to create strokes.   
 * @param   brush       Brush to create fill.   
 * @param   points      Array of polygon points (if null, will allocate a buffer without copying points).    
 * @param   count       Count of points.    
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kPoint32s*) kGraphic_DrawPolygon32s(kGraphic graphic, kGraphicPen pen, kGraphicBrush brush, const kPoint32s* points, kSize count);

/** 
 * Draws a polygon.
 *
 * The internal buffer returned by this function can be modifed. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   pen         Pen to create strokes.   
 * @param   brush       Brush to create fill.   
 * @param   points      Array of polygon points (if null, will allocate a buffer without copying points).    
 * @param   count       Count of points.    
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kPoint32f*) kGraphic_DrawPolygon32f(kGraphic graphic, kGraphicPen pen, kGraphicBrush brush, const kPoint32f* points, kSize count);

/** 
 * Draws a set of rectangles.
 *
 * The internal buffer returned by this function can be modifed. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   pen         Pen to create strokes.   
 * @param   brush       Brush to create fill.   
 * @param   rects       Array of rectangles (if null, will allocate a buffer without copying rectangles).    
 * @param   count       Count of rectangles.    
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kRect32s*) kGraphic_DrawRects32s(kGraphic graphic, kGraphicPen pen, kGraphicBrush brush, const kRect32s* rects, kSize count);

/** 
 * Draws a set of rectangles.
 *
 * The internal buffer returned by this function can be modifed. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   pen         Pen to create strokes.   
 * @param   brush       Brush to create fill.   
 * @param   rects       Array of rectangles (if null, will allocate a buffer without copying rectangles).    
 * @param   count       Count of rectangles.    
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kRect32f*) kGraphic_DrawRects32f(kGraphic graphic, kGraphicPen pen, kGraphicBrush brush, const kRect32f* rects, kSize count);

/** 
 * Draws a set of ellipses.
 *
 * The internal buffer returned by this function can be modifed. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   pen         Pen to create strokes.   
 * @param   brush       Brush to create fill.   
 * @param   rects       Array of bounding rectangles (if null, will allocate a buffer without copying rectangles).    
 * @param   count       Count of bounding rectangles.    
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kRect32s*) kGraphic_DrawEllipses32s(kGraphic graphic, kGraphicPen pen, kGraphicBrush brush, const kRect32s* rects, kSize count);

/** 
 * Draws a set of ellipses.
 *
 * The internal buffer returned by this function can be modifed. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   pen         Pen to create strokes.   
 * @param   brush       Brush to create fill.   
 * @param   rects       Array of bounding rectangles (if null, will allocate a buffer without copying rectangles).    
 * @param   count       Count of bounding rectangles.    
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kRect32f*) kGraphic_DrawEllipses32f(kGraphic graphic, kGraphicPen pen, kGraphicBrush brush, const kRect32f* rects, kSize count);

/** 
 * Draws a set of rotated rectangles.
 *
 * The internal buffer returned by this function can be modifed. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   pen         Pen to create strokes.   
 * @param   brush       Brush to create fill.   
 * @param   rects       Array of rectangles (if null, will allocate a buffer without copying rectangles).    
 * @param   count       Count of rectangles.    
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kRotatedRect32s*) kGraphic_DrawRotatedRects32s(kGraphic graphic, kGraphicPen pen, kGraphicBrush brush, const kRotatedRect32s* rects, kSize count);

/** 
 * Draws a set of rotated rectangles.
 *
 * The internal buffer returned by this function can be modifed. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   pen         Pen to create strokes.   
 * @param   brush       Brush to create fill.   
 * @param   rects       Array of rectangles (if null, will allocate a buffer without copying rectangles).    
 * @param   count       Count of rectangles.    
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kRotatedRect32f*) kGraphic_DrawRotatedRects32f(kGraphic graphic, kGraphicPen pen, kGraphicBrush brush, const kRotatedRect32f* rects, kSize count);

/** 
 * Draws a set of points using a pre-define shape.
 *
 * The internal buffer returned by this function can be modifed. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   pen         Pen to create strokes.   
 * @param   brush       Brush to create fill.   
 * @param   marker      Point shape.   
 * @param   points      Array of points (if null, will allocate a buffer without copying points).    
 * @param   count       Count of points.    
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kPoint32s*) kGraphic_DrawPoints32s(kGraphic graphic, kGraphicPen pen, kGraphicBrush brush, kGraphicMarker marker, const kPoint32s* points, kSize count);

/** 
 * Draws a set of points using a pre-define shape.
 *
 * The internal buffer returned by this function can be modifed. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   pen         Pen to create strokes.   
 * @param   brush       Brush to create fill.   
 * @param   marker      Point shape.   
 * @param   points      Array of points (if null, will allocate a buffer without copying points).    
 * @param   count       Count of points.    
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kPoint32f*) kGraphic_DrawPoints32f(kGraphic graphic, kGraphicPen pen, kGraphicBrush brush, kGraphicMarker marker, const kPoint32f* points, kSize count);

/** 
 * Draws a string. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   font        Font to render.  
 * @param   brush       Brush to create fill.   
 * @param   location    Origin point.   
 * @param   angle       Rotation angle.   
 * @param   alignment   Alignment of font relative to origin point.  
 * @param   text        String text.   
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kChar*) kGraphic_DrawString32s(kGraphic graphic, kGraphicFont font, kGraphicBrush brush, kPoint32s location, k32s angle, kAlignment alignment, const kChar* text);

/** 
 * Draws a string. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   font        Font to render.  
 * @param   brush       Brush to create fill.   
 * @param   location    Origin point.   
 * @param   angle       Rotation angle.   
 * @param   alignment   Alignment of font relative to origin point.  
 * @param   text        String text.   
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kChar*) kGraphic_DrawString32f(kGraphic graphic, kGraphicFont font, kGraphicBrush brush, kPoint32f location, k32f angle, kAlignment alignment, const kChar* text);

/** 
 * Draws an image. 
 *
 * The function accepts images with kArgb, kRgb, or k8u (no CFA) pixel types.
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   image       Image to draw (will be copied).  
 * @param   dest        Image destination.   
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(void*) kGraphic_DrawImage32s(kGraphic graphic, kImage image, kRect32s dest);

/** 
 * Draws an image. 
 *
 * The function accepts images with kArgb, kRgb, or k8u (no CFA) pixel types.
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   image       Image to draw (will be copied).  
 * @param   dest        Image destination.   
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(void*) kGraphic_DrawImage32f(kGraphic graphic, kImage image, kRect32f dest);

#endif
