/** 
 * @file    kMsgSet.h
 * @brief   Declares the kMsgSet type. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_MSG_SET_H
#define K_FIRESYNC_MSG_SET_H

#include <kFireSync/kNodeDef.h>
#include <kFireSync/Data/kMsgSet.x.h>

/**
 * @class   kMsgSet
 * @extends kObject
 * @ingroup kFireSync-Data
 * @brief   Abstract base class for FireSync data messages. 
 * 
 * kMsgSet defines a set of virtual methods that enable FireSync infrastructure to query a message for its stamps 
 * and data without needing to understand the data organization within each message type. The astractions provided by 
 * kMsgSet assume that derived instances could contain multiple stamps, and multiple data items per stamp.
 * 
 * While the virtual methods defined by kMsgSet offer flexibility and convenience, they are quite inefficient. These 
 * methods are intended for use only in diagnostic tools such as FireSync Studio, where performance is not a high priority.
 * In general, the type-specific interfaces provided by derived messages classes should be preferred over kMsgSet methods.
 * 
 * kMsg and kProfileMsg are the two most commonly used classes that extend kMsgSet. The kMsg class represents a 
 * message with a single stamp and single data object, often used for simple content such as video data. 
 * The kProfileMsg class is more specialized, optimized for delivery of laser profile data. Other standard message 
 * types may be added in future, and applications can also extend kMsgSet to represent custom message types (though this 
 * practice is not recommended, because developer tools such as FireSync Studio are not currently able to process 
 * custom message types). 
 * 
 * The following example illustrates how to use kMsgSet methods to access the stamps and data objects contained
 * in a message. As noted above, this approach is not efficient, and is not recommended for use outside of diagnostic 
 * applications. 
 * 
 @code {.c}
kStatus PrintMessage(kMsgInfo msgInfo)
{
    kMsgSet message = kMsgInfo_Message(msgInfo); 
    kArrayList data = kNULL; 
    kSize i, j; 

    kTry
    {    
        kLogf("Received %s with %u stamped items", kMsgSet_Count(message)); 

        for (i = 0; i < kMsgSet_Count(message); ++i)
        {
            const kStamp* stamp = kMsgSet_StampAt(message, i);

            kLogf("  Item: %u", i);
            kLogf("  Time: %llu", stamp->time);
            kLogf("  Encoder: %lld", stamp->encoder);

            //get the list of data items associated with this stamp; for a video message (kMsg), 
            //this would generate an arraylist with a single image object, where the image object was 
            //identical to the object from the original message (same object, reference count incremented); 
            //for a profile message (kProfileMsg), this would generate a pair of new kArray1 objects to 
            //represent the profile and intensity arrays; in either case, the arraylist generated by this 
            //call must be disposed (below)
            kTest(kMsgSet_DataAt(message, i, &data)); 

            kLogf("  Data Count: %u", kArrayList_Count(data)); 

            for (j = 0; j < kArrayList_Count(data); ++j)
            {
                kObject content = kArrayList_AsT(data, j, kObject); 

                kLogf("    Data Item %u: %s", j, kType_Name(kObject_Type(content))); 
            }

            //dispose the list of data items 
            kDisposeRef(&data); 
        }
    }
    kFinally
    {
        kDisposeRef(&data);

        kEndFinally(); 
    }
    
    return kOK; 
}

@endcode
 * 
 * @see     kMsg, kProfileMsg

 */
//typedef kObject kMsgSet;             --forward-declared in kFsDef.x.h

/** 
 * Gets the count of stamped items in this message set. 
 * 
 * This method must be overridden by child classes. 
 *
 * @public          @memberof kMsgSet
 * @param   set     kMsgSet object. 
 * @return          Count of items. 
 */
kInlineFx(kSize) kMsgSet_Count(kMsgSet set)
{
    return xkMsgSet_VTable(set)->VCount(set); 
}

/** 
 * Gets the stamp at the specified index.
 * 
 * This method must be overridden by child classes. 
 *
 * @public          @memberof kMsgSet
 * @param   set     kMsgSet object. 
 * @param   index   Item index. 
 * @return          Stamp. 
 */
kInlineFx(const kStamp*) kMsgSet_StampAt(kMsgSet set, kSize index)
{
    return xkMsgSet_VTable(set)->VStampAt(set, index); 
}

/** 
 * Sets the stamp at the specified index.
 * 
 * This method should be overridden by child classes that support stamp modification.
 *
 * @public          @memberof kMsgSet
 * @param   set     kMsgSet object. 
 * @param   index   Item index. 
 * @param   stamp   Stamp value.
 * @return          Stamp. 
 */
kInlineFx(kStatus) kMsgSet_SetStampAt(kMsgSet set, kSize index, const kStamp* stamp)
{
    return xkMsgSet_VTable(set)->VSetStampAt(set, index, stamp); 
}

/** 
 * Gets a list of data items at the specified stamp index.
 * 
 * The kMsgSet class assumes that child classes can have multiple data items per stamp. This function 
 * provides access to the data items corresponding to a single stamp. 
 * 
 * The kMsgSet_DataAt function is intended to be used in environments such as FireSync Studio, rather 
 * than the server/pipe processing environment. As such, the implementation of this function is free to 
 * assume that sub-objects within the kMsgSet object were all created using the same memory allocator, do 
 * not have any external attachments (e.g., via kImage_AttachT), and can therefore be emitted by 
 * using kObject_Share. 
 * 
 * In cases where the use of kObject_Share is not appropriate, new objects can be dynamically 
 * generated without concerns about performance. The memory allocator associated with the kMsgSet instance 
 * (accessible via kObject_Alloc) should be used to construct all new objects generated by this function.
 * 
 * The caller is reponsible for disposing the list returned by this function.
 * 
 * This method must be overridden by child classes. 
 * 
 * @public          @memberof kMsgSet
 * @param   set     kMsgSet object. 
 * @param   index   Item index. 
 * @param   list    Receives list of output items. 
 * @return          Operation status. 
 */
kInlineFx(kStatus) kMsgSet_DataAt(kMsgSet set, kSize index, kArrayList* list)
{
    return xkMsgSet_VTable(set)->VDataAt(set, index, list);
}

#endif
