/** 
 * @file    kSpot.h
 * @brief   Declares spot detection structures and functions.
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_SPOT_H
#define K_FIRESYNC_SPOT_H

#include <kFireSync/kFsDef.h>
#include <kFireSync/kNodeDef.h>
#include <kApi/Data/kArrayList.h>
#include <kApi/Data/kArray1.h>
#include <kApi/Data/kImage.h>
#include <kFireSync/Data/kSpot.x.h>

#define kSPOT_CENTRE_SHIFT      (8)                         // centre shift
#define kSPOT_CENTRE_SCALE      (1 << kSPOT_CENTRE_SHIFT)   // centre scale

/* 
 * kSpot
 */

typedef struct kSpot  
{
    k32u centre;             /* scaled centre of the spot (along slice) */
    k16u slice;              /* slice index */
    k16u strength;           /* spot strength metric (e.g. sum, peak) */
    k16u width;              /* spot width */
} kSpot;

/* 
 * kSpot2
 */

typedef struct kSpot2 
{
    k32u centre[2];          /* scaled centre of the spot in each dimension */
    k32u strength;           /* spot strength metric (e.g. sum, peak) */
} kSpot2;

/* 
 * kFpgaCSum
 */

typedef struct kFpgaCSum  
{
    k16u sum;                   /* sum of pixel intensities along slice */
    k16u wsum;                  /* weighted sum of pixel intensities along slice */
} kFpgaCSum;

/* 
 * kFpgaCSum3
 */

typedef struct kFpgaCSum3
{
    k32u csum3;
} kFpgaCSum3;

kInlineFx(k32u) kFpgaCSum3_Overflow(const kFpgaCSum3* cs)
{
    return cs->csum3 & kFPGA_CSUM3_OVERFLOW_MASK; 
}

kInlineFx(k32u) kFpgaCSum3_Sum(const kFpgaCSum3* cs)
{
    return (cs->csum3 & kFPGA_CSUM3_SUM_MASK) >> kFPGA_CSUM3_SUM_SHIFT; 
}

kInlineFx(k32u) kFpgaCSum3_WSum(const kFpgaCSum3* cs)
{
    return (cs->csum3 & kFPGA_CSUM3_WSUM_MASK) >> kFPGA_CSUM3_WSUM_SHIFT; 
}

/**
* @struct  kCSum4
* @extends kValue
* @ingroup kFireSync-Data
* @brief   Represents a CSum value in the format produced by the Binarize-CSum camera algorithm (version 4). 
* 
* Use the associated macros to extract individual CSum fields. 
* 
* kCSum4 supports the kdat6 serialization protocol.
*/
typedef struct kCSum4
{
    k32u csum;      ///< Column sum fields. 
} kCSum4;

/** @relates kCSum4 @{ */

/** 
 * Reports the maximum sum that can be represented by this data type.
 * 
 * @return          Maximum sum value.
 */
kInlineFx(k32u) kCSum4_MaxSum()
{
    return (kCSUM4_SUM_MASK >> kCSUM4_SUM_SHIFT) - 1; 
}

/** 
 * Reports the maximum weighted sum that can be represented by this data type.
 * 
 * @return          Maximum weighted sum value.
 */
kInlineFx(k32u) kCSum4_MaxWSum()
{
    return (kCSUM4_WSUM_MASK >> kCSUM4_WSUM_SHIFT) - 1;
}

/** 
 * Initializes a csum to a valid value.
 *
 * @param   cs          Pointer to kCSum4 value.
 * @param   sum         Sum component of the CSum value.          
 * @param   wsum        Weighted-sum component of the CSum value.          
 */
kInlineFx(void) kCSum4_Init(kCSum4* cs, k32u sum, k32u wsum)
{
    cs->csum = kField_CreateNamed_(kCSUM4_SUM, sum) | kField_CreateNamed_(kCSUM4_WSUM, wsum);
}

/** 
 * Initializes a csum to represent the 'overflow' condition.
 * 
 * @param   cs        Pointer to CSum value.
 */
kInlineFx(void) kCSum4_InitOverflow(kCSum4* cs)
{
    cs->csum = kCSUM4_OVERFLOW; 
}

/** 
 * Reports whether the csum represents an 'overflow' result. 
 * 
 * If counters overflowed, the sum and weighted-sum fields are not valid. 
 *
 * @param   cs      Pointer to CSum value.
 * @return          kTRUE if overlow occurred. 
 */
kInlineFx(kBool) kCSum4_IsOverflow(const kCSum4* cs)
{
    return cs->csum == kCSUM4_OVERFLOW; 
}

/** 
 * Sets the sum component of the CSum value. 
 *
 * @param   cs      Pointer to CSum value.
 * @param   sum     Sum component of the CSum value.          
 */
kInlineFx(void) kCSum4_SetSum(kCSum4* cs, k32u sum)
{
    kField_InsertNamed_(&cs->csum, kCSUM4_SUM, sum);
}

/** 
 * Reports the sum component of the CSum value. 
 *
 * @param   cs      Pointer to CSum value.
 * @return          Sum component of the CSum value. 
 */
kInlineFx(k32u) kCSum4_Sum(const kCSum4* cs)
{
    return kField_ExtractNamed_(cs->csum, kCSUM4_SUM);
}

/** 
 * Sets the weighted sum component of the CSum value. 
 *
 * @param   cs      Pointer to CSum value.
 * @param   wsum    Weighted-sum component of the CSum value.          
 */
kInlineFx(void) kCSum4_SetWSum(kCSum4* cs, k32u wsum)
{
    kField_InsertNamed_(&cs->csum, kCSUM4_WSUM, wsum);
}

/** 
 * Reports the weighted-sum component of the CSum value. 
 *
 * @param   cs      Pointer to CSum value.
 * @return          Weighted-sum component of the CSum value. 
 */
kInlineFx(k32u) kCSum4_WSum(const kCSum4* cs)
{
    return kField_ExtractNamed_(cs->csum, kCSUM4_WSUM);
}

/** @} */

/**
* @struct  kPhasePixel
* @extends kValue
* @ingroup kFireSync-Data
* @brief   Represents a phase result produced by PL stripe+phase hardware acceleration algorithm. 
* 
* kPhasePixel supports the kdat6 serialization protocol.
*/
typedef struct kPhasePixel
{
    k8u contrast;
    k8u intensity;
    k16s phase;
} kPhasePixel;

/**
* @struct  kPhasePixel2
* @extends kValue
* @ingroup kFireSync-Data
* @brief   Represents a phase result produced by PL stripe+phase hardware acceleration algorithm. 
* 
* kPhasePixel2 supports the kdat6 serialization protocol.
*/
typedef struct kPhasePixel2
{
    k32u content;             //packed bitfields
} kPhasePixel2;

#define kPHASE_PIXEL_2_NULL_PHASE           (0xFFFFFF)      ///< Null phase value for phase field in kPhasePixel2.

/** 
 * Initializes a kPhasePixel2 stucture.
 *
 * @public                @memberof kPhasePixel2
 * @param   phasePixel    Pointer to structure to be initialized.
 * @param   phase         Phase value. 
 * @param   intensity     Intensity value. 
 */
kInlineFx(void) kPhasePixel2_Init(kPhasePixel2* phasePixel, k32u phase, k8u intensity)
{
    kField_InsertNamed_(&phasePixel->content, kPHASE_PIXEL2_INTENSITY, intensity); 
    kField_InsertNamed_(&phasePixel->content, kPHASE_PIXEL2_PHASE, phase); 
}

/** 
 * Extracts the intensity field from a kPhasePixel2 structure.
 *
 * @public                @memberof kPhasePixel2
 * @param   phasePixel    kPhasePixel2 structure.
 * @return                Intensity value. 
 */
kInlineFx(k8u) kPhasePixel2_Intensity(kPhasePixel2 phasePixel)
{
    return (k8u) kField_ExtractNamed_(phasePixel.content, kPHASE_PIXEL2_INTENSITY); 
}

/** 
 * Sets the intensity value in a kPhasePixel2 structure.
 *
 * @public                @memberof kPhasePixel2
 * @param   phasePixel    Pointer to kPhasePixel2 structure.
 * @param   intensity     Intensity value. 
 */
kInlineFx(void) kPhasePixel2_SetIntensity(kPhasePixel2* phasePixel, k8u intensity)
{
    kField_InsertNamed_(&phasePixel->content, kPHASE_PIXEL2_INTENSITY, intensity); 
}

/** 
 * Extracts the phase field from a kPhasePixel2 structure.
 *
 * @public                @memberof kPhasePixel2
 * @param   phasePixel    kPhasePixel2 structure.
 * @return                Phase value. 
 */
kInlineFx(k32u) kPhasePixel2_Phase(kPhasePixel2 phasePixel)
{
    return kField_ExtractNamed_(phasePixel.content, kPHASE_PIXEL2_PHASE);  
}

/** 
 * Sets the phase value in a kPhasePixel2 structure.
 *
 * @public                @memberof kPhasePixel2
 * @param   phasePixel    Pointer to kPhasePixel2 structure.
 * @param   phase         Phase value. 
 */
kInlineFx(void) kPhasePixel2_SetPhase(kPhasePixel2* phasePixel, k32u phase)
{
    kField_InsertNamed_(&phasePixel->content, kPHASE_PIXEL2_PHASE, phase); 
}

/* 
 * kFpgaSpot
 */

typedef struct kFpgaSpot 
{
    k32u center_width;        /* end in 16.8 fixed point format, and width */
    k16u slice;               /* slice index */
    k16u sum_misc;            /* pixel intensity sum and misc info */
} kFpgaSpot;

kInlineFx(k32u) kFpgaSpot_Center(const kFpgaSpot* cg)
{
    return (((cg->center_width & kxFPGA_SPOT_CENTER_MASK2) >> kxFPGA_SPOT_CENTER_SHIFT2) << kxFPGA_SPOT_CENTER_SIZE1) | 
           (cg->center_width & kxFPGA_SPOT_CENTER_MASK1);
}

kInlineFx(k16u) kFpgaSpot_SpotWidth(const kFpgaSpot* cg)
{
    return (cg->center_width & kxFPGA_SPOT_SPOT_WIDTH_MASK) >> kxFPGA_SPOT_SPOT_WIDTH_SHIFT;
}

kInlineFx(k16u) kFpgaSpot_Slice(const kFpgaSpot* cg)
{
    return (((cg->sum_misc & kxFPGA_SPOT_SLICE_MASK2) >> kxFPGA_SPOT_SLICE_SHIFT2) << kxFPGA_SPOT_SLICE_SIZE1) | 
           (cg->slice & kxFPGA_SPOT_SLICE_MASK1);
}

kInlineFx(k16u) kFpgaSpot_Empty(const kFpgaSpot* cg)
{
    return cg->sum_misc & kxFPGA_SPOT_MISC_EMPTY_MASK;
}

kInlineFx(k16u) kFpgaSpot_Sum(const kFpgaSpot* cg)
{
    return (cg->sum_misc & kxFPGA_SPOT_MISC_SUM_MASK) >> kxFPGA_SPOT_MISC_SUM_SHIFT; 
}

kInlineFx(k32u) kFpgaSpot_EntryValue(const kFpgaSpot* cg)
{
    return ((cg->center_width & kxFPGA_SPOT_SOBEL_ENTRY_VALUE_MASK2) >> kxFPGA_SPOT_SOBEL_ENTRY_VALUE_SHIFT2) + 
           ((cg->slice        & kxFPGA_SPOT_SOBEL_ENTRY_VALUE_MASK1) >> kxFPGA_SPOT_SOBEL_ENTRY_VALUE_SHIFT1); 
}

/** 
 * Reports the spot entry offset. 
 * 
 * The entry offset is the approximate distance from the spot entry pixel to the spot centroid.
 * 
 * This field is optional. Refer to algorithm configuration to determine whether this field has been selected 
 * for inclusion in spot data. 
 *
 * @public          @memberof kFpgaSpot
 * @param   cg      Pointer to spot.
 * @return          Spot entry offset (1 fractional bit).
 */
kInlineFx(k32u) kFpgaSpot_EntryOffset(const kFpgaSpot* cg)
{
    k32u value = ((cg->center_width & kxFPGA_SPOT_SOBEL_ENTRY_VALUE_MASK2) >> kxFPGA_SPOT_SOBEL_ENTRY_VALUE_SHIFT2) + 
                 ((cg->slice        & kxFPGA_SPOT_SOBEL_ENTRY_VALUE_MASK1) >> kxFPGA_SPOT_SOBEL_ENTRY_VALUE_SHIFT1); 

    return value << (kSPOT_CENTRE_SHIFT - kxFPGA_WIDE_SPOT_SOBEL_ENTRY_OFFSET_FRACTIONAL_BITS);
}

/** 
 * Reports the spot exit offset. 
 * 
 * The exit offset is the approximate distance from the spot exit pixel to the spot centroid.
 * 
 * This field is optional. Refer to algorithm configuration to determine whether this field has been selected 
 * for inclusion in spot data. 
 *
 * @public          @memberof kFpgaSpot
 * @param   cg      Pointer to spot.
 * @return          Spot exit offset (1 fractional bit).
 */
kInlineFx(k32u) kFpgaSpot_ExitOffset(const kFpgaSpot* cg)
{
    k32u value = ((cg->center_width & kxFPGA_SPOT_SOBEL_ENTRY_VALUE_MASK2) >> kxFPGA_SPOT_SOBEL_ENTRY_VALUE_SHIFT2) + 
                 ((cg->slice        & kxFPGA_SPOT_SOBEL_ENTRY_VALUE_MASK1) >> kxFPGA_SPOT_SOBEL_ENTRY_VALUE_SHIFT1); 

    return value << (kSPOT_CENTRE_SHIFT - kxFPGA_WIDE_SPOT_SOBEL_EXIT_OFFSET_FRACTIONAL_BITS);
}

kInlineFx(kBool) kFpgaSpot_IsColumn(const kFpgaSpot* cg)
{
    return (cg->sum_misc & kxFPGA_SPOT_MISC_ORIENT_MASK) == 0; 
}

kInlineFx(kBool) kFpgaSpot_IsRow(const kFpgaSpot* cg)
{
    return (cg->sum_misc & kxFPGA_SPOT_MISC_ORIENT_MASK) != 0;
}

/**
* @struct  kFpgaWideSpot
* @extends kValue
* @ingroup kFireSync-Data
* @brief   Represents a laser line spot detected within an image. 
* 
* Use the associated methods to extract individual fields, rather than accessing structure fields directly.
* 
* kFpgaWideSpot supports the kdat5 and kdat6 serialization protocols.
*/

typedef struct kFpgaWideSpot 
{
    k32u center_width;        
    k16u slice;               
    k16u sum_misc;            
} kFpgaWideSpot;

/** 
 * Gets the spot centroid.
 *
 * @public          @memberof kFpgaWideSpot
 * @param   cg      Pointer to spot.
 * @return          Spot centroid (fixed point, 8 fractional bits). 
 */
kInlineFx(k32u) kFpgaWideSpot_Centre(const kFpgaWideSpot* cg)
{
    return (((cg->center_width & kxFPGA_WIDE_SPOT_CENTER_MASK2) >> kxFPGA_WIDE_SPOT_CENTER_SHIFT2) << kxFPGA_WIDE_SPOT_CENTER_SIZE1) | 
           (cg->center_width & kxFPGA_WIDE_SPOT_CENTER_MASK1);
}

/** 
 * Gets the spot width.
 *
 * @public          @memberof kFpgaWideSpot
 * @param   cg      Pointer to spot.
 * @return          Spot width, in pixels.
 */
kInlineFx(k16u) kFpgaWideSpot_Width(const kFpgaWideSpot* cg)
{
    return ((cg->center_width & kxFPGA_WIDE_SPOT_SPOT_WIDTH_MASK) >> kxFPGA_WIDE_SPOT_SPOT_WIDTH_SHIFT) << kxFPGA_WIDE_SPOT_MULTIPLIER_SHIFT;
}

/** 
 * Gets the spot slice index (column or row, depending on orientiation).
 *
 * @public          @memberof kFpgaWideSpot
 * @param   cg      Pointer to spot.
 * @return          Spot slice index, in pixels.
 */
kInlineFx(k16u) kFpgaWideSpot_Slice(const kFpgaWideSpot* cg)
{
    return (((cg->sum_misc & kxFPGA_WIDE_SPOT_SLICE_MASK2) >> kxFPGA_WIDE_SPOT_SLICE_SHIFT2) << kxFPGA_WIDE_SPOT_SLICE_SIZE1) | 
           (cg->slice & kxFPGA_WIDE_SPOT_SLICE_MASK1);
}

/** 
 * Reports whether the spot is empty (null, filler). 
 *
 * @public          @memberof kFpgaWideSpot
 * @param   cg      Pointer to spot.
 * @return          kTRUE if the spot is empty/null.
 */
kInlineFx(k16u) kFpgaWideSpot_IsEmpty(const kFpgaWideSpot* cg)
{
    return (cg->sum_misc & kxFPGA_WIDE_SPOT_MISC_EMPTY_MASK) != 0; 
}

/** 
 * Reports the spot intensity sum.
 *
 * @public          @memberof kFpgaWideSpot
 * @param   cg      Pointer to spot.
 * @return          Spot intensity sum.
 */
kInlineFx(k16u) kFpgaWideSpot_Sum(const kFpgaWideSpot* cg)
{
    return ((cg->sum_misc & kxFPGA_WIDE_SPOT_MISC_SUM_MASK) >> kxFPGA_WIDE_SPOT_MISC_SUM_SHIFT) << kxFPGA_WIDE_SPOT_MULTIPLIER_SHIFT;
}

/** 
 * Reports the spot entry intensity value. 
 * 
 * The entry value is typically the intensity of the pixel immediately prior to the spot entry pixel.   
 * 
 * This field is optional. Refer to algorithm configuration to determine whether this field has been selected 
 * for inclusion in spot data. 
 *
 * @public          @memberof kFpgaWideSpot
 * @param   cg      Pointer to spot.
 * @return          Spot entry intensity value.
 */
kInlineFx(k32u) kFpgaWideSpot_EntryValue(const kFpgaWideSpot* cg)
{
    return ((cg->center_width & kxFPGA_WIDE_SPOT_SOBEL_SPARE_MASK2) >> kxFPGA_WIDE_SPOT_SOBEL_SPARE_SHIFT2) + 
           ((cg->slice        & kxFPGA_WIDE_SPOT_SOBEL_SPARE_MASK1) >> kxFPGA_WIDE_SPOT_SOBEL_SPARE_SHIFT1); 
}

/** 
 * Reports the spot entry offset. 
 * 
 * The entry offset is the approximate distance from the spot entry pixel to the spot centroid.
 * 
 * This field is optional. Refer to algorithm configuration to determine whether this field has been selected 
 * for inclusion in spot data. 
 *
 * @public          @memberof kFpgaWideSpot
 * @param   cg      Pointer to spot.
 * @return          Spot entry offset.
 */
kInlineFx(k32u) kFpgaWideSpot_EntryOffset(const kFpgaWideSpot* cg)
{
    k32u value = ((cg->center_width & kxFPGA_WIDE_SPOT_SOBEL_SPARE_MASK2) >> kxFPGA_WIDE_SPOT_SOBEL_SPARE_SHIFT2) + 
                 ((cg->slice        & kxFPGA_WIDE_SPOT_SOBEL_SPARE_MASK1) >> kxFPGA_WIDE_SPOT_SOBEL_SPARE_SHIFT1); 

    return value << (kSPOT_CENTRE_SHIFT - kxFPGA_WIDE_SPOT_SOBEL_ENTRY_OFFSET_FRACTIONAL_BITS);
}

/** 
 * Reports the spot exit offset. 
 * 
 * The exit offset is the approximate distance from the spot exit pixel to the spot centroid.
 * 
 * This field is optional. Refer to algorithm configuration to determine whether this field has been selected 
 * for inclusion in spot data. 
 *
 * @public          @memberof kFpgaWideSpot
 * @param   cg      Pointer to spot.
 * @return          Spot exit offset.
 */
kInlineFx(k32u) kFpgaWideSpot_ExitOffset(const kFpgaWideSpot* cg)
{
    k32u value = ((cg->center_width & kxFPGA_WIDE_SPOT_SOBEL_SPARE_MASK2) >> kxFPGA_WIDE_SPOT_SOBEL_SPARE_SHIFT2) + 
                 ((cg->slice        & kxFPGA_WIDE_SPOT_SOBEL_SPARE_MASK1) >> kxFPGA_WIDE_SPOT_SOBEL_SPARE_SHIFT1); 

    return value << (kSPOT_CENTRE_SHIFT - kxFPGA_WIDE_SPOT_SOBEL_EXIT_OFFSET_FRACTIONAL_BITS);
}

//Deprecated: no longer supported
kInlineFx(kBool) kFpgaWideSpot_IsColumn(const kFpgaWideSpot* cg)
{
    return (cg->sum_misc & kxFPGA_WIDE_SPOT_MISC_ORIENT_MASK) == 0; 
}
kDeprecate(kFpgaWideSpot_IsColumn)

//Deprecated: no longer supported
kInlineFx(kBool) kFpgaWideSpot_IsRow(const kFpgaWideSpot* cg)
{
    return (cg->sum_misc & kxFPGA_WIDE_SPOT_MISC_ORIENT_MASK) != 0;
}
kDeprecate(kFpgaWideSpot_IsRow)

/*
* TODO: Sort out the naming conventions for the spot algorithms below. Some of these are type-specific, 
* others are generic (work with more than one spot type).
*/


// input expects an array list of kTypeOf(kFpgaSpot), kTYPE_FPGA_CG or kTYPE_FPGA_CG2
// output expects an initialized array list of kSpot
kFsFx(kStatus) kSpot_Unpack(kArrayList input, kArrayList output);
 
// spots expects an initialized array list of kSpot
kFsFx(kStatus) kSpot_ARCG(kImage img, k32u minSpotWidth, k32u maxSpotWidth, k32u threshold, kArrayList spots);

// FPGA ACCG implementation
kFsFx(kStatus) kFpgaSpot_ACCG(kImage image, k32u minSpotWidth, k32u maxSpotWidth, k32u threshold, kArrayList spots);

// spots expects an initialized array list of kSpot
kFsFx(kStatus) kSpot_ACCG(kImage img, k32u minSpotWidth, k32u maxSpotWidth, k32u threshold, kArrayList spots);

kFsFx(kStatus) kSpot_CSUM(kImage image, k32u threshold, kArrayList *sums);

// spots expects an initialized array list of kFpgaCSum3
kFsFx(kStatus) kSpot_CSum3(kImage image, k32u threshold, kArrayList csum3);

// spots expects an initialized array list of k64u
kFsFx(kStatus) kSpot_Binarize(kImage image, k32u threshold, kArrayList binarize);

//  kSpot_SobelArcg
//
//  Description:
//      Performs Sobel-filtered ARCG spot detection.  This is the old implementation.  It is currently being used as this version matches the FPGA version.
//  Inputs:
//      image - input kImage data to be processed, expects pixel type kTypeOf(k8s)
//      averageWindow - Sobel filter height in pixels
//      edgeWindow - Sobel filter width in pixels, expects odd number (counts center pixel)
//      threshold - Rising and falling edge threshold
//      widthThreshold - Used for width calculation, only pixels with an intensity greater than baseIntensity + widthThreshold are accepted.
//      minWidth - minimum spot width
//      maxWidth - maximum spot width
//      minSum - minimum spot sum
//  Outputs:
//      spots - returns detected spot objects, expects an initialized array of type kTYPE_SPOT, kTypeOf(kFpgaSpot), or kTypeOf(kFpgaWideSpot) 
//              with capacity equal to maximum expected number of spots
//      bounds - returns x coordinates of rising and falling edge spot boundaries, expects an initialized array of type kTYPE_POINT_32S or kNIL
//      edgeImage - returns the gradient image, expects an initialized image of pixel type kTypeOf(k8s) of size equal to the input image or kNIL

kFsFx(kStatus) kSpot_SobelArcg(kImage image, k32u averageWindow, k32u edgeWindow, k32u threshold, k32u widthThreshold, k32u minWidth,
                                      k32u maxWidth, k32u minSum, kArrayList spots, kArrayList bounds, kImage edgeImage);

//  kSpot_SobelArcg
//
//  Description:
//      Performs Sobel-filtered ARCG spot detection.  This is the new implementation of kSpot_SobelArcg.
//  Inputs:
//      image - input kImage data to be processed, expects pixel type kTypeOf(k8s)
//      averageWindow - Sobel filter height in pixels
//      edgeWindow - Sobel filter width in pixels, expects odd number (counts center pixel)
//      threshold - Rising and falling edge threshold
//      widthThreshold - Used for width calculation, only pixels with an intensity greater than baseIntensity + widthThreshold are accepted.
//      minWidth - minimum spot width
//      maxWidth - maximum spot width
//      minSum - minimum spot sum
//  Outputs:
//      spots - returns detected spot objects, expects an initialized array of type kTYPE_SPOT, kTypeOf(kFpgaSpot), or kTypeOf(kFpgaWideSpot) 
//              with capacity equal to maximum expected number of spots
//      bounds - returns x coordinates of rising and falling edge spot boundaries, expects an initialized array of type kTYPE_POINT_32S or kNIL
//      edgeImage - returns the gradient image, expects an initialized image of pixel type kTypeOf(k8s) of size equal to the input image or kNIL

kFsFx(kStatus) kSpot_SobelArcgV1(kImage image, k32u averageWindow, k32u edgeWindow, k32u threshold, k32u widthThreshold, k32u minWidth,
    k32u maxWidth, k32u minSum, kArrayList spots, kArrayList bounds, kImage edgeImage);

//  kSpot_SobelAccg
//
//  Description:
//      Performs Sobel-filtered ACCG spot detection
//  Input:
//      image - input kImage data to be processed, expects pixel type kTypeOf(k8s)
//  Output:
//      spots - returns detected spot objects, expects an initialized array of type kTYPE_SPOT, kTypeOf(kFpgaSpot), or kTypeOf(kFpgaWideSpot) 
//              with capacity equal to maximum expected number of spots
//      bounds - returns centroid coordinates of rising and falling edge spot boundaries, expects an initialized array of type kTYPE_POINT_32S or kNIL
//      edgeImage - returns the gradient image, expects an initialized image of pixel type kTypeOf(k8s) of size equal to the input image or kNIL
//  Parameters:
//      averageWindow - Sobel filter width in pixels
//      edgeWindow - Sobel filter height in pixels, expects odd number (counts center pixel)
//      threshold - Rising and falling edge threshold
//      widthThreshold - Used for width calculation, only pixels with an intensity greater than baseIntensity + widthThreshold are accepted.
//      minWidth - minimum spot width
//      maxWidth - maximum spot width
//      minSum - minimum spot sum
//      spotFormat - Choice of spot format: entry value, offset from entry to centroid, or offset from centroid to exit.
//      entryDebounceEnabled - If enabled, spot entry will be first one considered and no re-entry may occur. Otherwise, most recent spot entry will be used.
//      exitIntensityThreshold - Used for determining spot exit. Exit intensity must be less than or equal to baseIntensity + exitIntensityThreshold. Not used if set to 0.

kFsFx(kStatus) kSpot_SobelAccg(kImage image, k32u averageWindow, k32u edgeWindow, k32u threshold, k32u widthThreshold, k32u minWidth,
                                      k32u maxWidth, k32u minSum, kArrayList spots, kArrayList bounds, kImage edgeImage);

kFsFx(kStatus) kSpot_SobelAccgEx(kImage image, kArrayList spots, kArrayList bounds, kImage edgeImage, k32u averageWindow, k32u edgeWindow, k32u threshold, k32u widthThreshold, k32u minWidth,
    k32u maxWidth, k32u minSum, kAcgSpotFormat spotFormat, kBool entryDebounceEnabled, k32u exitIntensityThreshold);

kFsFx(kStatus) kSpot_YGradient(kImage input, k32u averageWindow, k32u edgeWindow, kImage output);
kFsFx(kStatus) kSpot_XGradient(kImage input, k32u averageWindow, k32u edgeWindow, kImage output);

kFsFx(k32u) kSpot_NoiseBackground(kImage image);

kFsFx(kStatus) kSpot_Sort(kArrayList spots, kArrayList sortedSpots, kSize sortedSpotCount, kCameraSpotSort sortType, kBool inverted);


// Constants used for moving average zero crossing integer math interpolation calculation
#define kSPOT_FPGA_2I_CENTRE_SHIFT              (kSPOT_CENTRE_SHIFT)  // Focalspec FPGA uses 9; we use 8 to match kSPOT
#define kSPOT_FPGA_2I_CENTRE_SCALE              (1 << kSPOT_FPGA_2I_CENTRE_SHIFT)
#define kSPOT_FPGA_2I_FRACTIONAL_SHIFT          (10)
#define kSPOT_FPGA_2I_FRACTIONAL_DISCARD_SHIFT  (kSPOT_FPGA_2I_FRACTIONAL_SHIFT - kSPOT_FPGA_2I_CENTRE_SHIFT)

#define kSPOT_FPGA_2I_SHIFT                     (4)
#define kSPOT_FPGA_2I_SCALE                     (1 << kSPOT_FPGA_2I_SHIFT)

// Constants used for moving average zero crossing filter calculation
#define kSPOT_FPGA_2I_MAX_FILTER_SIZE           (16)
#define kSPOT_FPGA_2I_FILTER_SHIFT              (kSPOT_FPGA_2I_MAX_FILTER_SIZE / 2)


/**
 * Bit-packed spot structure for FocalSpec spot algorithm
 * Range   Count  Use
 * 63..63    1    Patch
 * 62..62    1    Reserved
 * 48..61   12    Slice index (X)
 * 36..47   14    Intensity moving average
 * 24..35   12    Detection moving average
 * 22..23    2    Reserved
 * 21..0    22    Centre (Y)
 */
struct kFpgaSpot2i
{
    k64u data; /* Data member holding spot information */
};

#define kSPOT_FPGA_2I_CENTRE_LSB     ((k64u) 0)
#define kSPOT_FPGA_2I_CENTRE_MSB     ((k64u) 21)

#define kSPOT_FPGA_2I_DETECTION_LSB  ((k64u) 24)
#define kSPOT_FPGA_2I_DETECTION_MSB  ((k64u) 35)
#define kSPOT_FPGA_2I_INTENSITY_LSB  ((k64u) 36)
#define kSPOT_FPGA_2I_INTENSITY_MSB  ((k64u) 47)

#define kSPOT_FPGA_2I_SLICE_LSB      ((k64u) 48)
#define kSPOT_FPGA_2I_SLICE_MSB      ((k64u) 61)

#define kSPOT_FPGA_2I_PATCH_LSB      ((k64u) 63)
#define kSPOT_FPGA_2I_PATCH_MSB      ((k64u) 63)

// Macro to get # of bits for a given field
#define kFpgaSpot2i_BitCount_(FIELD) (kSPOT_FPGA_2I_##FIELD##_MSB - kSPOT_FPGA_2I_##FIELD##_LSB + 1)

/**
 * Set centre position in packed kFpgaSpot2i
 *
 * @param  spot    Pointer to kFpgaSpot2i
 * @param  centre  Centre position to set
 */
kInlineFx(void) kFpgaSpot2i_SetCentre(kFpgaSpot2i *spot, k64u const centre)
{
    kField_InsertNamedRange_(&spot->data, kSPOT_FPGA_2I_CENTRE, centre);
}

/**
 * Set detection filter value in packed kFpgaSpot2i
 *
 * @param  spot       Pointer to kFpgaSpot2i
 * @param  detection  Detection filter value to set
 */
kInlineFx(void) kFpgaSpot2i_SetDetection(kFpgaSpot2i *spot, k64u const detection)
{
    kField_InsertNamedRange_(&spot->data, kSPOT_FPGA_2I_DETECTION, detection);
}

/**
 * Set intensity in packed kFpgaSpot2i
 *
 * @param  spot       Pointer to kFpgaSpot2i
 * @param  intensity  Intensity value to set
 */
kInlineFx(void) kFpgaSpot2i_SetIntensity(kFpgaSpot2i *spot, k64u const intensity)
{
    kField_InsertNamedRange_(&spot->data, kSPOT_FPGA_2I_INTENSITY, intensity);
}

/**
 * Set slice index in packed kFpgaSpot2i

 * @param  spot   Pointer to kFpgaSpot2i
 * @param  slice  Slice index to set
 */
kInlineFx(void) kFpgaSpot2i_SetSlice(kFpgaSpot2i *spot, k64u const slice)
{
    kField_InsertNamedRange_(&spot->data, kSPOT_FPGA_2I_SLICE, slice);
}

/**
 * Set patch in packed kFpgaSpot2i
 *
 * @param  spot     Pointer to kFpgaSpot2i
 * @param  isPatch  Whether to set or unset patch
 */
kInlineFx(void) kFpgaSpot2i_SetPatch(kFpgaSpot2i *spot, kBool isPatch)
{
    kField_InsertNamedRange_(&spot->data, kSPOT_FPGA_2I_PATCH, (k64u) (isPatch == kTRUE));
}

/**
 * Zero the spot
 * @param  spot  Pointer to kFpgaSpot2i
 */
kInlineFx(void) kFpgaSpot2i_Zero(kFpgaSpot2i *spot)
{
    spot->data = 0;
}

/**
 * Extract spot centre from packed kFpgaSpot2i
 *
 * @param   spot  Pointer to kFpgaSpot2i
 * @returns k32u  Spot centre
 */
kInlineFx(k32u) kFpgaSpot2i_Centre(const kFpgaSpot2i *spot)
{
    return (k32u) kField_ExtractNamedRange_(spot->data, kSPOT_FPGA_2I_CENTRE);
}

/**
 * Extract Detection filter value from packed kFpgaSpot2i
 *
 * @param   spot  Pointer to kFpgaSpot2i
 * @returns k16u  Detection filter value
 */
kInlineFx(k16u) kFpgaSpot2i_Detection(kFpgaSpot2i const *spot)
{
    return (k16u)kField_ExtractNamedRange_(spot->data, kSPOT_FPGA_2I_DETECTION);
}

/**
 * Extract Intensity from packed kFpgaSpot2i
 *
 * @param   spot  Pointer to kFpgaSpot2i
 * @returns k16u  Intensity value
 */
kInlineFx(k16u) kFpgaSpot2i_Intensity(kFpgaSpot2i const *spot)
{
    return (k16u)kField_ExtractNamedRange_(spot->data, kSPOT_FPGA_2I_INTENSITY);
}

/**
 * Extract Slice index from packed kFpgaSpot2i
 *
 * @param   spot  Pointer to kFpgaSpot2i
 * @returns k16u  Slice index
 */
kInlineFx(k16u) kFpgaSpot2i_Slice(kFpgaSpot2i const *spot)
{
    return (k16u)kField_ExtractNamedRange_(spot->data, kSPOT_FPGA_2I_SLICE);
}

/**
 * Extract patch from packed kFpgaSpot2i
 *
 * @param   spot   Pointer to kFpgaSpot2i
 * @returns kBool  Whether patch is set
 */
kInlineFx(kBool) kFpgaSpot2i_Patch(kFpgaSpot2i const *spot)
{
    return (kBool)kField_ExtractNamedRange_(spot->data, kSPOT_FPGA_2I_PATCH);
}

/**
 * Populate kFpgaSpot2i(s) from kSpot(s)
 *
 * If kSpot->centre == k32U_NULL OR kSpot->slice == k16U_NULL
 * the kFpgaSpot2i patch bit is set and all fields are set to 0
 *
 * Moving average Intensity is written to kSpot->strength
 * Moving average Detection is written to kSpot->width
 * The values are copied over exactly, no scaling is performed.
 *
 * @param  outSpots  kFpgaSpot2i(s) to be populated
 * @param  inSpots   kSpot(s) from which to populate outSpots
 * @param  count     Count of spots. Default value of 1 can be used to convert single spot.
 * @returns kStatus
 */
kFsFx(kStatus) kFpgaSpot2i_ConvertFrom(kFpgaSpot2i *outSpots, kSpot const *inSpots, kSize count = 1);

/**
 * Populate kSpot(s) from kFpgaSpot2i(s)
 *
 * if kFpgaSpot2i patch bit is set, all fields in kSpot are set to *_NULL as appropriate.
 *
 * Moving average Intensity is read from kSpot->strength
 * Moving average Detection is read from kSpot->width
 * The values are copied over exactly, no scaling is performed.
 *
 * @param  inSpots   kFpgaSpot2i(s) from which to populate outSpots
 * @param  outSpots  kSpot(s) to be populated
 * @param  count     Count of spots, default value of 1 can be used to convert single spot
 * @returns kStatus
 */
kFsFx(kStatus) kFpgaSpot2i_ConvertTo(kFpgaSpot2i const *inSpots, kSpot *outSpots, kSize count = 1);

/**
 * Focalspec Moving Average + Zero Crossing spot detector reference implementation
 *
 * The region of interest parameter `roi` has the following behavior:
 * - roi = kNULL => full image
 * - roi != kNULL => roi must point to an array of size >= 4 with {rowStart, rowEnd, colStart, colEnd}
 * - If any value in (rowStart, rowEnd, colStart, colEnd) is set to kSIZE_NULL, the missing element is mapped
 *   rowStart -> 0; rowEnd -> imageHeight; colStart -> 0; colEnd -> imageWidth
 *
 * @param  image            Input image with pixel type k8u
 * @param  spots            kArrayList of type kFpgaSpot2i
 * @param  edgeWindow       Size of gradient filter
 * @param  averageWindow    Size of moving average filter used to determine weather a pixel is inside a spot
 * @param  intensityWindow  Size of moving average filter used to determine spot intensity
 * @param  threshold        Threshold against which detection-filtered image is compared to determine if a pixel is inside a spot
 * @param  edgeImage        Buffer to use for edge image, an internal buffer will be allocated and used if set to kNULL
 * @param  averageImage     Buffer to use for average image, an internal buffer will be allocated and used if set to kNULL
 * @param  intensityImage   Buffer to use for intensity image, an internal buffer will be allocated and used if set to kNULL
 * @param  roi              Region of interest
 * @returns kStatus
 */
kFsFx(kStatus) kSpot_SobelvZc(kImage image, kArrayList spots,
    kSize edgeWindow, kSize averageWindow, kSize intensityWindow, k16u threshold,
    kImage edgeImage=kNULL, kImage averageImage=kNULL, kImage intensityImage=kNULL, kSize const *roi=kNULL);

/**
 * Integer math Focalspec spot centre interpolation
 *
 * @param   lastPositiveIndex      Last index at which gradient filter value > 0
 * @param   lastPositiveGradient   Gradient value at lastPositiveIndex
 * @param   firstNegativeIndex     First index at which gradient filter value < 0
 * @param   firstNegativeGradient  Gradient value at firstNegativeIndex
 * @returns interpolatedCentre     Interpolated centre position. 11 MSBs are integer component & 9 LSBs are fractional
 */
kInlineFx(k32u) kSpot_InterpolateSobelvZcSpotCentre(
    kSize lastPositiveIndex, k16s lastPositiveGradient, kSize firstNegativeIndex, k16s firstNegativeGradient)
{
    // Convert inputs to k32u
    const k32u lastPositiveIndex32u = (k32u) lastPositiveIndex;
    const k32u firstNegativeIndex32u = (k32u) firstNegativeIndex;
    const k32u absPositiveGradient32u = (k32u) lastPositiveGradient;
    const k32u absNegativeGradient32u = (k32u) (-firstNegativeGradient);

    const k32u scaledPosition = firstNegativeIndex32u << kSPOT_FPGA_2I_CENTRE_SHIFT;

    // Note left shift by (CENTRE_SHIFT - 1) is effectively division by 2
    const k32u saturationOffset = firstNegativeIndex32u - lastPositiveIndex32u - 1;
    const k32u scaledSaturationOffset = saturationOffset << (kSPOT_FPGA_2I_CENTRE_SHIFT - 1);

    // Note that fractional value is computed with 10 bit precision and LSB is discarded
    const k32u deltaGrad = absPositiveGradient32u + absNegativeGradient32u;
    const k32u scaledNegativeGrad = absNegativeGradient32u << kSPOT_FPGA_2I_FRACTIONAL_SHIFT;
    const k32u scaledSubpixelShift = (scaledNegativeGrad / deltaGrad) >> kSPOT_FPGA_2I_FRACTIONAL_DISCARD_SHIFT;

    const k32u interpolatedCentre = scaledPosition - scaledSaturationOffset - scaledSubpixelShift;

    return interpolatedCentre;
}

#endif
