    /** 
 * @file    kDhcpClient.h
 * @brief   Declares the kDhcpClient class. 
 *
 * @internal
 * Copyright (C) 2019-2022 by LMI Technologies Inc.
  */
#ifndef K_FIRESYNC_DHCP_CLIENT_H
#define K_FIRESYNC_DHCP_CLIENT_H

#include <kApi/kApiDef.h>
#include <kApi/Io/kNetwork.h>
#include <kFireSync/kFsDef.h>

/**
 * @struct  kDhcpEvent
 * @extends kValue
 * @ingroup kFireSync
 * @brief   Represents a DHCP event. 
 */
typedef k32s kDhcpEvent; 

/** @relates kDhcpEvent @{ */
#define kDHCP_EVENT_TIMEOUT                             (0x0000)        ///< Initial negotiation timeout. 
#define kDHCP_EVENT_IP_CONFIG_ADDED                     (0x0001)        ///< Negotiation, valid IP config was provided by DHCP server. 
#define kDHCP_EVENT_IP_CONFIG_REMOVED                   (0x0002)        ///< Renew of IP config failed, previous IP config MUST discarded. 
/** @} */

typedef struct kDhcpClientEventArgs
{
    kIpAddress address;
    kIpAddress gateway;
    k32u prefixLength;

    kDhcpEvent eventType;
} kDhcpClientEventArgs;
 
/**
 * @class       kDhcpClient
 * @extends     kObject
 * @ingroup     kFireSync-Net
 * @brief       Implements the kDhcpClient format for IP config acquisition. 
 */

//typedef kObject kDhcpClient;      --forward-declared in kFsDef.x.h
  
/** 
 * Constructs a kDhcpClient object.
 * 
 * @public                      @memberof kDhcpClient
 * @param   client              Destination for the constructed object handle. 
 * @param   allocator           Memory allocator (or kNULL for default). 
 * @return                      Operation status. 
 */
kFsFx(kStatus) kDhcpClient_Construct(kDhcpClient* client, kAlloc allocator); 

/** 
 * Add a IP address change notification handler.
 * 
 * The registered callback will be invoked when an IP address is assigned to the
 * client or when a lease expires. This function is not thread-safe and therefore should not be 
 * called after calling kDhcpClient_Start.
 * 
 * When the callback is invoked, the 'receiver' argument passed to this function will be 
 * provided as the 'receiver' argument to the callback. The 'args' callback parameter contains a pointer
 * to a kDhcpClientEventArgs struct. The 'sender' parameter is not used at this time.
 *
 * @public              @memberof kDhcpClient
 * @param   client      kDhcpClient object.
 * @param   function    Callback function. 
 * @param   receiver    Callback receiver.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kDhcpClient_AddEventHandler(kDhcpClient client, kCallbackFx function, kPointer receiver); 

/** 
 * Starts the DHCP negotiation process.
 * 
 * @public                      @memberof kDhcpClient
 * @param   client              kDhcpClient object.
 * @param   adapterName         Name of the adapter to send and receive, e.g. 'eth0'. 
 * @param   macAddress          MAC address to announce to server. 
 * @return                      Operation status. 
 */
kFsFx(kStatus) kDhcpClient_Start(kDhcpClient client, const kChar* adapterName, kMacAddress macAddress);

#include <kFireSync/Net/kDhcpClient.x.h>

#endif
