/** 
 * @file    kDhcpServer.x.h
 *
 * @internal
 * Copyright (C) 2019-2022 by LMI Technologies Inc.
 */
#ifndef K_FIRESYNC_DHCP_SERVER_X_H
#define K_FIRESYNC_DHCP_SERVER_X_H

#include <kFireSync/Net/kDhcpClient.h>
#include <kFireSync/Net/kPacketUtils.h>

/* Timeouts */
#define xkDHCP_SERVER_RECV_TIMEOUT                      (500000)        // Maximal timeout for receiving messages.
#define xkDHCP_SERVER_MAX_PROCESS_TIMEOUT               (5000000)       // Maximal time for processing DHCP messages.

#define xkDHCP_SERVER_LOG                               (kFALSE)        // Set to kTRUE to enable logging.

#define xkDHCP_SERVER_INFINITE                          (0xFFFFFFFFu)   // Infinite lease time.

#define xkDHCP_SERVER_HEARTBEAT                         (500000)        // Heartbeat, in microseconds.

typedef struct kDhcpServerClass
{   
    kObjectClass base; 

    kIpAddress serverIp;            // Server IP address.
    kIpAddress poolStart;           // Pool start address.
    k32u poolSize;                  // Pool size.
    k32u prefix;                    // Prefix length to the client.
    kIpAddress gateway;             // gateway provided to the client.
    kMacAddress macAddress;         // Server MAC address.

    kThread recvThread;             // Receive thread.
    kAtomic32s shouldQuit;          // Thread quit flag;
    kPeriodic heartBeat;            // DHCP heartbeat handler.
    kPacketSocket sockt;            // Socket used for sending/receiving raw packets.
    kMsgQueue packetQueue;          // kMsgQueue<kArray1<kByte>>.
    kArray1 frame;                  // kArray1<kByte>, used to temporary store ethernet frames.

    kMap pool;                      // DHCP IP address pool, kMap<kMacAddress, kIpAddress>.
    
} kDhcpServerClass;

kDeclareClassEx(kFs, kDhcpServer, kObject)

/* 
* Private methods. 
*/

kFsFx(kStatus) xkDhcpServer_Init(kDhcpServer server, kType type, kAlloc alloc); 
kFsFx(kStatus) xkDhcpServer_VRelease(kDhcpServer server);

kFsFx(kStatus) xkDhcpServer_HeartBeat(kDhcpServer server, kPeriodic timer);

kFsFx(kStatus) xkDhcpServer_ProcessMessages(kDhcpServer server);

kFsFx(kStatus) xkDhcpServer_RecvThread(kDhcpServer server);

kFsFx(kStatus) xkDhcpServer_QueuePacket(kDhcpServer server, const kByte* buffer, kSize bufferSize);

kFsFx(kStatus) xkDhcpServer_SendDhcpOffer(kDhcpServer server, kIpAddress clientIp, kMacAddress macAddress, k32u transactionId);
kFsFx(kStatus) xkDhcpServer_SendDhcpAck(kDhcpServer server, kIpAddress clientIp, kMacAddress macAddress, k32u transactionId);

kFsFx(kBool) xkDhcpServer_IsDhcpMessage(kDhcpServer server, kByte* frame, kSize size);

kFsFx(kStatus) xkDhcpServer_ProcessDiscovers(kDhcpServer server, kDhcpHeader header);
kFsFx(kStatus) xkDhcpServer_ProcessRequests(kDhcpServer server, kDhcpHeader header);
kFsFx(kStatus) xkDhcpServer_ProcessRelease(kDhcpServer server, kDhcpHeader header);

kFsFx(kStatus) xkDhcpServer_SendFrame(kDhcpServer server, kByte* frame, kSize size, kIpAddress sourceIp, kIpAddress destIp, kMacAddress macAddress, kMacAddress destMac, kSize dataLength);

k16u xkDhcpClient_Time(k64u t);

#endif
