/** 
 * @file    kPacketUtils.h
 * @brief   Essential packet utilities implementation declarations. 
 *
 * @internal
 * Copyright (C) 2019-2022 by LMI Technologies Inc. All rights reserved.
 */
#ifndef K_PACKET_UTILS_H
#define K_PACKET_UTILS_H

#include <kFireSync/kFsDef.h>

#define kPACKET_UTILS_ETHERNET_HEADER_SIZE     (14)             //< Ethernet header size.
#define kPACKET_UTILS_ARP_HEADER_SIZE          (28)             //< ARP header size.
#define kPACKET_UTILS_IP_HEADER_SIZE           (20)             //< IP header size.
#define kPACKET_UTILS_UDP_HEADER_SIZE          (8)              //< UDP header size.
#define kPACKET_UTILS_DHCP_HEADER_SIZE         (240)            //< DHCP header size.

#define kPACKET_UTILS_DHCP_CHADDR_LEN          (16)             //< Length of DHCP hardware address.
#define kPACKET_UTILS_DHCP_SERVERNAME_LEN      (64)             //< Length of DHCP server name.
#define kPACKET_UTILS_DHCP_FILENAME_LEN        (128)            //< Length of DHCP boot file name.

#define kPACKET_UTILS_ETHERNET_PROTO_IP        (0x0800)         //< IPv4 ethernet protocol type. 
#define kPACKET_UTILS_ETHERNET_PROTO_ARP       (0x806)          //< ARP ethernet protocol type.

#define kPACKET_UTILS_ETHERNET_IP_PROTO_UDP    (17)             //< UDP IPv4 protocol type.

//< Minimal DHCP packet size (without any DHCP options). 
#define kPACKET_UTILS_DHCP_MIN_FRAME_SIZE      (kPACKET_UTILS_ETHERNET_HEADER_SIZE + kPACKET_UTILS_IP_HEADER_SIZE + kPACKET_UTILS_UDP_HEADER_SIZE + kPACKET_UTILS_DHCP_HEADER_SIZE)

//< Gives the position of the DHCP options in an ethernet frame. 
#define kPACKET_UTILS_DHCP_OPTIONS_POS(b)      ((kByte*)b + kPACKET_UTILS_DHCP_MIN_FRAME_SIZE)

/**
 * @struct  kEthernetHeader
 * @extends kValue
 * @ingroup kFireSync-Net
 * @brief   Represents an Ethernet frame header.
 */
typedef struct kEthernetHeader
{
    kMacAddress destAddress;        //< Destination MAC address
    kMacAddress sourceAddress;      //< Source MAC address
    k16u protocol;                  //< Protocol type, e.g. 0x0800 for IPv4.
} kEthernetHeader;

/**
 * @struct  kArpHeader
 * @extends kValue
 * @ingroup kFireSync-Net
 * @brief   Represents an Arp message.
 */
typedef struct kArpHeader
{
    k16u hardwareType;              //< Hardware type, e.g. 1 for Ethernet (10Mb).
    k16u protocolType;              //< Protocol type.
    k8u hardwareSize;               //< Size of hardware address.
    k8u protocolSize;               //< Size of protocol address.
    k16u operationCode;             //< Operation code.
    kMacAddress sourceMac;          //< Source MAC address.
    kIpAddress sourceIp;            //< Source IP address.
    kMacAddress destMac;            //< Destination MAC address.
    kIpAddress destIp;              //< Destination IP address.
} kArpHeader;

/**
 * @struct  kIpHeader
 * @extends kValue
 * @ingroup kFireSync-Net
 * @brief   Represents an IPv4 frame header.
 */
typedef struct kIpHeader
{
    k8u ihl4Version;                //< Version and internet header length.
    k8u tos;                        //< Type of service.
    k16u totalLength;               //< Total length in bytes of header and data.
    k16u id;                        //< Identification.
    k16u fragment;                  //< Flags.
    k8u ttl;                        //< Time To Live.
    k8u protocol;                   //< Protocol.
    k16u checksum;                  //< Header checksum.
    kIpAddress sourceIp;            //< Source IP address.
    kIpAddress destIp;              //< Destination IP address.
} kIpHeader;

/**
 * @struct  kUdpHeader
 * @extends kValue
 * @ingroup kFireSync-Net
 * @brief   Represents an UDP frame header.
 */
typedef struct kUdpHeader
{
    k16u sourcePort;                //< Source port.
    k16u destinationPort;           //< Destination port.
    k16u length;                    //< Total length in bytes of header and data.
    k16u checksum;                  //< Checksum.
} kUdpHeader;

/**
 * @struct  kDhcpHeader
 * @extends kValue
 * @ingroup kFireSync-Net
 * @brief   Represents an DHCP message header.
 */
typedef struct kDhcpHeader
{
    k8u opcode;                                             //< Operation code.
    k8u htype;                                              //< Hardware type.
    k8u hlen;                                               //< Hardware address length.
    k8u hops;                                               //< Number of relays agents that have forwarded this message.
    k32u xid;                                               //< Transaction identifier.
    k16u secs;                                              //< Elapsed time in seconds since client began negotiation process.
    k16u flags;                                             //< Broadcast flag.
    kIpAddress clientIp;                                    //< Client's IP address; set by the client.
    kIpAddress yourIp;                                      //< Client's IP address; set by the server to inform client.
    kIpAddress serverIp;                                    //< Server address.
    kIpAddress relayIp;                                     //< Relay agent IP address.
    kMacAddress clientMac;                                  //< Client's hardware address.
    kChar serverName[kPACKET_UTILS_DHCP_SERVERNAME_LEN];    //< Name of the server.
    kChar fileName[kPACKET_UTILS_DHCP_FILENAME_LEN];        //< Name of the boot file name.
    k32u magicCookie;                                       //< DHCP magic cookie.
} kDhcpHeader;

/** 
 * Reads an Ethernet header from an Ethernet frame.
 *
 * @public                      @memberof kPacketUtils
 * @param   frame               Full Ethernet frame.
 * @param   size                Size of the frame, in bytes.
 * @param   header              Receives the Ethernet content of the frame.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kPacketUtils_ReadEthernetHeader(const kByte* frame, kSize size, kEthernetHeader* header);

/** 
 * Writes an Ethernet header to an Ethernet frame.
 *
 * @public                      @memberof kPacketUtils
 * @param   frame               Full Ethernet frame.
 * @param   size                Size of the frame, in bytes.
 * @param   header              Ethernet header.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kPacketUtils_WriteEthernetHeader(kByte* frame, kSize size, const kEthernetHeader* header);

/** 
 * Reads an ARP header from an Ethernet frame.
 *
 * @public                      @memberof kPacketUtils
 * @param   frame               Full Ethernet frame.
 * @param   size                Size of the frame, in bytes.
 * @param   header              Receives the ARP content of the frame.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kPacketUtils_ReadArpHeader(const kByte* frame, kSize size, kArpHeader* header);

/** 
 * Writes an ARP header to an Ethernet frame.
 *
 * @public                      @memberof kPacketUtils
 * @param   frame               Full Ethernet frame.
 * @param   size                Size of the frame, in bytes.
 * @param   header              ARP header.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kPacketUtils_WriteArpHeader(kByte* frame, kSize size, const kArpHeader* header);

/** 
 * Reads an Ipv4 header from an Ethernet frame.
 *
 * @public                      @memberof kPacketUtils
 * @param   frame               Full Ethernet frame.
 * @param   size                Size of the frame, in bytes.
 * @param   header              Receives the IPv4 content of the frame.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kPacketUtils_ReadIpHeader(const kByte* frame, kSize size, kIpHeader* header);

/** 
 * Writes an Ipv4 header to an Ethernet frame. 
 *
 * The version, tos, id, fragment, ttl and checksum fields of the kIpHeader struct are set 
 * by this function so any previous values are overwritten.
 *
 * @public                      @memberof kPacketUtils
 * @param   frame               Full Ethernet frame.
 * @param   size                Size of the frame, in bytes.
 * @param   header              Ipv4 header.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kPacketUtils_WriteIpHeader(kByte* frame, kSize size, const kIpHeader* header);

/** 
 * Reads an UDP header from an Ethernet frame.
 *
 * @public                      @memberof kPacketUtils
 * @param   frame               Full Ethernet frame.
 * @param   size                Size of the frame, in bytes.
 * @param   header              Receives the UDP content of the frame.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kPacketUtils_ReadUdpHeader(const kByte* frame, kSize size, kUdpHeader* header);

/** 
 * Writes an UDP header to an Ethernet frame.
 *
 * @public                      @memberof kPacketUtils
 * @param   frame               Full Ethernet frame.
 * @param   size                Size of the frame, in bytes.
 * @param   header              UDP header.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kPacketUtils_WriteUdpHeader(kByte* frame, kSize size, const kUdpHeader* header);

/** 
 * Reads a DHCP header from an Ethernet frame.
 *
 * @public                      @memberof kPacketUtils
 * @param   frame               Full Ethernet frame.
 * @param   size                Size of the frame, in bytes.
 * @param   header              Receives the DHCP content of the frame.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kPacketUtils_ReadDhcpHeader(const kByte* frame, kSize size, kDhcpHeader* header);

/** 
 * Writes an DHCP header to an Ethernet frame.
 *
 * @public                      @memberof kPacketUtils
 * @param   frame               Full Ethernet frame.
 * @param   size                Size of the frame, in bytes.
 * @param   header              DHCP header.
 * @return                      Operation status. 
 */
kFsFx(kStatus) kPacketUtils_WriteDhcpHeader(kByte* frame, kSize size, const kDhcpHeader* header);

#include <kFireSync/Net/kPacketUtils.x.h>

#endif
