/** 
 * @file    kSvPipe.x.h
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYC_SERVER_PIPE_X_H
#define K_FIRESYC_SERVER_PIPE_X_H

#define kSV_PIPE_BLOCK_CONFIG_VERSION            (1)            //Block configuration format version.

#define kSV_PIPE_DEFAULT_MAX_QUEUE_SIZE          (1 << 23)      //Default maximum pipe queue size (bytes).
#define kSV_PIPE_DEFAULT_MAX_QUEUE_COUNT         (1 << 12)      //Default maximum pipe item count.

#define kSV_PIPE_ALLOC_BLOCK_SIZE                (1 << 16)      //Size of blocks used for small memory requests.
#define kSV_PIPE_ALLOC_MAX_BLOCK_BUFFER_SIZE     (1 << 14)      //Maximum memory request treated as a small memory request.

#define kSV_PIPE_ALLOC_RESERVATION_1            (1)      //Memory reservation for 1-byte requests (bytes).
#define kSV_PIPE_ALLOC_RESERVATION_2            (1)      //Memory reservation for 2-byte requests (bytes).
#define kSV_PIPE_ALLOC_RESERVATION_4            (1)      //Memory reservation for 4-byte requests (bytes).
#define kSV_PIPE_ALLOC_RESERVATION_8            (1)      //Memory reservation for 8-byte requests (bytes).
#define kSV_PIPE_ALLOC_RESERVATION_16           (1)      //Memory reservation for 16-byte requests (bytes).
#define kSV_PIPE_ALLOC_RESERVATION_32           (262144) //Memory reservation for 32-byte requests (bytes).
#define kSV_PIPE_ALLOC_RESERVATION_64           (524288) //Memory reservation for 64-byte requests (bytes).
#define kSV_PIPE_ALLOC_RESERVATION_128          (262144) //Memory reservation for 128-byte requests (bytes).
#define kSV_PIPE_ALLOC_RESERVATION_256          (65536)  //Memory reservation for 256-byte requests (bytes).
#define kSV_PIPE_ALLOC_RESERVATION_512          (65536)  //Memory reservation for 512-byte requests (bytes).
#define kSV_PIPE_ALLOC_RESERVATION_1024         (65536)  //Memory reservation for 1024-byte requests (bytes).
#define kSV_PIPE_ALLOC_RESERVATION_2048         (65536)  //Memory reservation for 2048-byte requests (bytes).
#define kSV_PIPE_ALLOC_RESERVATION_4096         (65536)  //Memory reservation for 4096-byte requests (bytes).
#define kSV_PIPE_ALLOC_RESERVATION_8192         (65536)  //Memory reservation for 8192-byte requests (bytes).
#define kSV_PIPE_ALLOC_RESERVATION_16384        (65536)  //Memory reservation for 16384-byte requests (bytes).

#define kSV_PIPE_DEFAULT_TIMER_PERIOD_US        (1000u)        //Default timer block callback period, in microseconds

typedef struct kSvPipeClass
{
    kObjectClass base; 
    
    kSvNode node;                               //Server node (parent). 

    kPipeEngine engineType;                     //Pipe engine type. 
    kSize threadCount;                          //Desired thread count.
    kThreadPriorityClass threadPriorityClass;   //Desired thread priority class (low, med, high).
    k32s threadPriorityOffset;                  //Desired thread priority offset.
    kBitArray threadAffinity;                   //Desired thread affinity.
    kSize maxQueueSize;                         //Pipe message input queue maximum size setting (bytes). 
    kSize maxQueueCount;                        //Pipe message input queue maximum count setting. 

    kArrayList assemblies;                      //List of assemblies searched for block types. 
    kArrayList userBlockTypes;                  //List of available user-defined block types -- kArrayList<kType>.

    kArrayList routes;                          //List of routes -- kArrayList<kRouteEntry>.
    kArrayList systemBlocks;                    //List of system blocks -- kArrayList<kPxBlock>.
    kArrayList userBlocks;                      //List of user blocks -- kArrayList<kPxBlock>.

    kPxPipe pipe;                               //Pipe execution engine. 

    kPoolAlloc poolMsgAlloc;                    //Innermost message allocator. 
    kDebugAlloc debugMsgAlloc;                  //Optional message allocator layer (leak tracking).
    kUserAlloc profileMsgAlloc;                 //Optional message allocator layer (code profiling). 
    kAlloc msgAlloc;                            //Topmost message allocator layer. 

    kProfileProbe msgAllocGetProfileProbe; 
    kProfileProbe msgAllocFreeProfileProbe; 

    kPxIoBlock timerBlock;                      //Timer block.
    kBool timerEnabled;                         //Is the timer block enabled.
    k64u timerPeriod;                           //Timer block callback period.
    kPeriodic timerPeriodic;                    //Timer block periodic callback timer.
    k64u timerFrameCount;                       //Number of frames in acquisition start.

} kSvPipeClass; 

kDeclareClassEx(kFs, kSvPipe, kObject)
        
kFsFx(kStatus) kSvPipe_Init(kSvPipe pipe, kType type, kSvNode node, kAlloc alloc);
kFsFx(kStatus) kSvPipe_VRelease(kSvPipe pipe);

kFsFx(kStatus) kSvPipe_MsgAllocGet(kSvPipe pipe, kSize size, void* mem, kMemoryAlignment alignment);
kFsFx(kStatus) kSvPipe_MsgAllocFree(kSvPipe pipe, void* mem);

kFsFx(kStatus) kSvPipe_FinalizeDebugAlloc(kSvPipe pipe); 

kFsFx(kStatus) kSvPipe_FormatBlockTypes(kSvPipe pipe, kXml xml, kXmlItem item); 
kFsFx(kStatus) kSvPipe_FormatBlocks(kSvPipe pipe, kXml xml, kXmlItem item, const kChar* itemName, kArrayList blockList); 
kFsFx(kStatus) kSvPipe_FormatRoutes(kSvPipe pipe, kXml xml, kXmlItem item); 

kFsFx(kStatus) kSvPipe_ParseUserBlocks(kSvPipe pipe, kXml xml, kXmlItem item); 
kFsFx(kStatus) kSvPipe_ParseRoutes(kSvPipe pipe, kXml xml, kXmlItem item); 

kFsFx(kStatus) kSvPipe_EnumerateBlockTypes(kSvPipe pipe);

kFsFx(kStatus) kSvPipe_FindBlockType(kSvPipe pipe, const kChar* typeName, kType* type); 

kFsFx(kStatus) kSvPipe_OnHealthUpdate(kSvPipe pipe, kObject sender, kPointer unused);

kFsFx(kStatus) kSvPipe_LogCudaStatus(kSvPipe pipe);

//kPxEnviron interface
kFsFx(kStatus) kSvPipe_FindVariable(kSvPipe pipe, const kChar* name, kObject* object);
kFsFx(kStatus) kSvPipe_FilePath(kSvPipe pipe, const kChar* fileName, kChar* filePath, kSize capacity);
kFsFx(kStatus) kSvPipe_Printvf(kSvPipe pipe, const kChar* format, va_list args);
kFsFx(kAlloc) kSvPipe_MessageAlloc(kSvPipe pipe);
kFsFx(kHealth) kSvPipe_Health(kSvPipe pipe);
kFsFx(k64u) kSvPipe_Timestamp(kSvPipe pipe); 
kFsFx(kThreadPriorityClass) kSvPipe_ThreadPriorityClass(kSvPipe pipe);
kFsFx(k32s) kSvPipe_ThreadPriorityOffset(kSvPipe pipe);
kFsFx(kBitArray) kSvPipe_ThreadAffinity(kSvPipe pipe);

kFsFx(kStatus) xkSvPipe_SetupTimerBlock(kSvPipe pipe);
kFsFx(kStatus) xkSvPipe_TimerCallback(kSvPipe pipe, kPeriodic timer);

#endif
