/** 
 * @file    kSvDataIn.x.h
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYC_SERVER_DATA_IN_X_H
#define K_FIRESYC_SERVER_DATA_IN_X_H

#if defined(K_TI_BIOS)
#   define kSV_DATA_IN_DEFAULT_CLIENT_READ_BUFFER          (32*1024)             
#   define kSV_DATA_IN_DEFAULT_SOCKET_READ_BUFFER          (32*1024)      
#elif defined(K_VX_KERNEL)
#   define kSV_DATA_IN_DEFAULT_CLIENT_READ_BUFFER          (128*1024)             
#   define kSV_DATA_IN_DEFAULT_SOCKET_READ_BUFFER          (128*1024)      
#else
#   define kSV_DATA_IN_DEFAULT_CLIENT_READ_BUFFER          (256*1024)             
#   define kSV_DATA_IN_DEFAULT_SOCKET_READ_BUFFER          (256*1024)      
#endif

#define kSV_DATA_IN_TPUT_AVERAGE_WINDOW                 (500000)        ///< Time averaging window for throughput calculations (us).
#define kSV_DATA_IN_TPUT_AVERAGE_MIN_SAMPLE             (100000)        ///< Minimum time between throughput samples (us).


typedef struct kSvDataInClass
{
    kObjectClass base; 
    kSvNode node;                       //Parent/owner.
    kAlloc messageAlloc;                //Message allocator. 

    kSize requiresConnectionRefresh;    //Has data channel configuration changed since last open?

    kThreadPriorityClass threadPriorityClass;   //Desired thread priority class (low, med, high) for network streaming threads.
    k32s threadPriorityOffset;                  //Desired thread priority offset for network streaming threads.
    kBitArray threadAffinity;                   //Desired thread affinity for network streaming threads.

    kSize socketBufferSize;             //Per-channel socket read buffer size.
    kArrayList ports;                   //List of configured input ports -- kArrayList<kDataInSource>.

    kArrayList connections;             //List of connections -- kArrayList<kSvDataInChannel>.

    kPxIoBlock block;                   //Output block to send data to pipe. 

    kSvDataInStats stats;               //Run-time statistics. 
    kLock statsLock;                    //Provides exclusive access to run-time statistics. 
    k64u channelBytesRead;              //Cumulative bytes read, as reported by channels. 

    kTimeAvg throughputAvg;             //Moving average stats for throughput. 

    kHealthProbe bytesReadProbe;        //Health probe for total bytes read by all channels.  
    kHealthProbe throughputProbe;       //Health probe for total throughput (bytes/sec). 
} kSvDataInClass; 

kDeclareClassEx(kFs, kSvDataIn, kObject)
        
kFsFx(kStatus) kSvDataIn_Init(kSvDataIn service, kType type, kSvNode node, kAlloc allocator);
kFsFx(kStatus) kSvDataIn_VRelease(kSvDataIn service);

kFsFx(kStatus) kSvDataIn_Parse(kSvDataIn service, kXml xml, kXmlItem item); 
kFsFx(kStatus) kSvDataIn_Format(kSvDataIn service, kXml xml, kXmlItem item); 

kFsFx(kStatus) kSvDataIn_SetSocketBufferSize(kSvDataIn service, kSize size);
kFsFx(kSize) kSvDataIn_SocketBufferSize(kSvDataIn service);

kFsFx(kStatus) kSvDataIn_SetPorts(kSvDataIn service, kArrayList ports);
kFsFx(kBool) kSvDataIn_PortsListsAreEqual(kSvDataIn service, kArrayList a, kArrayList b); 

kFsFx(kBool) kSvDataIn_HasEndPointChanges(kSvDataIn service); 
kFsFx(kStatus) kSvDataIn_CreateChannels(kSvDataIn service); 
kFsFx(kStatus) kSvDataIn_ConnectChannels(kSvDataIn service); 
kFsFx(kStatus) kSvDataIn_Connect(kSvDataIn service);
kFsFx(kStatus) kSvDataIn_Disconnect(kSvDataIn service);

kFsFx(kStatus) kSvDataIn_FindChannelByNode(kSvDataIn service, k32u sourceNode, kSvDataInChannel* channel); 

kFsFx(kStatus) kSvDataIn_UpdateBlock(kSvDataIn service); 

kFsFx(kStatus) kSvDataIn_AddChannelStats(kSvDataIn service, k64u newBytesRead); 

kFsFx(kStatus) kSvDataIn_OnHealthUpdate(kSvDataIn service, kObject sender, kPointer unused);

kInlineFx(kSvNode) kSvDataIn_Node(kSvDataIn service)
{
    kObj(kSvDataIn, service); 
    
    return obj->node; 
}

#endif
