#ifndef k_LD_CAL_LOOKUP_H
#define k_LD_CAL_LOOKUP_H

#include <kFireSync/kFsDef.h>
#include <kFireSync/kNodeDef.h>
#include <kFireSync/Utils/kLdCal.h>

/**
* @class       kLdCalLookup
* @extends     kObject
* @ingroup     kFireSync-Utils
* @brief       Class object for the kLdCalLookup class.
*/
//typedef kObject kLdCalLookup;            --forward-declared in kFsDef.x.h  

/**
* Constructs a kLdCalLookup object.
*
* @public               @memberof kLdCalLookup
* @param   lookup       Destination for the constructed object handle.
* @param   cal          Handle to the laser driver calibration object. Required only for construction and does not 
*                       need to be retained during the lifetime of this class
* @param   alloc        Memory allocator (or kNULL for default).
* @return               Operation status.
*/
kFsFx(kStatus) kLdCalLookup_Construct(kLdCalLookup* lookup, kLdCal cal, kAlloc alloc);

/**
* Configures temperature compensation properties for the power set point. 
* 
* Any driver which incorporates a photo-diode, does not need software temperature compensation and should be left 
* unconfigured or have the input coefficients configured to zero
*
* @public                   @memberof kLdCalLookup
* @param   lookup           Lookup object.
* @param   tempDeltaCoeff   Coefficient specifying power change as a function of temperature difference relative to that of calibration (mW/C)
*                           For example, a coefficient of 1.0, would yield increase of 1 mW for each degree relative to the
*                           calibration temperature. 
*                           By default, this value is set to zero, effectively disabling any temperature compensation 
* @return                   Operation status.
*/
kFsFx(kStatus) kLdCalLookup_SetPowerCompensationInfo(kLdCalLookup lookup, k64f tempDeltaCoeff);

/**
* Queries temperature compensation properties for the power set point.
*
* @public                   @memberof kLdCalLookup
* @param   lookup           Lookup object.
* @param   tempDeltaCoeff   Coefficient specifying power change as a function of temperature difference relative to that of calibration (mW/C).
* @return                   Operation status.
*/
kFsFx(kStatus) kLdCalLookup_PowerCompensationInfo(kLdCalLookup lookup, k64f* tempDeltaCoeff);

/**
* Configures temperature compensation properties for the current limit.
*
* @public                   @memberof kLdCalLookup
* @param   lookup           Lookup object.
* @param   tempDeltaCoeff   Coefficient specifying power change as a function of temperature difference relative to that of calibration (mW/C)
*                           For example, a coefficient of 1.0, would yield increase of 1 mW for each degree relative to the
*                           calibration temperature.
*                           By default, this value is set to zero, effectively disabling any temperature compensation.
* @return                   Operation status.
*/
kFsFx(kStatus) kLdCalLookup_SetLimitCompensationInfo(kLdCalLookup lookup, k64f tempDeltaCoeff);

/**
* Queries temperature compensation properties for the current limit.
*
* @public                   @memberof kLdCalLookup
* @param   lookup           Lookup object.
* @param   tempDeltaCoeff   Coefficient specifying power change as a function of temperature difference relative to that of calibration (mW/C).
* @return                   Operation status.
*/
kFsFx(kStatus) kLdCalLookup_LimitCompensationInfo(kLdCalLookup lookup, k64f* tempDeltaCoeff);

/**
* Queries whether limit power has a calibration record and is thus configurable.
*
* @public                   @memberof kLdCalLookup
* @param   lookup           Lookup object.
* @return                   Flag indicating whether limit calibration is present.
*/
kFsFx(kBool) kLdCalLookup_HasLimit(kLdCalLookup lookup);

/**
* Reports the minimum valid intensity value supported by the calibration. 
* 
* @public                   @memberof kLdCalLookup
* @param   lookup           Lookup object.
* @return                   Minimum intensity value (normalized).
*/
kFsFx(k64f) kLdCalLookup_MinIntensity(kLdCalLookup lookup);

/**
* Calculates optical power at a specific set point input value and driver temperature. 
* 
* This function can be used to determine the optical power at a previously configured set point input value. Specified input must be 
* within the calibrated range and inputs outside of that range will result in function returning kERROR_NOT_FOUND
*
* @public                   @memberof kLdCalLookup
* @param   lookup           Lookup object.
* @param   input            Input to the power pot, specified in device units. 
* @param   temperature      Driver temperature at which the power is queried. Specify k64F_NULL to default to the calibration temperature.
* @param   power            Pointer receiving the optical power corresponding to the specified set point input and temperature.
* @return                   Operation status.
*/
kFsFx(kStatus) kLdCalLookup_InputToPower(kLdCalLookup lookup, kSize input, k64f temperature, k64f* power);

/**
* Calculates the set point input expected to produce specified optical power at the specified temperature. 
* 
* Resulting output selects the _nearest_ input, even if it results in optical power that is higher than requested. 
*
* All specified powers _must_ already account for margins due to uncertainty in digital pot resolution, accuracy of the
* calibration methodology or efficiency of the optical path.
*
* For example, with resolution of the pot at 10% of the input, the returned power may fall within +/- 10% of the specified _before_
* accounting for calibration error, temperature compensation uncertainty, optical path efficiency uncertainty
* 
* While granularity of the digital pot is only one component of the uncertainty, the optional 'expectedPower' output parameter 
* may be passed in to determine the exact calibrated optical power corresponding to the specified input
*
* @public                   @memberof kLdCalLookup
* @param   lookup           Lookup object.
* @param   power            Requested optical power.
* @param   temperature      Driver temperature at which the power is queried. Specify k64F_NULL to default to the calibration temperature.
* @param   input            Driver input, in device units, closest to the requested power at specified temperature.
* @param   expectedPower    Optional output parameter receiving the exact calibrated optical power corresponding to the specified input.
* @return                   Operation status.
*/
kFsFx(kStatus) kLdCalLookup_PowerToInput(kLdCalLookup lookup, k64f power, k64f temperature, kSize* input, k64f* expectedPower);

/**
* Calculates optical power at a specific current limit input value and driver temperature. 
* 
* This function can be used to determine the optical power at a previously configured current limit input value. Specified input 
* must be within the calibrated range and inputs outside of that range will result in function returning kERROR_NOT_FOUND.
*
* @public                   @memberof kLdCalLookup
* @param   lookup           Lookup object.
* @param   input            Input to the power pot, specified in device units.
* @param   temperature      Driver temperature at which the power is queried. Specify k64F_NULL to default to the calibration temperature.
* @param   limit            Pointer receiving the optical power corresponding to the specified current limit input and temperature.
* @return                   Operation status.
*/
kFsFx(kStatus) kLdCalLookup_InputToLimit(kLdCalLookup lookup, kSize input, k64f temperature, k64f* limit);

/**
* Calculates the current limit input expected to produce specified optical power at the specified temperature. 
* 
* Resulting output selects the _nearest_ input, even if it results in optical power that is higher than requested. 
*
* All specified powers _must_ already account for margins due to uncertainty in digital pot resolution, accuracy of the
* calibration methodology or efficiency of the optical path.
*
* For example, with resolution of the pot at 10% of the input, the returned power may fall within +/- 10% of the specified _before_
* accounting for calibration error, temperature compensation uncertainty, optical path efficiency uncertainty
*
* While granularity of the digital pot is only one component of the uncertainty, the optional 'expectedLimit' output parameter
* may be passed in to determine the exact calibrated optical power corresponding to the specified input
*
* @public                   @memberof kLdCalLookup
* @param   lookup           Lookup object.
* @param   limit            Requested optical power.
* @param   temperature      Driver temperature at which the power is queried. Specify k64F_NULL to default to the calibration temperature.
* @param   input            Driver input, in device units, closest to the requested power at specified temperature.
* @param   expectedLimit    Optional output parameter receiving the exact calibrated optical power corresponding to the specified input.
* @return                   Operation status.
*/
kFsFx(kStatus) kLdCalLookup_LimitToInput(kLdCalLookup lookup, k64f limit, k64f temperature, kSize* input, k64f* expectedLimit);

/**
* Calculates the power set point input value required to produce optical power that's equivalent to specified calibration set point input. 
* 
* The function accepts the power set point value that's configured at the end of the manufacturing process, which maps to the desired power at calibration
* temperature. Output of the function is the power set point input required to produce the same desired optical power at any temperature within sensors temperature
* range. Specified input must be within the calibrated range and inputs outside of that range will result in function returning kERROR_NOT_FOUND
*
* @public                     @memberof kLdCalLookup
* @param   lookup             Lookup object.
* @param   calInput           Driver power set point input, in device units, configured at shipping (matching desired optical power at calibration temperature).
* @param   intensity          Desired intensity (normalized).
* @param   temperature        Driver temperature to compensate to. Specify k64F_NULL to default to the calibration temperature.
* @param   compensatedInput   Pointer receiving driver input, in device units, matching the optical power of the uncompensated input at the specified temperature.
* @return                     Operation status.
*/
kFsFx(kStatus) kLdCalLookup_CompensatePowerInput(kLdCalLookup lookup, kSize calInput, k64f intensity, k64f temperature, kSize* compensatedInput);

/**
* Calculates the limit input value required to produce optical power that's equivalent to specified calibration limit input. 
* 
* The function accepts the limit value that's configured at the end of the manufacturing process, which maps to the desired optical power at calibration
* temperature. Output of the function is the limit input required to produce the same desired optical power at any temperature within sensors temperature
* range. Specified input must be within the calibrated range and inputs outside of that range will result in function returning kERROR_NOT_FOUND
*
* @public                     @memberof kLdCalLookup
* @param   lookup             Lookup object.
* @param   calInput           Driver current limit input, in device units, configured at shipping (matching desired power at calibration temperature).
* @param   temperature        Driver temperature to compensate to. Specify k64F_NULL to default to the calibration temperature.
* @param   compensatedInput   Pointer receiving driver input, in device units, matching the optical power of the uncompensated current limit input at the specified temperature.
* @return                     Operation status.
*/
kFsFx(kStatus) kLdCalLookup_CompensateLimitInput(kLdCalLookup lookup, kSize calInput, k64f temperature, kSize* compensatedInput);

#include <kFireSync/Utils/kLdCalLookup.x.h>


#endif
