/**
 * @file    kPacketSocket.h
 * @brief   kPacketSocket declarations.
 *
 * @internal
 * Copyright (C) 2017-2022 by LMI Technologies Inc.
 */
#ifndef K_PACKET_SOCKET_H
#define K_PACKET_SOCKET_H

#include <kApi/kApiDef.h>
#include <kFireSync/kFsDef.h>

/**
  * @class   kPacketSocket
  * @extends kObject
  * @ingroup kFireSync-Utils
  * @brief   Represents layer 2 functionality (sending and receiving of raw ethernet frames).
  *
  * The kPacketSocket class lets you send and receive layer 2 (data link 
  * layer) frames. 
  * The internal driver buffer is of a fixed size. So if you cannot process the packets 
  * fast enough by calling kPacketSocket_Receive as they arrive any incoming frame 
  * gets dropped when buffer is full.
  * 
  * Implementation uses pcap library on Windows and Linux. If implementation cannot load the pcap library 
  * (e.g. WinPcap driver is not installed on Windows) the constructor will return kERROR_STATE.
  */

//typedef kObject kPacketSocket;        --forward-declared in kFsDef.x.h

/**
 * Constructs a kPacketSocket object (non-promiscuous mode).
 *
 * You can specify that only a specific ethernet type is received by using 
 * the protocol parameter. To receive all ethernet traffic set protocol to 0.
 * For list of protocol types see http://standards-oui.ieee.org/ethertype/eth.txt
 *
 * @public                  @memberof kPacketSocket
 * @param   sockt           Destination for the constructed object handle.
 * @param   adapterName     Adapter name (e.g. "eth0", "Local Area Network").
 * @param   protocols       Protocols, e.g. 0x8892 for Profibus or 0x0800 for IP.
 * @param   protocolCount   Number of protocols 
 * @param   allocator       Memory allocator (or kNULL for default).
 * @return                  Operation status.
 */
kFsFx(kStatus) kPacketSocket_Construct(kPacketSocket* sockt, const kChar* adapterName, k16u* protocols, kSize protocolCount, kAlloc allocator);

/**
 * Constructs a kPacketSocket object with additional options.
 *
 * @public                  @memberof kPacketSocket
 * @param   sockt           Destination for the constructed object handle.
 * @param   adapterName     Adapter name (e.g. "eth0", "Local Area Network").
 * @param   protocols       Protocols, e.g. 0x8892 for Profibus or 0x0800 for IP.
 * @param   protocolCount   Number of protocols 
 * @param   promiscuousMode If kTRUE device is put in promiscuous mode.
 * @param   allocator       Memory allocator (or kNULL for default).
 * @return                  Operation status.
 */
kFsFx(kStatus) kPacketSocket_ConstructEx(kPacketSocket* sockt, const kChar* adapterName, k16u* protocols, kSize protocolCount, kBool promiscuousMode, kAlloc allocator);

/**
 * Constructs a kPacketSocket object from an IP address (non-promiscuous mode).
 *
 * @public                  @memberof kPacketSocket
 * @param   sockt           Destination for the constructed object handle.
 * @param   ipAddress       Ip address of the adapter.
 * @param   protocols       Protocols, e.g. 0x8892 for Profibus or 0x0800 for IP.
 * @param   protocolCount   Number of protocols
 * @param   allocator       Memory allocator (or kNULL for default).
 * @return                  Operation status.
 */
kFsFx(kStatus) kPacketSocket_ConstructFromIp(kPacketSocket* sockt, kIpAddress ipAddress, k16u* protocols, kSize protocolCount, kAlloc allocator);

/**
 * Constructs a kPacketSocket object from an Ip address with additional options.
 *
 * @public                  @memberof kPacketSocket
 * @param   sockt           Destination for the constructed object handle.
 * @param   ipAddress       Ip address of the adapter.
 * @param   protocols       Protocols, e.g. 0x8892 for Profibus or 0x0800 for IP.
 * @param   protocolCount   Number of protocols
 * @param   promiscuousMode If kTRUE device is put in promiscuous mode.
 * @param   allocator       Memory allocator (or kNULL for default).
 * @return                  Operation status.
 */
kFsFx(kStatus) kPacketSocket_ConstructFromIpEx(kPacketSocket* sockt, kIpAddress ipAddress, k16u* protocols, kSize protocolCount, kBool promiscuousMode, kAlloc allocator);

/**
 * Reads up to the specified number of bytes from the socket.
 *
 * @public              @memberof kPacketSocket
 * @param   sockt       kPacketSocket object.
 * @param   buffer      Destination for bytes that are read.
 * @param   size        Maximum count of bytes to read.
 * @param   bytesRead   Receives count of bytes read.
 * @param   timeout     Timeout value, in microseconds.
 * @return              Operation status.
 */
kFsFx(kStatus) kPacketSocket_Receive(kPacketSocket sockt, void* buffer, kSize size, kSize* bytesRead, k64u timeout);

/**
 * Sends the specified number of bytes.
 *
 * @public              @memberof kPacketSocket
 * @param   sockt       kPacketSocket object.
 * @param   buffer      Buffer containing the packet to send.
 * @param   size        The number of bytes in the supplied buffer.
 * @return              Operation status.
 */
kFsFx(kStatus) kPacketSocket_Send(kPacketSocket sockt, const void* buffer, k32s size);

/** 
 * Fills a string with the source and destination MAC address and hex presentation of the data.
 *
 * @public                @memberof kPacketSocket
 * @param   buffer        Buffer containing the packet.
 * @param   size          Buffer length.
 * @param   includeData   Print data section.
 * @param   output        Receives printable representation of the packet.
 * @return                Operation status. 
 */
kFsFx(kStatus) kPacketSocket_PrintPacket(const kByte* buffer, kSize size, kBool includeData, kString output); 

/** 
 * Gives extended adapter information for a given Ip address.
 *
 * @public                  @memberof kPacketSocket
 * @param   ipAddress       Ip address of the adapter.
 * @param   adapterName     Receives name of the adapter (can be kNULL).
 * @param   capacity        Character array capacity, in bytes (includes null-terminator). 
 * @param   gateway         Receives gateway address (can be kNULL).
 * @param   prefixlength    Receives prefix length (can be kNULL).
 * @param   macAddress      Receives MAC address (can be kNULL).
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPacketSocket_AdapterInfo(kIpAddress ipAddress, kChar* adapterName, kSize capacity, kIpAddress* gateway, k32u* prefixlength, kMacAddress* macAddress); 

#include <kFireSync/Utils/kPacketSocket.x.h>

#endif
