/** 
 * @file    kPageCache.h
 * @brief   Declares the kPageCache type. 
 *
 * @internal
 * Copyright (C) 2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_PAGE_CACHE_H
#define K_FIRESYNC_PAGE_CACHE_H

#include <kFireSync/kFsDef.h>
#include <kFireSync/Utils/kPageCache.x.h>

/**
 * @class   kPageCache
 * @extends kObject
 * @ingroup kFireSync-Utils
 * @brief   Helper class to support the implementation of a RAM page cache for a flash storage device. 
 * 
 * At present, this class provides features to support a read cache. Additional support (e.g., dirty flags, eviction callbacks) 
 * would be required to support a writable cache. 
 */
//typedef kObject kPageCache;            --forward-declared in kFsDef.x.h  

/**
 * Represents an entry within a kPageCache object. 
 * 
 * @typedef kPointer kPageCacheItem
 * @relates kPageCache
 */
typedef kPointer kPageCacheItem;

/** 
 * Constructs a kPageCache object.
 *
 * @public                  @memberof kPageCache
 * @param   cache           Destination for the constructed object handle. 
 * @param   pageCapacity    Maximum count of pages to cache. 
 * @param   pageSize        Size of each page, in bytes.
 * @param   spareSize       Size of spare/tag memory associated with each page.
 * @param   allocator       Memory allocator (or kNULL for default).
 * @return                  Operation status. 
 */
kFsFx(kStatus) kPageCache_Construct(kPageCache* cache, kSize pageCapacity, kSize pageSize, kSize spareSize, kAlloc allocator); 

/** 
 * Invalidates the entire page cache.
 *
 * @public      @memberof kPageCache
 * @return      Operation status.  
 */
kFsFx(kStatus) kPageCache_Invalidate(kPageCache cache);

/** 
 * Invalidates a specific page.
 *
 * @public              @memberof kPageCache
 * @param   cache       kPageCache object.  
 * @param   pageIndex   Page index associated with the page to be invalidated. 
 * @return              Operation status.  
 */
kFsFx(kStatus) kPageCache_InvalidatePage(kPageCache cache, kSize pageIndex);

/** 
 * Invalidates a ranges of pages.
 *
 * @public              @memberof kPageCache
 * @param   cache       kPageCache object.  
 * @param   pageIndex   First page index associated with the range to be invalidated. 
 * @param   pageCount   Count of pages to be invalidated. 
 * @return              Operation status.  
 */
kFsFx(kStatus) kPageCache_InvalidateRange(kPageCache cache, kSize pageIndex, kSize pageCount);

/** 
 * Finds the cache page associated with the specified page index, if present. 
 *
 * @public              @memberof kPageCache
 * @param   cache       kPageCache object.  
 * @param   pageIndex   Page index. 
 * @param   page        Receives a reference to a page cache entry, if found.  
 * @return              Operation status.  
 */
kFsFx(kStatus) kPageCache_FindPage(kPageCache cache, kSize pageIndex, kPageCacheItem* page);

/** 
 * Finds the cache page associated with the specified page index (if present) and updates LRU information for the page.
 *
 * @public              @memberof kPageCache
 * @param   cache       kPageCache object.  
 * @param   pageIndex   Page index. 
 * @param   page        Receives a reference to a page cache entry, if found.  
 * @return              Operation status.  
 */
kFsFx(kStatus) kPageCache_AccessPage(kPageCache cache, kSize pageIndex, kPageCacheItem* page);

/** 
 * Adds a page associated with the specified page index to the page cache.
 *
 * If there is a page already associated with the specified page index, it will be replaced. 
 *
 * If there are no free pages available, the least recently used page will be replaced.  
 * 
 * @public              @memberof kPageCache
 * @param   cache       kPageCache object.  
 * @param   pageIndex   Page index. 
 * @param   page        Receives a reference to the new page cache entry.  
 * @return              Operation status.  
 */
kFsFx(kStatus) kPageCache_AllocatePage(kPageCache cache, kSize pageIndex, kPageCacheItem* page);

/** 
 * Copies page data into a cache page. 
 * 
 * @public              @memberof kPageCache
 * @param   cache       kPageCache object.  
 * @param   page        Reference to a page cache entry.  
 * @param   source      Pointer to source data. 
 * @param   length      Length of source data. 
 * @return              Operation status.  
 */
kFsFx(kStatus) kPageCache_CopyToPageData(kPageCache cache, kPageCacheItem page, const kPointer source, kSize length);

/** 
 * Copies page data out of a cache page. 
 * 
 * @public              @memberof kPageCache
 * @param   cache       kPageCache object.  
 * @param   page        Reference to a page cache entry.  
 * @param   dest        Pointer to destination buffer. 
 * @param   length      Length of destination buffer. 
 * @return              Operation status.  
 */
kFsFx(kStatus) kPageCache_CopyFromPageData(kPageCache cache, kPageCacheItem page, kPointer dest, kSize length);

/** 
 * Copies spare data into a cache page spare area. 
 * 
 * @public              @memberof kPageCache
 * @param   cache       kPageCache object.  
 * @param   page        Reference to a page cache entry.  
 * @param   source      Pointer to source spare data. 
 * @param   length      Length of source spare data. 
 * @return              Operation status.  
 */
kFsFx(kStatus) kPageCache_CopyToPageSpare(kPageCache cache, kPageCacheItem page, const kPointer source, kSize length);

/** 
 * Copies spare data out of a cache page spare area. 
 * 
 * @public              @memberof kPageCache
 * @param   cache       kPageCache object.  
 * @param   page        Reference to a page cache entry.  
 * @param   dest        Pointer to destination spare buffer. 
 * @param   length      Length of destination spare buffer. 
 * @return              Operation status.  
 */
kFsFx(kStatus) kPageCache_CopyFromPageSpare(kPageCache cache, kPageCacheItem page, kPointer dest, kSize length);

/** 
 * Reports the page size for the cache.
 * 
 * @public              @memberof kPageCache
 * @param   cache       kPageCache object.  
 * @return              Page size, in bytes.  
 */
kInlineFx(kSize) kPageCache_PageSize(kPageCache cache)
{
    kObj(kPageCache, cache);

    return obj->pageSize;
}

/** 
 * Reports the spare size for the cache.
 * 
 * @public              @memberof kPageCache
 * @param   cache       kPageCache object.  
 * @return              Spare size, in bytes.  
 */
kInlineFx(kSize) kPageCache_SpareSize(kPageCache cache)
{
    kObj(kPageCache, cache);

    return obj->spareSize;
}

/** 
 * Returns a pointer to a cache page data buffer. 
 * 
 * @public              @memberof kPageCache
 * @param   cache       kPageCache object.  
 * @param   page        Reference to a cache page.
 * @return              Pointer to page data.   
 */
kInlineFx(kPointer) kPageCache_PageData(kPageCache cache, kPageCacheItem page)
{
    kAssertType(cache, kPageCache);

    return ((kPageCacheEntry*)page)->data;
}

/** 
 * Returns a pointer to a cache spare data buffer. 
 * 
 * @public              @memberof kPageCache
 * @param   cache       kPageCache object.  
 * @param   page        Reference to a cache page.
 * @return              Pointer to page spare data.   
 */
kInlineFx(kPointer) kPageCache_PageSpare(kPageCache cache, kPageCacheItem page)
{
    kAssertType(cache, kPageCache);

    return ((kPageCacheEntry*)page)->spare;
}

#endif
