/** 
 * @file    kFireSync/Utils/kStorageStream.h
 * @brief   Declares the kStorageStream class. 
 *
 * @internal
 * Copyright (C) 2012-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_STORAGE_STREAM_H
#define K_FIRESYNC_STORAGE_STREAM_H

#include <kApi/Io/kStream.h>
#include <kFireSync/kFsDef.h>

/**
 * @class   kStorageStream
 * @extends kStream
 * @ingroup kFireSync-Utils
 * @brief   Represents a stream used to write/read files to/from a remote sensor.
 * 
 * This class uses a kMemory object to adapt the kStream interface to the kStorage API. 
 *
 * Read/write operations are performed on the kMemory object; flusing the stream or destroying
 * the kStorageStream object will cause a write to storage. 
 * 
 */
//typedef kStream kStorageStream;        --forward-declared in kFsDef.x.h

/** 
 * Constructs a kStorageStream object.
 * 
 * The module argument and device id are used to resolve the storage module. If the 
 * the module argument is not a kStorage or kNode object, its ancestors are traversed until a system 
 * object is located. The system is then searched for the requested device id, and 
 * the first storage module of the selected device is used. 
 *
 * @public              @memberof kStorageStream
 * @param   stream      Destination for the constructed object handle. 
 * @param   module      A kStorage, kNode, or kSystem object. 
 * @param   deviceId    Sensor serial number; ignored if the module argument is a node or storage module.
 * @param   fileName    File name. 
 * @param   read        If kTRUE, existing file contents are loaded upon opening.
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kStorageStream_Construct(kStorageStream* stream, kObject module, k32u deviceId, const kChar* fileName, kBool read, kAlloc allocator); 

/** 
 * Returns the current length of the stream. 
 *
 * @public              @memberof kStorageStream
 * @param   stream      StorageStream object. 
 * @return              Length of the stream, in bytes.
 */
kFsFx(k64u) kStorageStream_Length(kStorageStream stream); 

/** 
 * Returns the current position of the read/write pointer, relative to the beginning of the stream. 
 *
 * @public              @memberof kStorageStream
 * @param   stream      StorageStream object. 
 * @return              Offset of read/write pointer from beginning of stream. 
 */
kFsFx(k64u) kStorageStream_Position(kStorageStream stream); 

#include <kFireSync/Utils/kStorageStream.x.h>

#endif
