#!/usr/bin/python

import subprocess
import os
import io
import sys
import re

def makePlatformPathsRelative(path, inLines):

    realPath = os.path.realpath(path)
    dirName = os.path.dirname(os.path.dirname(realPath))

    outLines = []
    for readLine in inLines:
        readLine = readLine.rstrip()

        if (readLine[:3] == '-l"' and readLine[-1:] == '"' and not 'bios_6' in readLine):
            foundPath = readLine[3:-1]
            relativePath = os.path.relpath(foundPath, dirName)

            outLines.append('-l"' + relativePath + '"\n')
        else:
            outLines.append(readLine + '\n')

    return outLines

def addProgramMemory(inLines):

    # normally, the size of the DDR2 section known to the linker is defined entirely by the hardware platform (kHwDm647)
    # however, editing the hardware platform is tedious and doesn't allow to discriminate between Debug/Release
    # therefore, the value from the hardware platform is altered by this script

    ddrProgramMemToAddToDebugBuilds =   0x1000000
    ddrProgramMemToAddToReleaseBuilds = 0x0400000

    return [re.sub(r'([\s]+)DDR2 \(([A-Z]+)\) : org = (0x[0-9a-f]+), len = (0x[0-9a-f]+)',
        lambda m: 
            '#ifdef _DEBUG\n' + \
            m.groups()[0] + 'DDR2 (' + m.groups()[1] + ') : org = ' + m.groups()[2] + ', len = ' + hex(int(m.groups()[3], 16) + ddrProgramMemToAddToDebugBuilds) + '\n' + \
            '#else\n' + \
            m.groups()[0] + 'DDR2 (' + m.groups()[1] + ') : org = ' + m.groups()[2] + ', len = ' + hex(int(m.groups()[3], 16) + ddrProgramMemToAddToReleaseBuilds) + '\n' + \
            '#endif',
        inLine) for inLine in inLines]

def modifyLinkerScript(path):

    with io.open(path) as inFile:
        inLines = inFile.readlines()

    outLines = makePlatformPathsRelative(path, inLines)
    outLines = addProgramMemory(outLines)

    with io.open(path, 'w') as outFile:
        outFile.writelines(outLines)

def build(javaHome, xdcXs, biosPath, ccsPath, cgtoolsPath):

    herePath = os.path.dirname(os.path.realpath(__file__))
    os.environ['XDCTOOLS_JAVA_HOME'] = javaHome

    subprocess.check_call([xdcXs,
        '--xdcpath=' + os.path.join(biosPath, 'packages') + ';' + os.path.join(ccsPath, 'ccs_base') + ';.',
        'xdc.tools.configuro',
        '-o', os.path.join(herePath, 'configPkg'),
        '-t', 'ti.targets.elf.C64P',
        '-p', 'kHwDm647',
        '-r', 'release',
        '-c', cgtoolsPath,
        '--compileOptions=-mv64p --abi=eabi -q -o2',
        os.path.join(herePath, 'kHwDm647.cfg')],
        cwd=herePath)

    modifyLinkerScript(os.path.join(herePath, 'configPkg', 'linker.cmd'))

if __name__ == '__main__':

    build(sys.argv[1], sys.argv[2], sys.argv[3], sys.argv[4], sys.argv[5])
