/** 
 * @file    kSparseMatrix.x.h
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef kVS_SPARSE_MATRIX_X_H
#define kVS_SPARSE_MATRIX_X_H

#include <kApi/Data/kArray1.h>
#include <kApi/Data/kArray2.h>
#include <kApi/Data/kArrayList.h>
#include <kVision/Vs/kVsUtilities.h>

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

typedef struct kSparseMatrixEntry
{
   k32u row;   // size metters
   k64f value;

} kSparseMatrixEntry;

kDeclareValueEx(kVs, kSparseMatrixEntry, kValue)

//////////////////////////////////////////////////////////////////////////
// column major sparse matrix
//////////////////////////////////////////////////////////////////////////

typedef struct kSparseMatrixClass
{
    kObjectClass base;

    kSize rowCount;
    kSize columnCount;

    kArray1 outerIndexes;       // k32u // index of each column
    kArray1 innerNonZeroCounts; // k32u // number of non zeroes in each column

    kArrayList entries; // kSparseMatrixEntry
    
} kSparseMatrixClass;

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

kDeclareClassEx(kVs, kSparseMatrix, kObject)

//semi-private exported functions (virtual override methods)
kVsFx(kStatus) kSparseMatrix_VInitClone(kSparseMatrix matrix, kSparseMatrix source, kAlloc allocator);
kVsFx(kStatus) kSparseMatrix_VRelease(kSparseMatrix matrix);

//non-exported (private) methods
kStatus kSparseMatrix_Init(kSparseMatrix matrix, kAlloc allocator);

//cast macro

//////////////////////////////////////////////////////////////////////////
// Inline array acces with checks
// TEMP code till kArray is fixed!
//////////////////////////////////////////////////////////////////////////

kInlineFx(void*) kVsArray1Reference(kArray1 arr, kSize i)
{
    kAssert(kObject_Is(arr, kTypeOf(kArray1)));
    kAssert(i < kArray1_Count(arr));

    return kArray1_At(arr, i);
}

kInlineFx(k32u) kVsArray1Value32u(kArray1 arr, kSize i)
{
    return *(k32u*)kVsArray1Reference(arr, i);
}

kInlineFx(k32u*) kVsArray1Reference32u(kArray1 arr, kSize i)
{
    return (k32u*)kVsArray1Reference(arr, i);
}

kInlineFx(k64f) kVsArray1Value64f(kArray1 arr, kSize i)
{
    return *(k64f*)kVsArray1Reference(arr, i);
}

kInlineFx(k64f*) kVsArray1Reference64f(kArray1 arr, kSize i)
{
    return (k64f*)kVsArray1Reference(arr, i);
}

kInlineFx(kBool) kVsArray1ValueBool(kArray1 arr, kSize i)
{
    return *(kBool*)kVsArray1Reference(arr, i);
}

kInlineFx(kBool*) kVsArray1ReferenceBool(kArray1 arr, kSize i)
{
    return (kBool*)kVsArray1Reference(arr, i);
}

kInlineFx(kSparseMatrixEntry*) kVsArrayListReferenceEntry(kArrayList arr, kSize i)
{
    kAssert(kObject_Is(arr, kTypeOf(kArrayList)));
    kAssert(i < kArrayList_Count(arr));

    return (kSparseMatrixEntry*)kArrayList_At(arr, i);
}

//////////////////////////////////////////////////////////////////////////
// kSize vs k32u problem
//////////////////////////////////////////////////////////////////////////

kInlineFx(k32u) kS3dCastIndex(kSize index)
{
   kAssert(index < k32U_MAX);

   return (k32u)index;
}

//////////////////////////////////////////////////////////////////////////
// Size
//////////////////////////////////////////////////////////////////////////

kInlineFx(kSize) kSparseMatrix_RowCount(kSparseMatrix matrix) 
{ 
   return xkSparseMatrix_Cast(matrix)->rowCount; 
}

kInlineFx(kSize) kSparseMatrix_ColumnCount(kSparseMatrix matrix) 
{ 
   return xkSparseMatrix_Cast(matrix)->columnCount; 
}

//////////////////////////////////////////////////////////////////////////
// Column
//////////////////////////////////////////////////////////////////////////

kInlineFx(kSize) kSparseMatrix_ColumnIndex(kSparseMatrix matrix, kSize column) 
{ 
   return kVsArray1Value32u(xkSparseMatrix_Cast(matrix)->outerIndexes, column);
}

kInlineFx(kSize) kSparseMatrix_ColumnEntriesCount(kSparseMatrix matrix, kSize column) 
{ 
   return kVsArray1Value32u(xkSparseMatrix_Cast(matrix)->innerNonZeroCounts, column);
}

//////////////////////////////////////////////////////////////////////////
// Entry
//////////////////////////////////////////////////////////////////////////

kInlineFx(k64f) kSparseMatrix_EntryValueAt(kSparseMatrix matrix, kSize index) 
{ 
   return kVsArrayListReferenceEntry(xkSparseMatrix_Cast(matrix)->entries, index)->value; 
}

kInlineFx(kSize) kSparseMatrix_EntryRowAt(kSparseMatrix matrix, kSize index) 
{ 
   return kVsArrayListReferenceEntry(xkSparseMatrix_Cast(matrix)->entries, index)->row; 
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

kInlineFx(kSize) kSparseMatrix_NonZeroCount(kSparseMatrix matrix)
{
   kObj(kSparseMatrix, matrix);
   kSize i;
   kSize sum = 0;

   for (i = 0; i < kArray1_Length(obj->innerNonZeroCounts); ++i)
   {
      sum += kVsArray1Value32u(obj->innerNonZeroCounts, i);
   }

   return sum;
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kSparseMatrix_Insert(kSparseMatrix matrix, kSize row, kSize column, k64f value)
{
   kObj(kSparseMatrix, matrix);
   k32u newIndex;
   kSparseMatrixEntry *entry;

   kAssert(row < obj->rowCount && column < obj->columnCount);
   kAssert(kVsArray1Value32u(obj->innerNonZeroCounts, column) <= kVsArray1Value32u(obj->outerIndexes, column) + kVsArray1Value32u(obj->outerIndexes, column + 1));

   newIndex = kVsArray1Value32u(obj->outerIndexes, column) + (*kVsArray1Reference32u(obj->innerNonZeroCounts, column))++;

   entry = kVsArrayListReferenceEntry(obj->entries, newIndex);

   entry->row = kS3dCastIndex(row);
   entry->value = value;

   return kOK;
}

//////////////////////////////////////////////////////////////////////////
// internal 
//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kSparseMatrix_EndColumn(kSparseMatrix matrix, kSize column)
{
   kObj(kSparseMatrix, matrix);

   kAssert(kVsArray1Value32u(obj->outerIndexes, column) + kVsArray1Value32u(obj->innerNonZeroCounts, column) == kArrayList_Count(obj->entries));
   kAssert(kVsArray1Value32u(obj->outerIndexes, column + 1) == 0);

   *kVsArray1Reference32u(obj->outerIndexes, column + 1) = kVsArray1Value32u(obj->outerIndexes, column) + kVsArray1Value32u(obj->innerNonZeroCounts, column);

   return kOK;
}


kInlineFx(kStatus) kSparseMatrix_InsertColumnBack(kSparseMatrix matrix, kSize row, kSize column, k64f value)
{
   kObj(kSparseMatrix, matrix);
   kSparseMatrixEntry entry;

   kAssert(kVsArray1Value32u(obj->outerIndexes, column) + kVsArray1Value32u(obj->innerNonZeroCounts, column) == kArrayList_Count(obj->entries));
   
   ++(*kVsArray1Reference32u(obj->innerNonZeroCounts, column));

   entry.row = kS3dCastIndex(row);
   entry.value = value;

   kCheck(kArrayList_Add(obj->entries, &entry));

   return kOK;
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#endif  /* #ifndef kVS_SPARSE_MATRIX_X_H */
