/**
* @file    kG3dContourHistogram.h
* @brief   Declares the kG3dContourHistogram class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_kS3D_COUNTURHISTOGRAM_H
#define KVISION_kS3D_COUNTURHISTOGRAM_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>
#include <kVision/G3d/kG3dContourSegment.h>
#include <kVision/G3d/kG3dContourTemplate.h>

#define kG3D_COUNTURHISTOGRAM_DEFAULT_SIZEX    72
#define kG3D_COUNTURHISTOGRAM_DEFAULT_SIZEY    300

#define kG3D_COUNTURHISTOGRAM_DEFAULT_THRESHOLD_RATE    100

/**
* @class       kG3dContourHistogram
* @extends     kObject
* @ingroup     kVision-G3d
* @brief       Class for detection of straight lines from candidate points by using Hough transformation.
*              The points are eliminated or regrouped after detection.
*              Note: Unlike the standard definition, the gradient direction of the point is used in this algorithm.
*/
typedef kObject kG3dContourHistogram;

/**
* Constructs a kG3dContourHistogram object
*
* @public              @memberof kG3dContourHistogram
* @param   histogram   Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kG3dContourHistogram_Construct(kG3dContourHistogram* histogram, kAlloc allocator);

/**
* Performs the detection of straight lines from candidate points by using Hough transformation.
*
* @public                 @memberof kG3dContourFitLine
* @param   histogram      kG3dContourHistogram object.
* @param   contourPoints  Candidate points for the rough grouping of straight lines.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourHistogram_Run(kG3dContourHistogram histogram, kArray1 contourPoints);

/**
* Set the width and height of the corresponding height map.
*
* @public                 @memberof kG3dContourHistogram
* @param   histogram      kG3dContourHistogram object.
* @param   sizeX          Width of the corresponding height map.
* @param   sizeY          Height of the corresponding height map.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourHistogram_SetImageSize(kG3dContourHistogram histogram, kSize sizeX, kSize sizeY);

/**
* Get the number of detected straight lines.
*
* @public                 @memberof kG3dContourHistogram
* @param   histogram      kG3dContourHistogram object.
* @return                 Number of detected straight lines.
*/

kVsFx(kSize) kG3dContourHistogram_LineCount(kG3dContourHistogram histogram);

/**
* Set a boolean value to determine, whether only border points should be involved in the calculation
*
* @public                 @memberof kG3dContourFitLine
* @param   fitLine        kG3dContourFitLine object.
* @param   histogram      kG3dContourHistogram object.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourHistogram_SetBorderLineOnly(kG3dContourHistogram histogram, kBool borderOnly);

#include <kVision/G3d/kG3dContourHistogram.x.h>

#endif  /* KVISION_kS3D_COUNTURHISTOGRAM_H */
