/**
* @file    kG3dPalette.h
* @brief   Declares the kG3dPalette Class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_G3D_PALETTE_H
#define KVISION_G3D_PALETTE_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>
#include <kVision/S3d/kS3dVolumeCheckUtilities.h>

#define kVISIONG3D_PALETTE_NUMBER    (4)
#define kVISIONG3D_PALETTE_POINT_MAX (11)
#define kVISIONG3D_PALETTE_ARRAY_COUNT (256)

typedef enum kG3dPaletteType
{
    kG3dDepthImagePal = 0,
    kG3dDifferenceImagePal,
    kG3dPhotoFrontPal,
    kG3dPhotoBackPal,
    kG3dPaletteEnd
}kG3dPaletteType;

typedef struct kG3dPaletteItem
{
    k32s count;
    kBool hsvFormat;
    k32u supportingPoints[kVISIONG3D_PALETTE_POINT_MAX];
}kG3dPaletteItem;

/**
* @class       kG3dPalette
* @extends     kObject
* @ingroup     kVision-M3d
* @brief       Provides some utility functions to manage color palette for surface display.
*/
typedef kObject kG3dPalette;

/**
* Constructs a kG3dPalette object with default settings of the color palette tables
*
* @public             @memberof kG3dPalette
* @param   palette    Destination for the constructed object handle.
* @param   allocator  Memory allocator (or kNULL for default).
* @return             Operation status.
*/
kVsFx(kStatus) kG3dPalette_Construct(kG3dPalette* palette, kAlloc allocator);

/**
* Constructs a kG3dPalette object by readding the color palette tables from a tree data object.
*
* @public             @memberof kG3dPalette
* @param   palette    Destination for the constructed object handle.
* @param   tree       kDataTree object of system data.
* @param   item       kDataTree_Root(tree).
* @return             Operation status.
*/
kVsFx(kStatus) kG3dPalette_Load(kG3dPalette* palette, kDataTree tree, kDataTreeItem item);

/**
* Write color palette tables to a tree data object.
*
* @public             @memberof kG3dPalette
* @param   palette    kG3dPalette object.
* @param   tree       kDataTree object of system data.
* @param   item       kDataTree_Root(tree).
* @return             Operation status.
*/
kVsFx(kStatus) kG3dPalette_write(kG3dPalette palette, kDataTree tree, kDataTreeItem item);

/**
* Get color palette array with given type.
*
* @public            @memberof kG3dPalette
* @param   palette   kG3dPalette object.
* @param   type      Value of kG3dPaletteType.
* @return            color palette array with given type.
*/
kVsFx(kArray1) kG3dPalette_palArray(kG3dPalette palette, kG3dPaletteType type);

#include <kVision/G3d/kG3dPalette.x.h>

#endif
