/**
* @file    kG3dSegmentPlanes.h
* @brief   Declares the kG3dSegmentPlanes class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_kG3D_SEGMENT_PLANES_H
#define KVISION_kG3D_SEGMENT_PLANES_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>
#include <kVision/G3d/kG3dContourTemplate.h>

/**
* @class       kG3dSegmentPlanes
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       This is a modified class to determine the multiple equalization planes 
*              at the same time with overlay support.
*/
typedef kObject kG3dSegmentPlanes;

/**
* Constructs a kG3dSegmentPlanes object
*
* @public                 @memberof kG3dSegmentPlanes
* @param   planes         Destination for the constructed object handle.
* @param   allocator      Memory allocator (or kNULL for default).
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dSegmentPlanes_Construct(kG3dSegmentPlanes* planes, kAlloc allocator);

/**
* Performs the evaluation of glue band height measurements without template
*
* @public                 @memberof kG3dSegmentPlanes
* @param   planes         kG3dSegmentPlanes object.
* @param   heightMap      Data buffer for input height map.   kArray2 < k16s >
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dSegmentPlanes_Run(kG3dSegmentPlanes planes, kArray2 heightmap);

/**
* Set the data buffer of magnet track object, which includes information of glue band geometry for each track.
*
* @public                 @memberof kG3dSegmentPlanes
* @param   planes         kG3dSegmentPlanes object.
* @param   tracks         Data buffer of magnet track object. 
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dSegmentPlanes_SetMeasureTracks(kG3dSegmentPlanes planes, kArrayList tracks);

/**
* Set the scaling coeffizients of the main height map.
*
* @public                 @memberof kG3dSegmentPlanes
* @param   planes         kG3dSegmentPlanes object.
* @param   scale          scaling coeffizients of the height map in the 3 directions.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dSegmentPlanes_SetScale(kG3dSegmentPlanes planes, kPoint3d64f scale);

/**
* Set an affine transformation matrix.
*
* @public                 @memberof kG3dSegmentPlanes
* @param   planes         kG3dSegmentPlanes object.
* @param   trans          Affine transformation matrix
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dSegmentPlanes_SetTransform(kG3dSegmentPlanes planes, kL3dTransform3d trans);

/**
* Set a mask map for the evaluation.
*
* @public                 @memberof kG3dSegmentPlanes
* @param   planes         kG3dSegmentPlanes object.
* @param   maskMap        Mask map for the evaluation   kArray2 < 16s >
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dSegmentPlanes_SetMaskMap(kG3dSegmentPlanes planes, kArray2 maskMap);

/**
* Set a data buffer to store the deviation during the evaluation.
*
* @public                 @memberof kG3dSegmentPlanes
* @param   planes         kG3dSegmentPlanes object.
* @param   diffMap        Data buffer to store the deviation   kArray2 < 16s >
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dSegmentPlanes_SetDifferenceMap(kG3dSegmentPlanes planes, kArray2 diffMap);

/**
* Set the corner points for the evaluation.
*
* @public                 @memberof kG3dSegmentPlanes
* @param   planes         kG3dSegmentPlanes object.
* @param   points         Corner points
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dSegmentPlanes_SetCornerPoints(kG3dSegmentPlanes planes, kArray1 points);

/**
* Get the maximum limit of the height map.
*
* @public                 @memberof kG3dSegmentPlanes
* @param   planes         kG3dSegmentPlanes object.
* @return                 maximum limit of the height map.
*/
kVsFx(k64f) kG3dSegmentPlanes_MaximumZ(kG3dSegmentPlanes planes);

/**
* Get the minimum limit of the height map.
*
* @public                 @memberof kG3dSegmentPlanes
* @param   planes         kG3dSegmentPlanes object.
* @return                 maximum limit of the height map.
*/
kVsFx(k64f) kG3dSegmentPlanes_MinimumZ(kG3dSegmentPlanes planes);

/**
* Set window width for calculating the average height value.
*
* @public                 @memberof kG3dSegmentPlanes
* @param   planes         kG3dSegmentPlanes object.
* @param   width          Window width for calculating the average height value.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dSegmentPlanes_SetPointAverageWindowWidth(kG3dSegmentPlanes planes, kSize width);

/**
* Set minimum outlier rate. This parameter is used to restrict the outlier of the measured data on 
* the glue band according to a histogram of height. The height distrubution of points with an area 
* proportion to the total area of the glue band, which are determined by this values, are regarded 
* as outliers. They are not involved in the calculation.
*
* @public                 @memberof kG3dSegmentPlanes
* @param   planes         kG3dSegmentPlanes object.
* @param   rate           Minimum outlier rate.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dSegmentPlanes_SetOutlierMinRate(kG3dSegmentPlanes planes, k64f rate);

/**
* Set maximum outlier rate. 
*
* @public                 @memberof kG3dSegmentPlanes
* @param   planes         kG3dSegmentPlanes object.
* @param   rate           Maximum outlier rate. 
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dSegmentPlanes_SetOutlierMaxRate(kG3dSegmentPlanes planes, k64f rate);

#include <kVision/G3d/kG3dSegmentPlanes.x.h>

#endif  /* KVISION_kG3D_SEGMENT_PLANES_H */
