/**
 * @file    kS3dEmbeddedPhaseDecoder.x.h
 *
 * @internal
 * Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef kS3D_EMBEDDED_PHASE_DECODER_X_H
#define kS3D_EMBEDDED_PHASE_DECODER_X_H

//////////////////////////////////////////////////////////////////////////
// Micro Phase Shifting http://www.cs.columbia.edu/CAVE/projects/micro_phase_shifting/
// solving for M*U = R same as in MPS paper (better described than in EPS paper)
//
// Embedded Phase Shifting http://mesh.brown.edu/eps/
// Paper SVN:  https://svn1.lmi-tech.com/sensors/Gocator/Documents/3xxx/Papers/Moreno_Embedded_Phase_Shifting_2015_CVPR_paper.pdf
//////////////////////////////////////////////////////////////////////////

typedef struct kS3dEmbeddedPhaseDecoderPhaseAccum 
{
    k32f accumCos; 
    k32f accumSin;

} kS3dEmbeddedPhaseDecoderPhaseAccum;

typedef struct kS3dEmbeddedPhaseDecoderPrimeAccum 
{
    k8u minValue;
    k8u maxValue;
    k16u sum;

} kS3dEmbeddedPhaseDecoderPrimeAccum;

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

typedef struct kS3dEmbeddedPhaseDecoderClass 
{
    kObjectClass base;

    // params //
    kSize imageWidth;
    kSize imageHeight;

    kArray1 coefficients;    // k64f
    kArray1 embeddedPeriods; // k64f
    kArray1 patternPeriods;  // k64f

    kArray1 stepCounts;      // kSize, images per period

    kSize phaseScale;        // k16s phase = [0,1] * phaseScale;
    kSize contrastThreshold;

    // internals //
    kSize periodCount;       // num frequencies
    kSize imageCount;        // sequence length
    kSize pixelCount;        // imageHeight * imageWidth

    kSize filterWindow;

    kSize updatedImageCount; // how many images already in

    kBool updateSystem;
    kBool updateImageSize;
    
    // Using paper notations here 
    // M*U = R               
    // Mt*M*U = Mt*R
    // U = inv(Mt*M)*Mt*R
    // A = inv(Mt*M)*Mt
    // U = A*R
    //
    // R - intensities vector, no variable
    // A - shiftMatrix mangled as M is a true shift matrix
    // U - phaseAccumArray

    kArray2 shiftMatrix; // kArray2<k64f>
    kArray1 phaseAccumArray; // kArray1<kArray2<kS3dEmbeddedPhaseDecoderPhaseAccum>> per period

    kArray2 primeAccum; // kArray2<kS3dEmbeddedPhaseDecoderPrimeAccum>
    kArray1 phaseArray; // kArray1<kArray2<k32f>> per period    

    kArray2 tmpPhase;       // temp buffer

    kArray2 outputBaseMap;  // k8u
    kArray2 outputPhaseMap; // kPhasePixel

    kArrayList outputPhaseArrayState; // unit test report kArrayList<kArray2<k32f>> [n alg steps][coefficientCount]

    // moving sum array used by kS3dEmbeddedPhaseDecoder_SmoothKuwahara()
    // TODO: create a standalone class
    kArray2 movingSum;

    // atan2() lookup table used by kS3dEmbeddedPhaseDecoder_Atan2()
    // TODO: create a standalone class
    kArray1 atanLut;

    // Job Support //
    kVsJobQueue jobQueue;
    kArray2 sourceData;   // used to pass params to jobs
    kArray2 targetData;   // same
    kImage sourceImage;   // same

   kS3dEmbeddedPhaseDecoderOrientation orientation;

} kS3dEmbeddedPhaseDecoderClass;

kDeclareClassEx(kVs, kS3dEmbeddedPhaseDecoder, kObject)

//////////////////////////////////////////////////////////////////////////
//semi-private exported functions (virtual override methods)
//////////////////////////////////////////////////////////////////////////

kVsFx(kStatus) kS3dEmbeddedPhaseDecoder_VRelease(kS3dEmbeddedPhaseDecoder decoder);
kVsFx(kSize)   kS3dEmbeddedPhaseDecoder_VSize(kS3dEmbeddedPhaseDecoder decoder);

//////////////////////////////////////////////////////////////////////////
//non-exported (private) methods
//////////////////////////////////////////////////////////////////////////

kStatus kS3dEmbeddedPhaseDecoder_Init(kS3dEmbeddedPhaseDecoder decoder, kAlloc allocator);

kStatus kS3dEmbeddedPhaseDecoder_UpdateSystem(kS3dEmbeddedPhaseDecoder decoder);
kStatus kS3dEmbeddedPhaseDecoder_UpdateAccumulators(kS3dEmbeddedPhaseDecoder decoder);
kStatus kS3dEmbeddedPhaseDecoder_UpdateAtanLut(kS3dEmbeddedPhaseDecoder decoder);

kStatus kS3dEmbeddedPhaseDecoder_UpdateFinishF3(kS3dEmbeddedPhaseDecoder decoder);
kStatus kS3dEmbeddedPhaseDecoder_UpdateFinishF2(kS3dEmbeddedPhaseDecoder decoder);
kStatus kS3dEmbeddedPhaseDecoder_UpdateFinishFN(kS3dEmbeddedPhaseDecoder decoder);

kStatus kS3dEmbeddedPhaseDecoder_CalculatePhase(kS3dEmbeddedPhaseDecoder decoder, kArray2* accum, kArray2* phase);
kStatus kS3dEmbeddedPhaseDecoder_SubtractPhase(kS3dEmbeddedPhaseDecoder decoder, kArray2 fN, kArray2 f1);
kStatus kS3dEmbeddedPhaseDecoder_SubtractPhaseBase(kS3dEmbeddedPhaseDecoder decoder, kArray2 fN, kArray2 f1);

kStatus kS3dEmbeddedPhaseDecoder_SmoothKuwahara(kS3dEmbeddedPhaseDecoder decoder, kSize window, kArray2 inPhase, kArray2 outPhase);

kStatus kS3dEmbeddedPhaseDecoder_BaseRamp(kS3dEmbeddedPhaseDecoder decoder, k32f primePeriod, kArray2 primePhase, k32f basePeriod, kArray2 basePhase);

kStatus kS3dEmbeddedPhaseDecoder_Unwrap(kS3dEmbeddedPhaseDecoder decoder, k32f phasePeriod, kArray2 phase, k32f basePeriod, kArray2 base);

kStatus kS3dEmbeddedPhaseDecoder_Export(kS3dEmbeddedPhaseDecoder decoder, kArray2 base);

kStatus kS3dEmbeddedPhaseDecoder_MatrixConvert(kArray2 mIn, kArray2 mOut, k32s value);

//////////////////////////////////////////////////////////////////////////
//cast macro, virtual table macro
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
// row bounds for multi-threading
//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kS3dEmbeddedPhaseDecoder_BlockInfo(kS3dEmbeddedPhaseDecoder decoder, kSize id, kSize* outStartRow, kSize* outRowCount)
{
    kObj(kS3dEmbeddedPhaseDecoder, decoder);

    const kSize height = obj->imageHeight;
    const kSize blockCount = kVsJobQueue_ThreadCount(obj->jobQueue);
    const kSize blockSize = height / blockCount + 1; // + 1 in case height < blockCount 
    
    const kSize startRow = blockSize * id;

    if (startRow >= height) return kERROR_ABORT;

    *outStartRow = startRow;
    *outRowCount = kMin_(blockSize, height - startRow);

    return kOK;
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

kInlineFx(kStatus) kS3dEmbeddedPhaseDecoder_UpdatePhaseState(kS3dEmbeddedPhaseDecoder decoder)
{
    kObj(kS3dEmbeddedPhaseDecoder, decoder);

    if (kIsNull(obj->outputPhaseArrayState)) return kOK;    

    kStatus status;
    kAlloc alloc = kObject_Alloc(decoder);
    kArray1 tmp = kNULL;

    kTry;
    {
        kTest(kObject_Clone(&tmp, obj->phaseArray, alloc));

        kTest(kArrayList_Add(obj->outputPhaseArrayState, &tmp));
    }
    kCatch(&status);
    {
        kEndCatch(status);
    }

    return kOK;
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#endif  /* #ifndef kS3D_EMBEDDED_PHASE_DECODER_X_H */
