/**
* @file    kS3dPhaseDecoder.h
* @brief   Declares the kS3dPhaseDecoder class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef kS3D_PHASE_DECODER_H
#define kS3D_PHASE_DECODER_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>

/**
* @class       kS3dPhaseDecoder
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Performs phase shift sequence decoding
*/
typedef kObject kS3dPhaseDecoder;

/**
* Constructs a kS3dPhaseDecoder object
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder     Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseDecoder_Construct(kS3dPhaseDecoder* decoder, kAlloc alloc);

/**
* Constructs a kS3dPhaseDecoder object
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder     Destination for the constructed object handle.
* @param   useCuda     Enable CUDA?
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseDecoder_ConstructEx(kS3dPhaseDecoder* decoder, kBool useCuda, kAlloc alloc);

/**
* Updates working buffers of the implementation to reflect current set of algorithm parameters.
* Calling this function is optional, as this validation step is also performed during each execution
* of the algorithm (kS3dPhaseDecoder_Begin).
* However, the initialization time may be non-negligible, which would affect the first execution of the algorithm.
* To eliminate the added delay from the first algorithm execution, the user should call kS3dPhaseDecoder_Setup
* after all of the parameters have been configured.
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder     Decoder object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseDecoder_Setup(kS3dPhaseDecoder decoder);

/**
* Initiates decoding sequence and initializes the output buffer for use.  Should be called prior to calling 
* kS3dPhaseDecoder_Update. The output buffer handle is retained by the decoder
* object and is used to populate the phase map. Upon completion of the sequence (kS3dPhaseDecoder_IsComplete returns kTRUE)
* each entry in the output buffer is populated with the decoded phase value, contrast and intensity of the phase map. Pixels which were
* not decoded are identified by having the phase value of the entry set to k16S_NULL
* 
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder     Decoder object
* @param   output      Output buffer. Expected kArray2<kPhasePixel> equal in size to the image data
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseDecoder_Begin(kS3dPhaseDecoder decoder, kArray2 output);

/**
* Returns true when all phase sequence images have been processed and the output has been finalized. 
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder     Decoder object
* @return              Operation status.
*/
kVsFx(kBool) kS3dPhaseDecoder_IsComplete(kS3dPhaseDecoder decoder);

/**
* Update phase shift sequence image. 
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder     Decoder object
* @param   index       Index of the image within the sequence
* @param   image       Phase image (kImage<k8u>)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseDecoder_Update(kS3dPhaseDecoder decoder, kSize index, kImage image);

/**
* Sets the size of the expected input images.
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder      Decoder object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseDecoder_SetImageSize(kS3dPhaseDecoder decoder, kSize imageWidth, kSize imageHeight);

/**
* Gets the size of input images
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder      Decoder object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseDecoder_ImageSize(kS3dPhaseDecoder decoder, kSize* imageWidth, kSize* imageHeight);

/**
* Sets the number of images in the stripe sequence (excluding the reference image) as well as the bit depth of the stripe map
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder     Decoder object
* @param   imageCount  Number of images in the stripe sequence (excluding the reference image)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseDecoder_SetImageCount(kS3dPhaseDecoder decoder, kSize imageCount);

/**
* Gets the number of images in the stripe sequence (excluding the reference image) as well as the bit depth of the stripe map
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder     Decoder object
* @return              Number of images in the stripe sequence (excluding the reference image)
*/
kVsFx(kSize) kS3dPhaseDecoder_ImageCount(kS3dPhaseDecoder decoder);

/**
* Sets contrast threshold for phase decoding. The range of intensities for a given pixel location among all phase
* images must exceed this threshold value in order for the phase decoding to produce a valid result. 
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder     Decoder object
* @param   threshold   Contrast threshold for phase decoding
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseDecoder_SetContrastThreshold(kS3dPhaseDecoder decoder, kSize threshold);

/**
* Gets contrast threshold for phase decoding
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder     Decoder object
* @return              Contrast threshold for phase decoding
*/
kVsFx(kSize) kS3dPhaseDecoder_ContrastThreshold(kS3dPhaseDecoder decoder);

/**
* Sets the scale of the phase period. The resulting phase output is scaled such that 0 to 2Pi phase values are mapped to 0 to period 
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder     Decoder object
* @param   period      Phase period scale
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseDecoder_SetPhasePeriod(kS3dPhaseDecoder decoder, kSize period);

/**
* Gets the scale of the phase period. 
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder     Decoder object
* @return              Phase period scale
*/
kVsFx(kSize) kS3dPhaseDecoder_PhasePeriod(kS3dPhaseDecoder decoder);

/**
* Specify whether floating point arctan function should be used instead of a precomputed lookup table. Disabled by default
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder     Decoder object
* @param   enable      Flag specifying whether exact arctan should be used
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseDecoder_EnableExactArctan(kS3dPhaseDecoder decoder, kBool enable);

/**
* Query whether floating point arctan function is used for decoding 
*
* @public              @memberof kS3dPhaseDecoder
* @param   decoder     Decoder object
* @return              Flag specifying whether exact arctan is used
*/
kVsFx(kBool) kS3dPhaseDecoder_ExactArctanEnabled(kS3dPhaseDecoder decoder);

/**
* Sets cuda stream for cuda parallel executions.
*
* @public              @memberof kS3dPhaseDecoder
* @param   sampler     Sampler object
* @param   cudaStream  kCudaStream class
* @return              Operation status.
*/

kVsFx(kStatus) kS3dPhaseDecoder_SetCudaStream(kS3dPhaseDecoder decoder, kCudaStream cudaStream);

#include <kVision/S3d/kS3dPhaseDecoder.x.h>

#endif  /* kS3D_PHASE_DECODER_H */
