/**
* @file    kS3dPhaseViewProcessor.h
* @brief   Declares the kS3dPhaseViewProcessor class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef kS3D_PHASE_VIEW_PROCESSOR_H
#define kS3D_PHASE_VIEW_PROCESSOR_H

#include <kVision/S3d/kS3dStereoProfiler.h>
#include <kVision/S3d/kS3dStripeDecoder.h>
#include <kVision/S3d/kS3dPhaseDecoder.h>
#include <kVision/S3d/kS3dEmbeddedPhaseDecoder.h>
#include <kVision/S3d/kS3dPhaseCombiner.h>
#include <kVision/S3d/kS3dPhaseSampler.h>
#include <kVision/S3d/kS3dPhaseFilter.h>
#include <kVision/S3d/kS3dPhaseCorrection.h>

/**
* @class       kS3dPhaseViewProcessor
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Rectifies and resamples camera phase maps to a uniform step in projector space and uniform step in rectified Y space. For a
*              a pair of phase maps acquired from a stereo camera pair, this sampling ensures that corresponding locations in the output buffers
*              form matching stereo points
*/
typedef kObject kS3dPhaseViewProcessor;

/**
* Constructs a kS3dPhaseViewProcessor object
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor     Destination for the constructed object handle.
* @param   profiler    Profiler object to be used for rectification
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_Construct(kS3dPhaseViewProcessor* processor, kSize viewIndex, kS3dStereoProfiler profiler, kAlloc alloc);

/**
* Updates working buffers of the implementation to reflect current set of algorithm parameters.
* Calling this function is optional, as this validation step is also performed during each execution
* of the algorithm (kS3dPhaseViewProcessor_Begin, kS3dPhaseViewProcessor_Update).
* However, the initialization time may be non-negligible, which would affect the first execution of the algorithm.
* To eliminate the added delay from the first algorithm execution, the user should call kS3dPhaseViewProcessor_Setup
* after all of the parameters have been configured.
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_Setup(kS3dPhaseViewProcessor processor);

/**
* Initiates image processing sequence and initializes the output buffers. Should be called prior to calling
* kS3dPhaseViewProcessor_UpdateImageInput or kS3dPhaseViewProcessor_UpdateMapInput.  The output buffer handles are retained 
* by the processor object and are used to populate the output rectified coordinate and intensity maps. 
*
* Upon completion of the sequence (kS3dPhaseViewProcessor_IsComplete returns kTRUE)
* each entry in the output x projection buffer 'outputXProj' is populated with rectified X coordinates of the points sampled at 
* resolution specified via kS3dPhaseViewProcessor_SetPhaseRoi and kS3dPhaseViewProcessor_SetYProjectionRoi
* Invalid points are set to k16S_NULL
* 
* Optional parameter 'outputIntensity' provides a buffer for intensity output for the points corresponding to the X projection
* coordinates at the same locations. Invalid values are set to zero. 
*
* Optional parameter 'outputTexture' provides a buffer for texture output for the points corresponding to the X projection
* coordinates at the same locations. Texture pixel coordinates are returned with 8-bit fixed point precision with invalid coordinates
* specified as {k32S_NULL, k32S_NULL}
*
* @public                   @memberof kS3dPhaseViewProcessor
* @param   processor        Processor object
* @param   outputXProj      Output buffer for rectified X coordinates. Expects kArray2<k16s> with width determined by 
*                           kS3dPhaseViewProcessor_SetYProjectionRoi 'count' parameter and height determined by 
*                           kS3dPhaseViewProcessor_SetPhaseRoi 'count' parameter
* @param   outputIntensity  Output intensity buffer. Can be set to kNULL if intensity output isn't needed. Expect kArray2<k8u> with
*                           dimensions equal to those of 'outputXProj'
* @param   outputTexture    Output texture buffer. Can be set to kNULL if texture output isn't needed. Expect kArray2<kPoint32s> with
*                           dimensions equal to those of 'outputXProj'. 
* @return                   Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_Begin(kS3dPhaseViewProcessor processor, kArray2 outputXProj, kArray2 outputIntensity, kArray2 outputTexture);

/**
* Returns true when all image or map sequences have been processed and the outputs supplied to kS3dPhaseViewProcessor_Begin
* have been filled
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @return              Operation status.
*/
kVsFx(kBool) kS3dPhaseViewProcessor_IsComplete(kS3dPhaseViewProcessor processor);

/**
* Update the processor with a single input image in the processing sequence.
* The images are expected in order of reference, stripe, phase with the exact number of images 
* in each sub-sequence controlled via kS3dPhaseViewProcessor_SetSequenceLengthInfo()
* The total number of sequences must match the count specified via kS3dPhaseViewProcessor_SetSequenceCount() function
* 
* Prior to beginning image processor the user should enable image input via kS3dPhaseViewProcessor_EnableImageInput
* and call kS3dPhaseViewProcessor_Begin to supply output buffers. The function kS3dPhaseViewProcessor_IsComplete will
* return true once all of the expected input images have been received
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @param   image       Image input. Expect kImage<k8u> with size determined by kS3dPhaseViewProcessor_SetImageSize call
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_UpdateImage(kS3dPhaseViewProcessor processor, kImage image);

/**
* Update the processor a previously decoded stripe and phase map for a single sequence. 
* The total number of sequences must match the count specified via kS3dPhaseViewProcessor_SetSequenceCount() function
*
* A call kS3dPhaseViewProcessor_Begin to supply output buffers is required prior to calling this function.
* The function kS3dPhaseViewProcessor_IsComplete will return true once all of the expected sequence inputs have been processed
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @param   phasePixel2 Decoded and unwrapped phase map. Expect kArray2<kPhasePixel2 or k32s> with size determined by kS3dPhaseViewProcessor_SetImageSize call.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_UpdatePixel2(kS3dPhaseViewProcessor processor, kArray2 phasePixel2);

/**
* Plot stripe, phase, intensity and contrast data resulting from the decoding stage of the pipeline
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @param   rawImage    Optional handle to a raw image to be copied as a top layer to the plot. The class does not 
*                      take ownership of the image
* @param   plot        Output pointer to the resulting plot object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_DecodePlot(kS3dPhaseViewProcessor processor, kImage rawImage, kPlot* plot);

/**
* Plot stripe and phase data as well as image coordinates sampled along the projection axis
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @param   plot        Output pointer to the resulting plot object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_PhaseSamplePlot(kS3dPhaseViewProcessor processor, kPlot* plot);

/**
* Enables image input. Must be set to kTRUE prior to using kS3dPhaseViewProcessor_UpdateImageInput function. 
* If disabled, only direct map input is possible (kS3dPhaseViewProcessor_UpdateMapInput)
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @param   enable      Flag specifying whether image input should be enabled
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_EnableImageInput(kS3dPhaseViewProcessor processor, kBool enable);

/**
* Gets the value of the flag specifying whether raw image input is enabled. 
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @return              Flag specifying whether image input is enabled
*/
kVsFx(kBool) kS3dPhaseViewProcessor_ImageInputEnabled(kS3dPhaseViewProcessor processor);

/**
* Sets cuda stream for parallel cuda executions.
*
* @public                        @memberof kS3dPhaseViewProcessor
* @param   processor             Processor object.
* @param   cudaStream            kCudaStream object.
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_SetCudaStream(kS3dPhaseViewProcessor processor, kCudaStream cudaStream);

/**
* Gets cuda stream for parallel cuda executions.
*
* @public                        @memberof kS3dPhaseViewProcessor
* @param   processor             Processor object.
* @return                        kCudaStream object in use.
*/
kVsFx(kCudaStream) kS3dPhaseViewProcessor_CudaStream(kS3dPhaseViewProcessor processor);

/**
* Sets the lengths of the Gray code and phase subsequences. At most one reference image is supported. 
* If no stripe sequence is used (phase period occupies full projection)
* the number of reference images must also be zero. Total number of expected images is the sum of the individual 
* sub-sequence counts
*
* @public                        @memberof kS3dPhaseViewProcessor
* @param   processor             Processor object
* @param   referenceImageCount   Number of reference images in the image sequence (<= 1)
* @param   stripeImageCount      Number of stripe images in the image sequence (<= 7)
* @param   phaseImageCount       Number of phase images in the sequence
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_SetStripeSequenceInfo(kS3dPhaseViewProcessor processor, kSize referenceImageCount, kSize stripeImageCount, kSize phaseImageCount);

/**
* Gets the lengths of the Gray code and phase subsequences.
*
* @public                        @memberof kS3dPhaseViewProcessor
* @param   processor             Processor object
* @param   referenceImageCount   Number of reference images in the image sequence (<= 1)
* @param   stripeImageCount      Number of stripe images in the image sequence (<= 7)
* @param   phaseImageCount       Number of phase images in the sequence
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_StripeSequenceInfo(kS3dPhaseViewProcessor processor, kSize* referenceImageCount, kSize* stripeImageCount, kSize* phaseImageCount);

/**
* Sets period coefficients and the number of images per period for the embedded phase scanning.
*
* @public                     @memberof kS3dPhaseViewProcessor
* @param   processor          Processor object
* @param   coefficients       Array of coefficients describing embedded periods (k64f).
* @param   stepCounts         Array specifying number of images per period to use (kSize same size as "coefficients" and == coefficientCount).
* @param   coefficientCount   Size of the "coefficients" and "stepCounts" arrays.
* @return                     Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_SetEmbeddedPhaseSequenceInfo(kS3dPhaseViewProcessor processor, const k64f* coefficients, const kSize* stepCounts, kSize coefficientCount);

/**
* Gets period coefficients and the number of images per period.
* Coefficients are used to calculate the phase periods per set of patterns used.
*
* @public                        @memberof kS3dPhaseViewProcessor
* @param   processor             Processor object
* @param   outCoefficients       Array of coefficients describing embedded periods (k64f).
* @param   outStepCounts         Array specifying number of images per period to use (kSize same size as "coefficients" and == coefficientCount).
* @param   outCoefficientCount   Number of values in the "coefficients" and "stepCounts" arrays returned 
* @param   capacity              Size of "coefficients" and "stepCounts" arrays passed in (capacity should be >= coefficientCount).
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_EmbeddedSequenceInfo(kS3dPhaseViewProcessor processor, k64f* outCoefficients, kSize* outStepCounts, kSize* outCoefficientCount, kSize capacity);

/**
* Sets the image encoding (decoding) type 
*
* @public                        @memberof kS3dPhaseViewProcessor
* @param   processor             Processor object
* @param   type                  One of: kS3D_ENCODE_TYPE_STRIPE, kS3D_ENCODE_TYPE_EMBEDDED_PHASE
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_SetEncodingType(kS3dPhaseViewProcessor processor, kS3dEncodeType type);

/**
* Gets the image encoding (decoding) type
*
* @public                        @memberof kS3dPhaseViewProcessor
* @param   processor             Processor object
* @return                        One of: kS3D_ENCODE_TYPE_STRIPE, kS3D_ENCODE_TYPE_EMBEDDED_PHASE
*/
kVsFx(kS3dEncodeType) kS3dPhaseViewProcessor_EncodingType(kS3dPhaseViewProcessor processor);

/**
* Sets the total number of decoding sequences
*
* @public                @memberof kS3dPhaseViewProcessor
* @param   processor     Processor object
* @param   sequenceCount Number of expected decoding sequences
* @return                Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_SetSequenceCount(kS3dPhaseViewProcessor processor, kSize count);

/**
* Gets the total number of decoding sequences
*
* @public                @memberof kS3dPhaseViewProcessor
* @param   processor     Processor object
* @return                Number of expected decoding sequences
*/
kVsFx(kSize) kS3dPhaseViewProcessor_SequenceCount(kS3dPhaseViewProcessor processor);

/**
* Sets the index of the sequence to be used for intensity output
*
* @public                @memberof kS3dPhaseViewProcessor
* @param   processor     Processor object
* @param   index         Index of the sequence to be used for intensity output
* @return                Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_SetIntensitySequenceIndex(kS3dPhaseViewProcessor processor, kSize index);

/**
* Gets the index of the sequence to be used for intensity output
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @return              Index of the sequence to be used for intensity output
*/
kVsFx(kSize) kS3dPhaseViewProcessor_IntensitySequenceIndex(kS3dPhaseViewProcessor processor);

/**
* Sets contrast threshold for stripe decoding. The difference between stripe image pixel intensity and reference pixel intensity
* must be higher or equal to this value in order for the decoding at that pixel to be valid. Note that the value set by this function 
* has no effect if map input is used
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @param   threshold   Contrast threshold for stripe decoding
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_SetStripeContrastThreshold(kS3dPhaseViewProcessor processor, kSize threshold);

/**
* Gets contrast threshold for stripe decoding
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @return              Contrast threshold for stripe decoding
*/
kVsFx(kSize) kS3dPhaseViewProcessor_StripeContrastThreshold(kS3dPhaseViewProcessor processor);

/**
* Sets contrast threshold for phase decoding. The range of intensities for a given pixel location among all phase
* images must exceed this threshold value in order for the phase decoding to produce a valid result.
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @param   threshold   Contrast threshold for phase decoding
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_SetPhaseContrastThreshold(kS3dPhaseViewProcessor processor, kSize threshold);

/**
* Gets contrast threshold for phase decoding
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @return              Contrast threshold for phase decoding
*/
kVsFx(kSize) kS3dPhaseViewProcessor_PhaseContrastThreshold(kS3dPhaseViewProcessor processor);

/**
* Sets the size of the expected input images.
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor      Processor object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_SetImageSize(kS3dPhaseViewProcessor processor, kSize imageWidth, kSize imageHeight);

/**
* Gets the size of input images
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor     Processor object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_ImageSize(kS3dPhaseViewProcessor processor, kSize* imageWidth, kSize* imageHeight);

/**
* Sets the scale of the phase period. The phase values expected by the kS3dPhaseViewProcessor_Run function must be scaled such that
* 0 to 2Pi phase offset is mapped to 0 to period
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @param   period      Phase period scale
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_SetPhasePeriod(kS3dPhaseViewProcessor processor, kSize period);

/**
* Gets the scale of the phase period.
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @return              Phase period scale
*/
kVsFx(kSize) kS3dPhaseViewProcessor_PhasePeriod(kS3dPhaseViewProcessor processor);

/**
* Sets the parameters determining resampling region of interest along projection phase modulation axis (rows in both camera and projection images)
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @param   begin       Starting cumulative phase value (starting stripe * phasePeriod + periodic starting phase)
* @param   stepShift   Shift value representing the phase increment between columns in the output buffers. Exact increment is calculated as
*                      (1 << stepShift).
* @param   count       Number of points along the projection modulation axis (output buffer height)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_SetPhaseRoi(kS3dPhaseViewProcessor processor, kSize begin, kSize stepShift, kSize count);

/**
* Gets the parameters determining resampling region of interest along projection phase modulation axis (rows in both camera and projection images)
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @param   begin       Output pointer to the starting cumulative phase value (starting stripe * phasePeriod + periodic starting phase)
* @param   stepShift   Output pointer to the shift value representing the phase increment between columns in the output buffers. Exact increment is calculated as
*                      (1 << stepShift).
* @param   count       Output pointer to the number of points along the projection modulation axis (output buffer height)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_PhaseRoi(kS3dPhaseViewProcessor processor, kSize* begin, kSize* stepShift, kSize* count);

/**
* Sets the parameters determining resampling region of interest along the rectified Y axis (columns in both camera and projection images)
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @param   begin       Starting rectified Y value expressed in calibration object projection resolution units
*                      (begin * kS3dStereoCal_Parameters(cal)->projectionResolution converts to mm value)
* @param   step        Step size along the rectified Y axis (step * kS3dStereoCal_Parameters(cal)->projectionResolution converts to mm value)
* @param   count       Number of points along the rectified Y axis (output buffer width)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_SetYProjectionRoi(kS3dPhaseViewProcessor processor, kSSize begin, kSize step, kSize count);

/**
* Gets the parameters determining resampling region of interest along the rectified Y axis (columns in both camera and projection images)
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @param   begin       Output pointer to the starting rectified Y value expressed in calibration object projection resolution units
*                      (begin * kS3dStereoCal_Parameters(cal)->projectionResolution converts to mm value)
* @param   step        Output pointer to the step size along the rectified Y axis
*                      (step * kS3dStereoCal_Parameters(cal)->projectionResolution converts to mm value)
* @param   count       Output pointer to the number of points along the rectified Y axis (output buffer width)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_YProjectionRoi(kS3dPhaseViewProcessor processor, kSSize* begin, kSize* step, kSize* count);

/**
* Set the type of correction to be applied to phase map data. Supported options:
* kS3D_PHASE_CORRECTION_TYPE_NONE - do not perform any correction (default when processor class is constructed)
* kS3D_PHASE_CORRECTION_TYPE_MULTI_REFLECTION - correct phase from multi reflection e.g.Combustion chamber.
* kS3D_PHASE_CORRECTION_TYPE_TRANSLUCENCY - correct phase for transparent surface, e.g. skin surface measurement.
*
* @public                  @memberof kS3dPhaseViewProcessor
* @param   processor       Processor object
* @param   correctionType  Phase correction type
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_SetPhaseCorrectionType(kS3dPhaseViewProcessor processor, kS3dPhaseCorrectionType correctionType);

/**
* Gets the type of correction to be applied to phase map data.
*
* @public              @memberof kS3dPhaseViewProcessor
* @param   processor   Processor object
* @return              Phase correction type
*/
kVsFx(kS3dPhaseCorrectionType) kS3dPhaseViewProcessor_PhaseCorrectionType(kS3dPhaseViewProcessor processor);

/**
* Set the margin around valid phase boundary to erode. This operation can help clean up spikes and transitions around object edges caused by lens blur. 
* The value is specified in pixels and is zero by default
*
* @public                  @memberof kS3dPhaseViewProcessor
* @param   processor       Processor object
* @param   erosionMargin   Erosion margin (px)
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_SetErosionMargin(kS3dPhaseViewProcessor processor, kSize erosionMargin);

/**
* Gets the erosion margin around valid phase boundary.
*
* @public                  @memberof kS3dPhaseViewProcessor
* @param   processor       Processor object
* @return                  Erosion margin (px)
*/
kVsFx(kSize) kS3dPhaseViewProcessor_ErosionMargin(kS3dPhaseViewProcessor processor);

/**
* Get the raw stripe output as kImage<k8u>.
*
* @public                  @memberof kS3dPhaseViewProcessor
* @param   processor       Processor object
* @param   stripe          Pointer to a kImage that will be filled with the output.
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_Stripe(kS3dPhaseViewProcessor processor, kImage* stripe);

/**
* Get the raw phase output as kImage<k16s>.
*
* @public                  @memberof kS3dPhaseViewProcessor
* @param   processor       Processor object
* @param   phase           Pointer to a kImage that will be filled with the output.
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_Phase(kS3dPhaseViewProcessor processor, kImage* phase);

/**
* Get the intensity output as kImage<k8u>.
*
* @public                  @memberof kS3dPhaseViewProcessor
* @param   processor       Processor object
* @param   intensity       Pointer to a kImage that will be filled with the output.
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_Intensity(kS3dPhaseViewProcessor processor, kImage* intensity);

/**
* Get the unwrapped (stripe+phase) output as a kImage<k64f>. Unwrapped phase value is StripeValue * PhasePeriod + PhaseValue
* providing a continuous index across the phase pattern, combining the stripe and phase data.
* 
* @public                  @memberof kS3dPhaseViewProcessor
* @param   processor       Processor object
* @param   unwrappedPhase  Pointer to a kImage that will be filled with the unwrapped phase output
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_UnwrappedPhase(kS3dPhaseViewProcessor processor, kImage* unwrappedPhase);

/**
* Get the high frequency unwrapped phase output as a kImage<k64f>. High frequency phase is the difference between the 
* unwrapped phase and a 5x5 pixel moving average in X. In other words, a high-pass filter is applied. This method produces
* reliable results on contiguous targets only, since the moving average presumes continuity.
*
* @public                  @memberof kS3dPhaseViewProcessor
* @param   processor       Processor object
* @param   unwrappedPhase  Pointer to a kImage that will be filled with the unwrapped phase output
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_HighFrequencyPhase(kS3dPhaseViewProcessor processor, kImage* highFreqPhase);

/**
* Sets the flag for cuda optimization.
*
* @public                        @memberof kS3dPhaseViewProcessor
* @param   processor             Processor object
* @param   cudaEnabled           Flag to enable cuda optimizations
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_EnableCuda(kS3dPhaseViewProcessor processor, kBool cudaEnabled);

/**
* Gets the flag for cuda optimization.
*
* @public                        @memberof kS3dPhaseViewProcessor
* @param   processor             Processor object
* @return                        Cuda flag state.
*/
kVsFx(kBool) kS3dPhaseViewProcessor_CudaEnabled(kS3dPhaseViewProcessor processor);

/**
* Sets the flag for retain phase map input. This will allow phase combiner to save phase map input buffer which used 
* to generate decode plot
*
* @public                        @memberof kS3dPhaseViewProcessor
* @param   processor             Processor object
* @param   enable                Flag to retain phase map input
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_EnableRetainInput(kS3dPhaseViewProcessor processor, kBool enable);

/**
* Gets the flag for retain phase map input.
*
* @public                        @memberof kS3dPhaseViewProcessor
* @param   processor             Processor object
* @return                        retain input flag state.
*/
kVsFx(kBool) kS3dPhaseViewProcessor_RetainInputEnabled(kS3dPhaseViewProcessor processor);

/**
* Sets the flag to enable the use of cuda managed memory.
*
* @public                        @memberof kS3dPhaseViewProcessor
* @param   processor             Processor object
* @param   cudaManagedMem        Flag to enable cuda managed memory
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseViewProcessor_EnableManagedMemory(kS3dPhaseViewProcessor processor, kBool cudaManagedMem);

/**
* Gets the flag for cuda managed memory.
*
* @public                        @memberof kS3dPhaseViewProcessor
* @param   processor             Processor object
* @return                        Cuda managed memory state.
*/
kVsFx(kBool) kS3dPhaseViewProcessor_ManagedMemoryEnabled(kS3dPhaseViewProcessor processor);

#include <kVision/S3d/kS3dPhaseViewProcessor.x.h>

#endif  /* kS3D_PHASE_VIEW_PROCESSOR_H */
