/**
* @file    kS3dSpeckleMatchAlg.h
* @brief   Declares the kS3dSpeckleMatchAlg class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef KS3D_SPECKLE_MATCH_ALG_H
#define KS3D_SPECKLE_MATCH_ALG_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>

/**
* @class       kS3dSpeckleMatchAlg
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Base class for block matching algorithm implementation
*/
typedef kObject kS3dSpeckleMatchAlg;

/**
* Executes the algorithm on a pair of rectified and pre-filtered images and outputs the disparity information.
* Currently, the algorithm is limited to 8-bit unsigned images only. Disparity should be a pre-allocated
* kArray2 of type k16s. Both images and the output disparity array must have the same dimensions.
* The output disparity values are fixed point with fractional number of bits controlled by
* kS3dSpeckleMatchAlg_SetDisparityBitDepth function. Cells, where matches could not be found
* are set to k16S_NULL.
*
* For all supported algorithms (BM, SGBM) images should be pre-filtered leave only the local texture
* capped at a value range specified to the algorithm through the 'filter cap' parameter (kS3dSpeckleMatchAlg_SetFilterCap).
* Since the provided images are unsigned, the algorithm expects the intensity values to be offset by filter cap (thus mapping it to zero)
* This pre-processing step can be performed using algorithms available in kS3dSpecklePreFilter class (normalized response, Sobel). Please 
* note that each algorithm may have different restrictions on the maximum allowable value for the filter cap. Review the class description
* for more specific details
* 
* Note that the implementation of the algorithm requires that the images and the disparity array
* include padding proportional to the specified disparity range. The exact number of columns required
* for padding should be queried using kS3dSpeckleMatchAlg_DisparityMargin function. The values in the padded
* region should be set to zero. If the padding is not included then r.h.s portion of the disparity
* array equivalent to the margin will not be populated with valid disparity data. This is a limitation
* of the source OpenCV implementation and will likely be addressed in further revisions to this algorithm.
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @param   image0      First rectified image
* @param   image1      Second rectified image
* @param   disparity   Output disparity map
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleMatchAlg_Run(kS3dSpeckleMatchAlg alg, kImage image0, kImage image1, kArray2 disparity);

/**
* Updates working buffers of the implementation to reflect current set of algorithm parameters.
* Calling this function is optional, as this validation step is also performed during each execution
* of the algorithm (kS3dSpeckleMatchAlg_Run). However, the initialization time may be non-negligible, which would
* affect the first execution of the algorithm. To eliminate the added delay from the first algorithm
* execution, the user should call kS3dSpeckleMatchAlg_Setup after all of the parameters have been configured.
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleMatchAlg_Setup(kS3dSpeckleMatchAlg alg);

/**
* Sets the size of the expected input images. Dimensions of the disparity output buffer should equal that of the image with
* the addition of the horizontal margin of size returned by kS3dSpeckleMatchAlg_DisparityMargin
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleMatchAlg_SetImageSize(kS3dSpeckleMatchAlg alg, kSize imageWidth, kSize imageHeight);

/**
* Gets the size of input images
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleMatchAlg_ImageSize(kS3dSpeckleMatchAlg alg, kSize* imageWidth, kSize* imageHeight);

/**
* Set output disparity bit depth. The precision should be set such that the remaining bits in the
* 16-bit output value are sufficient to cover the user-specified disparity range (e.g. 6-bit bit
* depth allows for maximum disparity range of 1024).
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @param   bitDepth    Bit depth for disparity calculation and output
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleMatchAlg_SetDisparityBitDepth(kS3dSpeckleMatchAlg alg, k32u bitDepth);

/**
* Get output disparity bit depth
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @return              Bit depth for disparity calculation and output
*/
kVsFx(k32u) kS3dSpeckleMatchAlg_DisparityBitDepth(kS3dSpeckleMatchAlg alg);

/**
* Set matching window size (in rectified pixels). It must be an odd number >=1. Values should approximately
* range between 3 and 11 pixels. Higher window size decreases the amount of detail, but improves robustness
* in detecting matches in low texture cases
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @param   window      Block matching window size (px)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleMatchAlg_SetMatchWindow(kS3dSpeckleMatchAlg alg, k32u window);

/**
*  Get matching window size
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @return              Block matching window size (px)
*/
kVsFx(k32u) kS3dSpeckleMatchAlg_MatchWindow(kS3dSpeckleMatchAlg alg);

/**
* Set the intensity truncation value for the image filter output (from 1 to 127). Variation in image intensity between
* the two camera images will generally be proportional to the absolute magnitude. Truncating the filter output can help
* improve the robustness of the cost calculation by clipping higher intensities to a fixed value (which would be the
* same for both images). For images where the intensity variation between the two cameras is expected to be low, the
* truncation should be set high or disabled (using maximum of 127 counts). The field is set to maximum by default.
*
* @public               @memberof kS3dSpeckleMatchAlg
* @param   alg          Algorithm object
* @param   filterCap    Image filter truncation value
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleMatchAlg_SetFilterCap(kS3dSpeckleMatchAlg alg, k32u filterCap);
/**
*  Get the image filter truncation value
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @return              Image filter truncation value
*/
kVsFx(k32u) kS3dSpeckleMatchAlg_FilterCap(kS3dSpeckleMatchAlg alg);

/**
* Set search disparity range (in rectified pixels). The value should be a multiple of
* the value returned by kS3dSpeckleMatchAlg_DisparityRangeGranularity. 
* Outside of very small disparity ranges (< 32 pixels)
* the computation time is directly proportional to the specified value
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @param   range       Disparity search range (px)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleMatchAlg_SetDisparityRange(kS3dSpeckleMatchAlg alg, k32u range);

/**
*  Get search disparity range (in rectified pixels)
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @return              Disparity search range (px)
*/
kVsFx(k32u) kS3dSpeckleMatchAlg_DisparityRange(kS3dSpeckleMatchAlg alg);

/**
*  Set search disparity centre (in rectified pixels).
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @param   centre      Disparity search centre (px)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleMatchAlg_SetDisparityCentre(kS3dSpeckleMatchAlg alg, k32s centre);

/**
*  Get search disparity centre (in rectified pixels).
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @return              Disparity search centre (px)
*/
kVsFx(k32s) kS3dSpeckleMatchAlg_DisparityCentre(kS3dSpeckleMatchAlg alg);

/**
* Set margin in percentage (0 to 1) by which the best (minimum) computed cost function value should "win" the second
* best value to consider the found match correct. Normally, a value within the 0.05-0.15 range is sufficient
*
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @param   ratio       Uniqueness ratio (0 to 1)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dSpeckleMatchAlg_SetUniquenessRatio(kS3dSpeckleMatchAlg alg, k64f ratio);

/**
*  Get the match uniqueness ratio
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @return              Uniqueness ratio (0 to 1)
*/
kVsFx(k64f) kS3dSpeckleMatchAlg_UniquenessRatio(kS3dSpeckleMatchAlg alg);

/**
* Maximum allowed difference (in integer pixel units) in the left-right disparity check. Set it to k32U_NULL
* to disable the check
*
*
* @public               @memberof kS3dSpeckleMatchAlg
* @param   alg          Algorithm object
* @param   threshold    Left-right disparity check threshold (pixels)
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleMatchAlg_SetDisparityLrThreshold(kS3dSpeckleMatchAlg alg, k32u threshold);

/**
*  Get the left-right disparity check threshold
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @return              Left-right disparity check threshold (number of pixels)
*/
kVsFx(k32u) kS3dSpeckleMatchAlg_DisparityLrThreshold(kS3dSpeckleMatchAlg alg);

/**
* Some algorithms (e.g. BM) support implicitly filtering out matches which did not produce enough contrast. Use 
* the texture threshold setting to invalidate those points. For algorithms, where this behaviour is not supported, 
* filtering the disparity values after the algorithm completed is needed. The default, which produces good results in most
* cases, is 5 counts (max - min within a patch).
*
* @public               @memberof kS3dSpeckleMatchAlg
* @param   alg          Algorithm object
* @param   threshold    Texture threshold
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleMatchAlg_SetTextureThreshold(kS3dSpeckleMatchAlg alg, k32u threshold);

/**
*  Get the texture threshold
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @return              Left-right disparity check threshold (number of pixels)
*/
kVsFx(k32u) kS3dSpeckleMatchAlg_TextureThreshold(kS3dSpeckleMatchAlg alg);

/**
* Set the weight of the adjacency cost used in selecting a particular disparity path.
* Higher values (default is 1.0) increase the smoothness of the data as the adjacency of pixels in a path begins to outweigh
* the difference between actual greyscale intensities. The combination of continuity and occlusion weight
* scaled by the matching window size produces P1 and P2 penalty values in the SGBM algorithm
* NOTE: The parameter has no effect on local algoithms 
*
*
* @public               @memberof kS3dSpeckleMatchAlg
* @param   alg          Algorithm object
* @param   weight       Weight of adjacency within a disparity path
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleMatchAlg_SetContinuityWeight(kS3dSpeckleMatchAlg alg, k64f weight);

/**
*  Get the continuity weight
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @return              Weight of adjacency within a disparity path
*/
kVsFx(k64f) kS3dSpeckleMatchAlg_ContinuityWeight(kS3dSpeckleMatchAlg alg);

/**
* Set the weight of the penalty introduced in a correspondence path due to a change of more than one pixel
* Higher values will penalize occlusions (jumps in disparity match sequence) 
* more. In most cases the default of 4.0 does not need to be modified.
*
* @public               @memberof kS3dSpeckleMatchAlg
* @param   alg          Algorithm object
* @param   ratio        Weight of occlusion penalty in a disparity path
* @return               Operation status.
*/
kVsFx(kStatus) kS3dSpeckleMatchAlg_SetOcclusionWeight(kS3dSpeckleMatchAlg alg, k64f weight);

/**
*  Get the occlusion weight
*   
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @return              Weight of occlusion penalty in a disparity path
*/
kVsFx(k64f) kS3dSpeckleMatchAlg_OcclusionWeight(kS3dSpeckleMatchAlg alg);

/**
* Retrieve the additional number of columns that needs to be included in the input images and the disparity buffer for
* internal calculations by the BM algorithm. The padding is proportional to the specified disparity range.
* The values in the padded region should be set to zero. If the padding is not included then r.h.s portion of the disparity
* array equivalent to the margin will not be populated with valid disparity data. This is a limitation
* of the source OpenCV implementation and will likely be addressed in further revisions to this algorithm.
*
* For supported matching algorithms the value is not fixed and may depend on other parameters, such as 
* matching window, disparity centre and disparity range
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @return              Disparity margin (number of columnns)
*/
kVsFx(k32u) kS3dSpeckleMatchAlg_DisparityMargin(kS3dSpeckleMatchAlg alg);

/**
* Retrieve the granularity and the minimum of the disparity range setting. For currently supported
* implementations the value is fixed.
* 
*
*
* @public              @memberof kS3dSpeckleMatchAlg
* @param   alg         Algorithm object
* @return              Disparity range granularity
*/
kVsFx(k32u) kS3dSpeckleMatchAlg_DisparityRangeGranularity(kS3dSpeckleMatchAlg alg);

#include <kVision/S3d/kS3dSpeckleMatchAlg.x.h>

#endif  /* KS3D_SPECKLE_MATCH_ALG_H */
