/**
* @file    kS3dVolumeCheckBlockInfo.h
* @brief   Declares the kS3dVolumeCheckBlock class.
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_KS3D_VOLUMECHECKBLOCKINFO_H
#define KVISION_KS3D_VOLUMECHECKBLOCKINFO_H

#include <kApi/kApiDef.h>
#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dVolumeCheckDef.h>

#define kS3D_VOLUMCHECK_BLOCK_NAME                        "VolumeCheck"
#define kS3D_VOLUMCHECK_BLOCK_TYPE_NAME                   "VolumeCheck"

#define kS3D_VOLUMCHECK_BLOCK_IN_SURFACE_NAME             "InSurface"
#define kS3D_VOLUMCHECK_BLOCK_IN_ADDITIONAL_NAME          "InAdditional"

#define kS3D_VOLUMCHECK_BLOCK_OUTPUT_VOLUME_NAME          "OutputVolume"
#define kS3D_VOLUMCHECK_BLOCK_OUTPUT_OFFSET_X_NAME        "OutputOffsetX"
#define kS3D_VOLUMCHECK_BLOCK_OUTPUT_OFFSET_Y_NAME        "OutputOffsetY"

#define kS3D_VOLUMCHECK_BLOCK_OUTPLOT_NAME                "OutPlot"
#define kS3D_VOLUMCHECK_BLOCK_OUT_REPORT_NAME             "OutReport"

#define kS3D_VOLUMCHECK_BLOCK_OUTPUT_VALVES_NAME          "OutputValves"

typedef enum kS3dVolumeCheckBlockPort
{
    kS3D_VOLUMCHECK_BLOCK_IN_SURFACE = 0,
    kS3D_VOLUMCHECK_BLOCK_IN_ADDITIONAL = 1,

    kS3D_VOLUMCHECK_BLOCK_OUTPUT_VOLUME = 1000,
    kS3D_VOLUMCHECK_BLOCK_OUTPUT_VALVES = 1001,
    kS3D_VOLUMCHECK_BLOCK_OUTPLOT = 1010,
    kS3D_VOLUMCHECK_BLOCK_OUT_REPORT = 1011

} kS3dVolumeCheckBlockPort;

typedef struct kS3dVolumeCheckBlockVolumeOutput
{
    k64f volume;
    k64f decision;
    kPoint64f offset;
    k64f difference;
    kPoint3d64f center;
    k64f slopeAngle;
    k64f polarAngle;
    k64f rotationAngle;
} kS3dVolumeCheckBlockVolumeOutput;

typedef struct kS3dVolumeCheckBlockValveItem
{
    k64f ringGaps;
    k64f difference;
    kPoint3d64f center;
    k64f slopeAngle;
    k64f polarAngle;
    k64f rotationAngle;

}kS3dVolumeCheckBlockValveItem;

typedef struct kS3dVolumeCheckBlockValvesOutput
{
    k64f valveCount;
    k64f decision;
    kS3dVolumeCheckBlockValveItem items[kS3D_VOLUMECHECK_VALVE_MAX];
} kS3dVolumeCheckBlockValvesOutput;

/**
* @class       kS3dVolumeCheckBlockInfo
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       The block info class for the VolumeChecker application.
*/
typedef kObject kS3dVolumeCheckBlockInfo;


/**
* Constructs a kS3dVolumeCheckBlockInfo object
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   info        Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_Construct(kS3dVolumeCheckBlockInfo* info, kAlloc allocator);

/**
* Constructs a kS3dVolumeCheckBlockInfo object by reading the content from a xml object
*
* @public              @memberof kS3dVolumeCheckAlg
* @param   info        Destination for the constructed object handle.
* @param   xml         kXml object. 
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_Load(kS3dVolumeCheckBlockInfo* info, kXml xml, kAlloc allocator);

/**
* Constructs a kXml object and uses it to store the content of this kS3dVolumeCheckBlockInfo instance.
*
* @public              @memberof kStartActionInfo
* @param   info        Info object.
* @param   xml         Receives constructed kXml object.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_Store(kS3dVolumeCheckBlockInfo info, kXml* xml);

/**
* Reads kS3dVolumeCheckBlockInfo content from a kXml object.
*
* @public              @memberof kStartActionInfo
* @param   info        Info object.
* @param   xml         XML object.
* @param   item        XML item.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_Read(kS3dVolumeCheckBlockInfo info, kXml xml, kXmlItem item);

/**
* Writes kS3dVolumeCheckBlockInfo content to a kXml object.
*
* @public              @memberof kStartActionInfo
* @param   info        Info object.
* @param   xml         XML object.
* @param   item        XML item.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_Write(kS3dVolumeCheckBlockInfo info, kXml xml, kXmlItem item);

/**
*  Get the valve number of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              valve number.
*/
kVsFx(k32u) kS3dVolumeCheckBlockInfo_ValveCount(kS3dVolumeCheckBlockInfo info);

/**
*  Set the valve number of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   valveCount  valve number.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetValveCount(kS3dVolumeCheckBlockInfo info, k32u valveCount);

/**
*  Get the spark number of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              spark number.
*/
kVsFx(k32u) kS3dVolumeCheckBlockInfo_SparkCount(kS3dVolumeCheckBlockInfo info);

/**
*  Set the spark number of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   sparkCount  spark number.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetSparkCount(kS3dVolumeCheckBlockInfo info, k32u sparkCount);

/**
*  Get the volume offset value of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Volume offset value.
*/
kVsFx(k64f) kS3dVolumeCheckBlockInfo_VolumeOffset(kS3dVolumeCheckBlockInfo info);

/**
*  Set the volume offset value of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   offset      Volume offset value.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetVolumeOffset(kS3dVolumeCheckBlockInfo info, k64f offset);


/**
*  Get the nominal volume value of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Nominal volume value.
*/
kVsFx(k64f) kS3dVolumeCheckBlockInfo_NominalVolume(kS3dVolumeCheckBlockInfo info);


/**
*  Set the nominal volume value of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   volume      Nominal volume value.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetNominalVolume(kS3dVolumeCheckBlockInfo info, k64f volume);

/**
*  Get the volume tolerance of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Volume tolerance.
*/
kVsFx(k64f) kS3dVolumeCheckBlockInfo_VolumeTolerance(kS3dVolumeCheckBlockInfo info);

/**
*  Set the volume tolerance value of the combustion chamber
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   tol         Volume tolerance.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetVolumeTolerance(kS3dVolumeCheckBlockInfo info, k64f tol);

/**
*  Get the increment of searching of valve center in mm. default value: 0.01 mm
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Increment.
*/
kVsFx(k64f) kS3dVolumeCheckBlockInfo_ValveSearchStep(kS3dVolumeCheckBlockInfo info);

/**
*  Set the increment of searching of valve center in mm. default value: 0.01 mm
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   step        Increment.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetValveSearchStep(kS3dVolumeCheckBlockInfo info, k64f step);

/**
*  Get magnification factor relative to nominal diameter to determine the search window.
*  The value depends on the positioning accuracy of the combustion chamber measurement. default value: 1.2
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Magnification factor.
*/
kVsFx(k64f) kS3dVolumeCheckBlockInfo_ValveRadiusFactor(kS3dVolumeCheckBlockInfo info);

/**
*  Set magnification factor relative to nominal diameter to determine the search window.
*  The value depends on the positioning accuracy of the combustion chamber measurement. default value: 1.4
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   factor      magnification factor.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetValveRadiusFactor(kS3dVolumeCheckBlockInfo info, k64f factor);

/**
*  Get center of valve seat ring in mm.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              center of valve seat ring.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckBlockInfo_ValveCenter(kS3dVolumeCheckBlockInfo info, k32u valveIndex);

/**
*  Set center of valve seat ring in mm.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   center      center of valve seat ring.
* @param   valveIndex  number of the valve.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetValveCenter(kS3dVolumeCheckBlockInfo info, kPoint3d64f center, k32u valveIndex);

/**
*  Get rotation angle of valve seat ring in degree.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   valveIndex  number of valve.
* @return              rotation angle.
*/
kVsFx(k64f)  kS3dVolumeCheckBlockInfo_ValvePolarAngle(kS3dVolumeCheckBlockInfo info, k32u valveIndex);

/**
*  Set rotation angle of valve seat ring in degree.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   polarAngle  rotation angle.
* @param   valveIndex  number of the valve.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetValvePolarAngle(kS3dVolumeCheckBlockInfo info, k64f polarAngle, k32u valveIndex);

/**
*  Get slope angle of valve seat ring in degree.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   valveIndex  number of valve.
* @return              slope angle.
*/
kVsFx(k64f)  kS3dVolumeCheckBlockInfo_ValveSlopeAngle(kS3dVolumeCheckBlockInfo info, k32u valveIndex);

/**
*  Set slope angle of valve seat ring in degree.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   slopeAngle  slope angle.
* @param   valveIndex  number of the valve.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetValveSlopeAngle(kS3dVolumeCheckBlockInfo info, k64f slopeAngle, k32u valveIndex);

/**
*  Get nominal diameter of valve seat ring in mm.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   valveIndex  number of valve.
* @return              nominal diameter.
*/
kVsFx(k64f)  kS3dVolumeCheckBlockInfo_ValveDiameter(kS3dVolumeCheckBlockInfo info, k32u valveIndex);

/**
*  Set nominal diameter of valve seat ring in mm.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   diameter    nominal diameter.
* @param   valveIndex  number of the valve.
* @return              Operation status.
*/
kVsFx(kStatus)  kS3dVolumeCheckBlockInfo_SetValveDiameter(kS3dVolumeCheckBlockInfo info, k64f diameter, k32u valveIndex);

// 
/**
*  Get Z search range in mm, default 2.5 mm
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   valveIndex  number of valve.
* @return              Z search range.
*/
kVsFx(k64f)  kS3dVolumeCheckBlockInfo_ValveZRange(kS3dVolumeCheckBlockInfo info, k32u valveIndex);

/**
*  Set Z search range in mm, default 2.5 mm
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   zRange      Z search range.
* @param   valveIndex  number of the valve.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetValveZRange(kS3dVolumeCheckBlockInfo info, k64f zRange, k32u valveIndex);

// 
/**
*  Get phase thick of the seat ring in mm, default kS3D_VOLUMECHECK_DEFAUT_SEATRING_THICK mm
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   valveIndex  number of valve.
* @return              phase thick of the seat ring.
*/
kVsFx(k64f)  kS3dVolumeCheckBlockInfo_RingPhaseThick(kS3dVolumeCheckBlockInfo info, k32u valveIndex);

/**
*  Set phase thick of the seat ring in mm, default kS3D_VOLUMECHECK_DEFAUT_SEATRING_THICK mm
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   thick       phase thick of the seat ring.
* @param   valveIndex  number of the valve.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetRingPhaseThick(kS3dVolumeCheckBlockInfo info, k64f thick, k32u valveIndex);

/**
*  Get software cover mode of the valve.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   valveIndex  number of valve.
* @return              is provided?
*/
kVsFx(kS3dVolumeCheckValveSoftwareCoverMode) kS3dVolumeCheckBlockInfo_ValveSoftCoverMode(kS3dVolumeCheckBlockInfo info, k32u valveIndex);

/**
*  Set software cover mode of the valve.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   softCover   Software cover mode.
* @param   valveIndex  number of the valve.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetValveSoftCoverMode(kS3dVolumeCheckBlockInfo info, kS3dVolumeCheckValveSoftwareCoverMode softCover, k32u valveIndex);

/**
*  Get center of the spark in mm.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   sparkIndex  number of the spark.
* @return              center of the spark.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckBlockInfo_SparkCenter(kS3dVolumeCheckBlockInfo info, k32u sparkIndex);

/**
*  Set center of the spark in mm.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   center      center of the spark.
* @param   sparkIndex  number of spark.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetSparkCenter(kS3dVolumeCheckBlockInfo info, kPoint3d64f center, k32u sparkIndex);

/**
*  Get rotation angle of the spark in degree.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   sparkIndex  number of spark.
* @return              rotation angle.
*/
kVsFx(k64f)  kS3dVolumeCheckBlockInfo_SparkPolarAngle(kS3dVolumeCheckBlockInfo info, k32u sparkIndex);

/**
*  Set rotation angle of the spark in degree.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   polarAngle  rotation angle.
* @param   sparkIndex  number of spark.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetSparkPolarAngle(kS3dVolumeCheckBlockInfo info, k64f polarAngle, k32u sparkIndex);

/**
*  Get slope angle of the spark in degree.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   sparkIndex  number of spark.
* @return              slope angle.
*/
kVsFx(k64f)  kS3dVolumeCheckBlockInfo_SparkSlopeAngle(kS3dVolumeCheckBlockInfo info, k32u sparkIndex);

/**
*  Set slope angle of the spark in degree.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   slopeAngle  slope angle.
* @param   sparkIndex  number of spark.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetSparkSlopeAngle(kS3dVolumeCheckBlockInfo info, k64f slopeAngle, k32u sparkIndex);

/**
*  Get nominal diameter of the spark in mm.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   sparkIndex  number of spark.
* @return              nominal diameter.
*/
kVsFx(k64f)  kS3dVolumeCheckBlockInfo_SparkDiameter(kS3dVolumeCheckBlockInfo info, k32u sparkIndex);

/**
*  Set nominal diameter of the spark in mm.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   diameter    nominal diameter.
* @param   sparkIndex  number of spark.
* @return              Operation status.
*/
kVsFx(kStatus)  kS3dVolumeCheckBlockInfo_SetSparkDiameter(kS3dVolumeCheckBlockInfo info, k64f diameter, k32u sparkIndex);

/**
*  Queries whether software cover for the spark is provided.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   sparkIndex  number of spark.
* @return              is provided?
*/
kVsFx(kBool) kS3dVolumeCheckBlockInfo_SparkSoftCover(kS3dVolumeCheckBlockInfo info, k32u sparkIndex);

/**
*  Set a boolean value to determine of the software cover for the spark.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   softCover   Set value.
* @param   sparkIndex  number of spark.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetSparkSoftCover(kS3dVolumeCheckBlockInfo info, kBool softCover, k32u sparkIndex);

/**
*  Get offset values between the defined coordinate origin and calculated roughly chamber center in mm.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Lateral offset value.
*/
kVsFx(kPoint64f) kS3dVolumeCheckBlockInfo_ChamberCenterOffset(kS3dVolumeCheckBlockInfo info);

/**
*  Set offset values between the defined coordinate origin and calculated roughly chamber center in mm.
*  (center of valid measuring surface area).
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   center      Lateral offset value of kPoint64f.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetChamberCenterOffset(kS3dVolumeCheckBlockInfo info, kPoint64f center);

/**
* Set maximum iterative steps for the surface alignment.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   repeat      Maximum iterative steps
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetMaxAlignRepeat(kS3dVolumeCheckBlockInfo info, k32u repeat);

/**
* Get maximum iterative steps for the surface alignment.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              maximum iterative steps.
*/
kVsFx(k32u) kS3dVolumeCheckBlockInfo_MaxAlignRepeat(kS3dVolumeCheckBlockInfo info);

/**
* Set maximum iterative steps for the surface 2D rotation.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   repeat      maximum iterative steps
* @return              Operation status.
*/

kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetMaxRotationRepeat(kS3dVolumeCheckBlockInfo info, k32u repeat);

/**
* Get maximum iterative steps for the surface 2D rotation.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              maximum iterative steps.
*/
kVsFx(k32u) kS3dVolumeCheckBlockInfo_MaxRotationRepeat(kS3dVolumeCheckBlockInfo info);

/**
* Set the tolerance limit for the surface 2D rotation to stop the iterative operation.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   tolerance   Tolerance limit for the surface 2D rotation.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetRotationTolerance(kS3dVolumeCheckBlockInfo info, k64f tolerance);

/**
* Get the tolerance limit for the surface 2D rotation to stop the iterative operation.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Tolerance limit for the surface 2D rotation.
*/
kVsFx(k64f) kS3dVolumeCheckBlockInfo_RotationTolerance(kS3dVolumeCheckBlockInfo info);

/**
* Set the tolerance limit for the surface 2D translation to stop the iterative operation.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   tolerance   Tolerance limit for the surface 2D translation.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetMoveTolerance(kS3dVolumeCheckBlockInfo info, k64f tolerance);

/**
* Get the tolerance limit for the surface 2D translation to stop the iterative operation.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Tolerance limit for the surface 2D translation.
*/
kVsFx(k64f) kS3dVolumeCheckBlockInfo_MoveTolerance(kS3dVolumeCheckBlockInfo info);

/**
* Set window size for the smoothing border contour.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   window      Window size for the smoothing border contour.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetContourAverageWindow(kS3dVolumeCheckBlockInfo info, k32s window);

/**
* Get window size for the smoothing border contour.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Window size for the smoothing border contour.
*/
kVsFx(k32s) kS3dVolumeCheckBlockInfo_ContourAverageWindow(kS3dVolumeCheckBlockInfo info);

/**
*  Set a boolean value to determine of the outlier filtering operation.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   used        Set value.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetOutlierFilterUsed(kS3dVolumeCheckBlockInfo info, kBool used);

/**
*  Queries whether outlier filtering is used.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              is used?
*/
kVsFx(kBool) kS3dVolumeCheckBlockInfo_OutlierFilterUsed(kS3dVolumeCheckBlockInfo info);

/**
* Set window size for outlier filtering on the surface.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   window      Window size for Outlier filtering.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetOutlierFilterWindow(kS3dVolumeCheckBlockInfo info, k32u window);

/**
* Get window size for outlier filtering on the surface.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Window size for Outlier filtering.
*/
kVsFx(k32u) kS3dVolumeCheckBlockInfo_OutlierFilterWindow(kS3dVolumeCheckBlockInfo info);

/**
* Set repeat count for the outlier filtering on the surface.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   repeat      Repeat count for the outlier filtering
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetOutlierFilterRepeat(kS3dVolumeCheckBlockInfo info, k32u repeat);

/**
* Get repeat count for the outlier filtering on the surface.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Repeat count for the outlier filtering.
*/
kVsFx(k32u) kS3dVolumeCheckBlockInfo_OutlierFilterRepeat(kS3dVolumeCheckBlockInfo info);

/**
* Set tolerance value to identify the outliers. Points within the tolerance range are not affected.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   tolerance   Tolerance value to identify the outliers
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetOutlierFilterTolerance(kS3dVolumeCheckBlockInfo info, k64f tolerance);

/**
* Get tolerance value to identify the outliers. Points within the tolerance range are not affected.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Tolerance value to identify the outliers.
*/
kVsFx(k64f) kS3dVolumeCheckBlockInfo_OutlierFilterTolerance(kS3dVolumeCheckBlockInfo info);

/**
*  Set a boolean value to determine of the smooth filtering operation.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   used        Set value.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetSmoothFilterUsed(kS3dVolumeCheckBlockInfo info, kBool used);

/**
*  Queries whether smooth filtering is used.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              is used?
*/
kVsFx(kBool) kS3dVolumeCheckBlockInfo_SmoothFilterUsed(kS3dVolumeCheckBlockInfo info);

/**
* Set window size for smooth filtering on the surface.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   window      Window size for smooth filtering.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetSmoothFilterWindow(kS3dVolumeCheckBlockInfo info, k32u window);

/**
* Get window size for smooth filtering on the surface.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Window size for smooth filtering.
*/
kVsFx(k32u) kS3dVolumeCheckBlockInfo_SmoothFilterWindow(kS3dVolumeCheckBlockInfo info);

/**
* Set repeat count for the smooth filtering on the surface.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   repeat      Repeat count for the smooth filtering
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetSmoothFilterRepeat(kS3dVolumeCheckBlockInfo info, k32u repeat);

/**
* Get repeat count for the smooth filtering on the surface.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Repeat count for the smooth filtering.
*/
kVsFx(k32u) kS3dVolumeCheckBlockInfo_SmoothFilterRepeat(kS3dVolumeCheckBlockInfo info);

/**
* Set a correction factor to correct the calculated volume value when the sensor is found a systematic deviation,
* for example, by measuring a master-body. Normally the factor should be 1.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   factor      correction factor
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetVolumeCorrectFactor(kS3dVolumeCheckBlockInfo info, k64f factor);

/**
* Get the correction factor to correct the calculated volume value when the sensor is found a systematic deviation.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Correction factor.
*/
kVsFx(k64f) kS3dVolumeCheckBlockInfo_VolumeCorrectFactor(kS3dVolumeCheckBlockInfo info);

/**
* Set the width of the specified output height map kArray2 < k32s >.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   sizeX       Width of height map
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetOutputSizeX(kS3dVolumeCheckBlockInfo info, k32u sizeX);

/**
* Get the width of the specified output height map kArray2 < k32s >.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Width of height map
*/
kVsFx(k32u) kS3dVolumeCheckBlockInfo_OutputSizeX(kS3dVolumeCheckBlockInfo info);

/**
* Set the height of the specified output height map kArray2 < k32s >.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   sizeY       Height of height map
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetOutputSizeY(kS3dVolumeCheckBlockInfo info, k32u sizeY);

/**
* Get the height of the specified output height map kArray2 < k32s >.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Height of height map
*/
kVsFx(k32u) kS3dVolumeCheckBlockInfo_OutputSizeY(kS3dVolumeCheckBlockInfo info);

/**
* Set scaling of the specified output height map kArray2 < k32s >. It is always scale.x == scale.y
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   scale       Scaling of the height map
* @return              Operation status.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckBlockInfo_OutputScale(kS3dVolumeCheckBlockInfo info);

/**
* Get scaling of the specified output height map kArray2 < k32s >. It is always scale.x == scale.y
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Scaling of the height map.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetOutputScale(kS3dVolumeCheckBlockInfo info, kPoint3d64f scale);

/**
* Set scaling of the specified input height map or point clouds of kArray2. 
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   scale       Scaling of the pouint cloud or height map
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetInputScale(kS3dVolumeCheckBlockInfo info, kPoint3d64f scale);

/**
* Get scaling of the specified input height map or point clouds of kArray2. 
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Scaling the pouint cloud or height map.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckBlockInfo_InputScale(kS3dVolumeCheckBlockInfo info);


/**
* Set a known or unknow type of the cylinder head from definition in kS3dVolumeCheckDefaultChamberType.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   type        Cylinder head type.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetChamberType(kS3dVolumeCheckBlockInfo info, kS3dVolumeCheckDefaultChamberType type);

/**
* Get a known or unknow type of the cylinder head from definition in kS3dVolumeCheckDefaultChamberType.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Cylinder head type.
*/
kVsFx(kS3dVolumeCheckDefaultChamberType) kS3dVolumeCheckBlockInfo_ChamberType(kS3dVolumeCheckBlockInfo info);

/**
* Set a boolean value to determine whether the detail report should be cleaned after each measurement.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   clearReport Detail report should be cleaned?
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetClearReport(kS3dVolumeCheckBlockInfo info, kBool clearReport);

/**
* Set a boolean value to determine whether the contents of plane alignment should be displayed in the detail report.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   planeInfo   Diaplays contents of plane alignment?
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetPlaneInfo(kS3dVolumeCheckBlockInfo info, kBool planeInfo);

/**
* Set a boolean value to determine whether the contents of center offset should be displayed in the detail report.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   offsetInfo  Diaplays center offset?
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetOffsetInfo(kS3dVolumeCheckBlockInfo info, kBool offsetInfo);

/**
* Set a boolean value to determine whether the transformation information should be displayed in the detail report.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   transInfo   Diaplays transformation coefficents?
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetTransInfo(kS3dVolumeCheckBlockInfo info, kBool transInfo);

/**
* Set a boolean value to determine whether information about each valve should be displayed in the detail report.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   valvesInfo  Diaplays information about each valve?
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetValvesInfo(kS3dVolumeCheckBlockInfo info, kBool valvesInfo);

/**
* Set a boolean value to determine whether information about running time should be displayed in the detail report.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   timeInfo    Diaplays running time?
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetTimeInfo(kS3dVolumeCheckBlockInfo info, kBool timeInfo);

/**
* Set a boolean value to determine whether volume value should be displayed in the detail report.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   volumeInfo  Diaplays volume value?
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetVolumeInfo(kS3dVolumeCheckBlockInfo info, kBool volumeInfo);

/**
* This value could be used, only if the evaluation must be carried out by absolute positioning of the sensor.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   type        Surface frame of reference. 
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetForType(kS3dVolumeCheckBlockInfo info, kS3dVolumeCheckSurfaceFrameOfReferenceType type);

/**
* Queries whether the detail report should be cleaned after each measurement.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              State report should be cleaned?
*/
kVsFx(kBool) kS3dVolumeCheckBlockInfo_ClearReport(kS3dVolumeCheckBlockInfo info);

/**
* Queries whether the contents of plane alignment should be displayed in the detail report.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Diaplays contents of plane alignment?
*/
kVsFx(kBool) kS3dVolumeCheckBlockInfo_PlaneInfo(kS3dVolumeCheckBlockInfo info);

/**
* Queries whether the contents of center offset should be displayed in the detail report.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Diaplays center offset?.
*/
kVsFx(kBool) kS3dVolumeCheckBlockInfo_OffsetInfo(kS3dVolumeCheckBlockInfo info);

/**
* Queries whether the transformation information should be displayed in the detail report.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Diaplays transformation coefficents?
*/
kVsFx(kBool) kS3dVolumeCheckBlockInfo_TransInfo(kS3dVolumeCheckBlockInfo info);

/**
* Queries whether information about each valve should be displayed in the detail report.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Diaplays information about each valve?
*/
kVsFx(kBool) kS3dVolumeCheckBlockInfo_ValvesInfo(kS3dVolumeCheckBlockInfo info);

/**
* Queries whether information about running time should be displayed in the detail report.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Diaplays running time?
*/
kVsFx(kBool) kS3dVolumeCheckBlockInfo_TimeInfo(kS3dVolumeCheckBlockInfo info);

/**
* Queries whether volume value should be displayed in the detail report.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Diaplays volume value?
*/
kVsFx(kBool) kS3dVolumeCheckBlockInfo_VolumeInfo(kS3dVolumeCheckBlockInfo info);

/**
* Get the rotation mode of the input data.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Rotation mode of the input data.
*/
kVsFx(kS3dVolumeCheckSensorRotationMode) kS3dVolumeCheckBlockInfo_RotationMode(kS3dVolumeCheckBlockInfo info);

/**
* Set the rotation mode of the input data.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @param   mode        Rotation mode of the input data.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckBlockInfo_SetRotationMode(kS3dVolumeCheckBlockInfo info, kS3dVolumeCheckSensorRotationMode mode);

/**
* This value could be used, only if the evaluation must be carried out by absolute positioning of the sensor.
*
* @public              @memberof kS3dVolumeCheckBlockInfo
* @param   info        kS3dVolumeCheckBlockInfo object.
* @return              Surface frame of reference.
*/
kVsFx(kS3dVolumeCheckSurfaceFrameOfReferenceType) kS3dVolumeCheckBlockInfo_ForType(kS3dVolumeCheckBlockInfo info);

#include <kVision/S3d/kS3dVolumeCheckBlockInfo.x.h>

#endif /* #ifndef KVISION_KS3D_VOLUMECHECKBLOCKINFO_H  */
