/**
* @file    kS3dVolumeCheckValveAlg.h
* @brief   Declares the kS3dVolumeCheckValveAlg class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_KS3D_VOLUMECHECKVALVE_H
#define KVISION_KS3D_VOLUMECHECKVALVE_H

#include <kVision/S3d/kS3dVolumeCheckUtilities.h>
#include <kVision/S3d/kS3dVolumeCheckAlign.h>
#include <kVision/S3d/kS3dVolumeCheckSubPixLut.h>
#include <kApi/Data/kImage.h>

#define kS3DVOLUMECHECK_VALVE_PHASE_ANGLE_MAX (60)
#define kS3DVOLUMECHECK_VALVE_PHASE_ANGLE_NOMINAL (45)
#define kS3DVOLUMECHECK_VALVE_PHASE_ANGLE_MIN (30)

#define kS3DVOLUMECHECK_VALVE_FIT_SEATRING_REPEAT (4)

/**
* @class       kS3dVolumeCheckValveAlg
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Class for software cover of the valve.
*/
typedef kObject kS3dVolumeCheckValveAlg;

/**
* Constructs a kS3dVolumeCheckValveAlg object
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       Destination for the constructed object handle.
* @param   subPixLut   Lookup table to speed up the resampling of height image with small 3D transformation. 
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckValveAlg_Construct(kS3dVolumeCheckValveAlg* valve, kS3dVolumeCheckSubPixLut subPixLut, kAlloc allocator);

/**
* Executes the algorithm for software cover of the valve.

* 1. The measurement data will be tilted, aligned to the sealant surface according to the known angles and center.
* 2. The valve hole will be checked.
* 3. The valve-ring profile will be reconstructed along the Z-axis, taking into account of the circular contour-course for each level.
*    It is known the central portion of the valve seat ring is conical and under 45 deg angle to the valve axis.
* 4. The valve hole will be covered with the defined norminal diameter.
*
* @public              @memberof S3dVolumeCheckValveAlg
* @param   valve       S3dVolumeCheckValveAlg object
* @param   align       kS3dVolumeCheckAlign object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckValveAlg_Run(kS3dVolumeCheckValveAlg valve, kS3dVolumeCheckAlign align);

/**
* Start setup process.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckValveAlg_Setup(kS3dVolumeCheckValveAlg valve);

/**
* Ensures the setup process.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @param   align       kS3dVolumeCheckAlign object.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckValveAlg_UpdateSetup(kS3dVolumeCheckValveAlg valve, kS3dVolumeCheckAlign align);

/**
*  Get center of valve seat ring in mm.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @return              center of valve seat ring.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckValveAlg_ValveCenter(kS3dVolumeCheckValveAlg valve);

/**
*  Set center of valve seat ring in mm.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @param   center      center of valve seat ring.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckValveAlg_SetValveCenter(kS3dVolumeCheckValveAlg valve, kPoint3d64f center);


/**
*  Get actual meassured center of valve seat ring in mm.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @return              actual meassured center of valve seat ring.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckValveAlg_ActualValveCenter(kS3dVolumeCheckValveAlg valve);
/**
*  Get rotation angle of valve seat ring in degree.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @return              rotation angle.
*/
kVsFx(k64f)  kS3dVolumeCheckValveAlg_ValvePolarAngle(kS3dVolumeCheckValveAlg valve);

/**
*  Set rotation angle of valve seat ring in degree.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @param   polarAngle  rotation angle.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckValveAlg_SetValvePolarAngle(kS3dVolumeCheckValveAlg valve, k64f polarAngle);

/**
*  Get slope angle of valve seat ring in degree.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @return              slope angle.
*/
kVsFx(k64f)  kS3dVolumeCheckValveAlg_ValveSlopeAngle(kS3dVolumeCheckValveAlg valve);

/**
*  Set slope angle of valve seat ring in degree.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @param   slopeAngle  slope angle.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckValveAlg_SetValveSlopeAngle(kS3dVolumeCheckValveAlg valve, k64f slopeAngle);

/**
*  Get nominal diameter of valve seat ring in mm.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @return              nominal diameter.
*/
kVsFx(k64f)  kS3dVolumeCheckValveAlg_ValveDiameter(kS3dVolumeCheckValveAlg valve);

/**
*  Set nominal diameter of valve seat ring in mm.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @param   diameter    nominal diameter.
* @return              Operation status.
*/
kVsFx(kStatus)  kS3dVolumeCheckValveAlg_SetValveDiameter(kS3dVolumeCheckValveAlg valve, k64f diameter);

// 
/**
*  Get Z search range in mm.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @return              Z search range.
*/
kVsFx(k64f)  kS3dVolumeCheckValveAlg_ValveZRange(kS3dVolumeCheckValveAlg valve);

/**
*  Set Z search range in mm
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @param   zRange      Z search range.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckValveAlg_SetValveZRange(kS3dVolumeCheckValveAlg valve, k64f zRange);

/**
*  Get phase thick of the seat ring in mm, default kS3D_VOLUMECHECK_DEFAUT_SEATRING_THICK mm
*
* @public                 @memberof kS3dVolumeCheckValveAlg
* @param   valve          kS3dVolumeCheckValveAlg object.
* @return                 phase thick of the seat ring.
*/
kVsFx(k64f)  kS3dVolumeCheckValveAlg_RingPhaseThick(kS3dVolumeCheckValveAlg valve);

/**
*  Set phase thick of the seat ring in mm, default kS3D_VOLUMECHECK_DEFAUT_SEATRING_THICK mm
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @param   thick       Phase thick of the seat ring.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckValveAlg_SetRingPhaseThick(kS3dVolumeCheckValveAlg valve, k64f thick);

/**
* Queries whether software cover for the valve is provided.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @return              Software cover mode.
*/
kVsFx(kS3dVolumeCheckValveSoftwareCoverMode) kS3dVolumeCheckValveAlg_ValveSoftCoverMode(kS3dVolumeCheckValveAlg valve);

/**
*  Set a boolean value to determine of the software cover for the valve.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @param   softCover   Set value.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckValveAlg_SetValveSoftCoverMode(kS3dVolumeCheckValveAlg valve, kS3dVolumeCheckValveSoftwareCoverMode softCover);

/**
* Get the increment for ring seat profile in mm.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @return              increment for ring seat profile.
*/
kVsFx(k64f) kS3dVolumeCheckValveAlg_SearchStep(kS3dVolumeCheckValveAlg valve);

/**
*  Set the increment for valve seat ring profile in mm. The default value is 0.01 mm.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @param   searchStep  increment for ring seat profile.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckValveAlg_SetSearchStep(kS3dVolumeCheckValveAlg valve, k64f searchStep);

/**
*Get magnification factor relative to nominal diameter to determine the search window.
*  The value depends on the positioning accuracy of the combustion chamber measurement. default value: 1.4
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @return              magnification factor..
*/

kVsFx(k64f) kS3dVolumeCheckValveAlg_RadiusFactor(kS3dVolumeCheckValveAlg valve);

/**
*  Set magnification factor relative to nominal diameter to determine the search window.
*  The value depends on the positioning accuracy of the combustion chamber measurement. default value: 1.4
*
* @public                @memberof kS3dVolumeCheckValveAlg
* @param   valve         kS3dVolumeCheckValveAlg object.
* @param   radiusFactor  magnification factor.
* @return                Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckValveAlg_SetRadiusFactor(kS3dVolumeCheckValveAlg valve, k64f radiusFactor);


/**
* Queries whether valve hole exists.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @return              does valve hole exist?
*/
kVsFx(kBool) kS3dVolumeCheckValveAlg_ValveHoleExists(kS3dVolumeCheckValveAlg valve);

/**
*  Amalgamation of the real seat ring surface and the reconstructed surface. 
*
* @public                @memberof kS3dVolumeCheckValveAlg
* @param   valve         kS3dVolumeCheckValveAlg object.
* @param   align         kS3dVolumeCheckAlign object that contains the height map and the information about the transformation.
* @return                Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckValveAlg_MergeSeatRingMap(kS3dVolumeCheckValveAlg valve, kS3dVolumeCheckAlign align);

/**
*  Fill the valve hole with the specified parameter and the fitting ring.
*
* @public                @memberof kS3dVolumeCheckValveAlg
* @param   valve         kS3dVolumeCheckValveAlg object.
* @param   output        Output height map of kArray2< k32s >.
* @param   weight        Weighting map of kArray2< k32s >.
* @param   valid         Valid map of kArray2 < k8u >. The surface is segmented into different parts.
* @param   xyScale       Lateral scaling of the output height map.
* @return                Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckValveAlg_AddValveArea(kS3dVolumeCheckValveAlg valve, kArray2 output, kArray2 weight, kArray2 valid, k64f xyScale);

/**
*  Get the bit value for the segmentation in order to distinguish the different parts..
*
* @public                @memberof kS3dVolumeCheckValveAlg
* @param   valve         kS3dVolumeCheckValveAlg object.
* @return                The bit value identifying the valve.
*/
kVsFx(k8u) kS3dVolumeCheckValveAlg_SegmentBit(kS3dVolumeCheckValveAlg valve);

/**
*  Set a bit value for the segmentation in order to distinguish the different parts..
*
* @public                @memberof kS3dVolumeCheckValveAlg
* @param   valve         kS3dVolumeCheckValveAlg object.
* @param   segmentBit    The bit value identifying the valve.
* @return                Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckValveAlg_SetSegmentBit(kS3dVolumeCheckValveAlg valve, k8u segmentBit);


/**
* Get the range rasolution of the point cloud with kArray2 < kPoint3d32s >
*
* @public                @memberof kS3dVolumeCheckValveAlg
* @param   valve         kS3dVolumeCheckValveAlg object.
* @return                Range rasolution.
*/
kVsFx(k64f)  kS3dVolumeCheckValveAlg_RangeScale(kS3dVolumeCheckValveAlg valve);

/**
* Set the range rasolution of the point cloud with kArray2 < kPoint3d32s >
*
* @public                @memberof kS3dVolumeCheckValveAlg
* @param   valve         kS3dVolumeCheckValveAlg object.
* @param   rangeScale    Range rasolution.
* @return                Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckValveAlg_SetRangeScale(kS3dVolumeCheckValveAlg valve, k64f rangeScale);


/**
* Create a height map of valve seat ring for drawing: kArray2 <k64f>. Only for verification
*
* @public                @memberof kS3dVolumeCheckValveAlg
* @param   valve         kS3dVolumeCheckValveAlg object.
* @param   map           Object handle of a height map that will be created.
* @param   isInput       If idInput == kTRUE return the height map from original data else return the fitting height map.
* @param   rect          The rect of the to be displayed range.
* @return                Operation status.
*/
kVsFx(kStatus)  kS3dVolumeCheckValveAlg_ValveSeatRingMap64f(kS3dVolumeCheckValveAlg valve, kImage*  map, kBool isInput, kRect3d64f* rect);

/**
* Return sectional profile of kArrayList <kPoint32f>. Only used for verification
*
* @public                @memberof kS3dVolumeCheckValveAlg
* @param   valve         kS3dVolumeCheckValveAlg object.
* @param   axisNo        axisNo= 0: Profile of x center position;  axisNo= 1: Profile of y center position; axisNo= 2: z / R Profile.
* @return                Sectional profile.
*/
kVsFx(kArrayList) kS3dVolumeCheckValveAlg_SeatRingProfile(kS3dVolumeCheckValveAlg valve, k32s axisNo);

/**
* Creates a sectional profile of the specified valve seat ring. Only used for verification
*
* @public                @memberof kS3dVolumeCheckValveAlg
* @param   valve         kS3dVolumeCheckValveAlg object.
* @return                Operation status.
*/
kVsFx(kStatus)  kS3dVolumeCheckValveAlg_SetupSeatRingProfile(kS3dVolumeCheckValveAlg valve);

/**
* Set the current number of measurement. Only used for internal statistic calculation and for storing files.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @param   index       current number of measurement.
* @return              Operation status.
*/
kVsFx(kStatus)  kS3dVolumeCheckValveAlg_SetMeasureIndex(kS3dVolumeCheckValveAlg valve, k32s index);

/**
* Get the actual z position of the valve seat ring center.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @return              Z position.
*/
kVsFx(k64f) kS3dVolumeCheckValveAlg_ActualZposition(kS3dVolumeCheckValveAlg valve);

/**
*  Get the indicator gaps-area ratio in determining the valve seat ring in %.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @return              Gaps-area ratio of the valve seat ring.
*/
kVsFx(k64f) kS3dVolumeCheckValveAlg_ValveRingGaps(kS3dVolumeCheckValveAlg valve);

/**
*  Get the Indicator uncertainty of the profile of the valve seat ring in mm.
*
* @public              @memberof kS3dVolumeCheckValveAlg
* @param   valve       kS3dVolumeCheckValveAlg object.
* @return              Uncertainty of the profile of the valve seat ring.
*/
kVsFx(k64f)  kS3dVolumeCheckValveAlg_ValveAveDifference(kS3dVolumeCheckValveAlg valve);

#include <kVision/S3d/kS3dVolumeCheckValveAlg.x.h>

#endif /* #ifndef KVISION_KS3D_VOLUMECHECKVALVE_H */  
