/**
* @file    kVsJobQueue.h
* @brief   Declares the kVsJobQueue class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef KVS_JOB_QUEUE_H
#define KVS_JOB_QUEUE_H

#include <kVision/Common/kVision.h>

/**
* @class       kVsJobQueue
* @extends     kObject
* @ingroup     kVision-Vs
* @brief       Implement a minimal job queue with user defined number of worker threads. 
*/
typedef kObject kVsJobQueue;

/**
* Handler for a user defined job. Should accept a user defined context and an 'id' field. The 
* id field is inteded to allow the user to divide processing of similar inputs into uniform parts.
* For example, it can be used to perform processing on data from two cameras (id would then be set to the
* camera index). 
*
* @public              @memberof kVsJobQueue
* @param   context     User defined context passed into kVsJobQueue_Add
* @param   id          User defined worked id. Can be used to provide a common worker implementation 
*                      context on the portion of the work that should be done without modifying the context
* @return              Operation status.
*/
typedef kStatus(kCall *kVsJobQueueEntry)(kPointer context, k64u id);

/**
* Constructs a kVsJobQueue object
*
* @public              @memberof kVsJobQueue
* @param   queue       Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kVsJobQueue_Construct(kVsJobQueue* queue, kSize threadCount, kAlloc alloc);

/**
* Adds an entry to the job queue. 
*
* @public              @memberof kVsJobQueue
* @param   queue       Destination for the constructed object handle.
* @param   entry       Pointer to the function executing the task.
* @param   context     Context to be passed into the entry function when it is called
* @param   id          Id to be passed into the entry function when it is called
* @return              Operation status.
*/
kVsFx(kStatus) kVsJobQueue_Add(kVsJobQueue queue, kVsJobQueueEntry entry, kPointer context, k64u id);

/**
* Get the number of tasks that are queued or in progress. 
*
* @public              @memberof kVsJobQueue
* @param   queue       Destination for the constructed object handle.
* @return              Number of tasks that are queued or in progress
*/

kVsFx(kSize) kVsJobQueue_Count(kVsJobQueue queue);

/**
* Executes all queued jobs in a blocking fashion. The function returns when all queued jobs have completed (successfully or not)
* If one or more jobs fails (returns an error) the first reported error code is returned by this function. 
*
* @public              @memberof kVsJobQueue
* @param   queue       Destination for the constructed object handle.
* @return              Operation status (kOK or the first error code raised by any of the executed jobs).
*/
kVsFx(kStatus) kVsJobQueue_Run(kVsJobQueue queue);

/**
* Executes a batch of parallel tasks specified via the handler entry in a blocking fashion. Each job is
* called with the specified context and job index (0..count - 1) as the 'id' parameter
*
* @public              @memberof kVsJobQueue
* @param   queue       Destination for the constructed object handle.
* @param   entry       Pointer to the function executing the task.
* @param   context     Context to be passed into the entry function when it is called
* @param   count       Number of handlers to be executed
* @return              Operation status (kOK or the first error code raised by any of the executed jobs).
*/
kVsFx(kStatus) kVsJobQueue_RunIndexedBatch(kVsJobQueue queue, kVsJobQueueEntry entry, kPointer context, kSize count);

/**
* Executes all queued jobs in a non-blocking fashion. The function requests all outstanding jobs to start execution
* and returns. New jobs can be added after this point. Their exit should be gated by calling kVsJobQueue_End
* which will block until all have exited.
*
* @public              @memberof kVsJobQueue
* @param   queue       Destination for the constructed object handle.
* @return              Operation status (kOK or the first error code raised by any of the executed jobs).
*/
kVsFx(kStatus) kVsJobQueue_Begin(kVsJobQueue queue);

/**
* The function is used to gate asynchronous execution of queued jobs initiated throguh kVsJobQueue_Begin. 
* The function should be called when waiting for the jobs to complete. The function returns after
* all jobs have completed. If one or more jobs fails (returns an error) the first reported error code is returned 
* by this function. 
*
*
* @public              @memberof kVsJobQueue
* @param   queue       Destination for the constructed object handle.
* @return              Operation status (kOK or the first error code raised by any of the executed jobs).
*/
kVsFx(kStatus) kVsJobQueue_End(kVsJobQueue queue);


/**
* Gets the number of created threads
*
*
* @public              @memberof kVsJobQueue
* @param   queue       Destination for the constructed object handle.
* @return              Number of created threads
*/
kVsFx(kSize) kVsJobQueue_ThreadCount(kVsJobQueue queue);

#include <kVision/Vs/kVsJobQueue.x.h>

#endif  /* KVS_JOB_QUEUE_H */
