/**
* @file    kVsLicense.h
* @brief   Declares the kVsLicense class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef K_VISION_LICENSE_H
#define K_VISION_LICENSE_H

#include <kVision/Common/kVision.h>
#include <kVision/L3d/kL3dUtilities.h>
#include <kApi/Io/kFile.h>

/**
* @class        kVsLicense
* @extends      kObject
* @ingroup      kVision-Vs
* @brief        Generates encrypted Gocator software licenses.
*/
typedef kObject kVsLicense;

/**
* Constructs a kVsLicense object.
*
* @public                   @memberof kVsLicense
* @param    license         Destination for the constructed object handle.
* @param    allocator       Memory allocator (or kNULL for default).
* @return                   Operation status.
*/
kVsFx(kStatus) kVsLicense_Construct(kVsLicense *license, kAlloc allocator);

/**
* Reads a license file from sensor storage.
*
* If sensor is set to kNULL, path specifies a location on the host machine.
*
* @public                   @memberof kVsLicense
* @param    license         kVsLicense object.
* @param    sensor          kNode handle for the target sensor (or kNULL to read from file).
* @param    path            License file path.
* @return                   Operation status.
*/
kVsFx(kStatus) kVsLicense_Load(kVsLicense license, kNode sensor, const kChar* path);

/**
* Writes a license file to sensor storage.
*
* If sensor is set to kNULL, path specifies a location on the host machine.
* This function will return kERROR_PARAMETER if SerialNumber or OptionCode has not been set.
*
* @public                   @memberof kVsLicense
* @param    license         kVsLicense object.
* @param    sensor          kNode handle for the target sensor (or kNULL to write to file).
* @param    path            License file path.
* @return                   Operation status.
*/
kVsFx(kStatus) kVsLicense_Save(kVsLicense license, kNode sensor, const kChar* path);


/**
* Reads a license file from a kStream.
*
* The stream should contain only the encrypted license data.
*
* @public                   @memberof kVsLicense
* @param    license         kVsLicense object.
* @param    stream          Input stream handle
* @return                   Operation status.
*/
kVsFx(kStatus) kVsLicense_LoadStream(kVsLicense license, kStream stream);

/**
* Writes a license file to a kStream.
*
* @public                   @memberof kVsLicense
* @param    license         kVsLicense object.
* @param    stream          Output stream handle
* @return                   Operation status.
*/
kVsFx(kStatus) kVsLicense_SaveStream(kVsLicense license, kStream stream);

/**
* Returns the serial number assigned to the current license.
*
* @public                   @memberof kVsLicense
* @param    license         kVsLicense object.
* @return                   Node id of target sensor.
*/
kVsFx(k32u) kVsLicense_SerialNumber(kVsLicense license);

/**
* Sets the serial number for the current license.
*
* @public                   @memberof kVsLicense
* @param    license         kVsLicense object.
* @param    serialNumber    Node id of target sensor.
* @return                   Operation status.
*/
kVsFx(kStatus) kVsLicense_SetSerialNumber(kVsLicense license, k32u serialNumber);

/**
* Returns the option code assigned to the current license.
*
* @public                   @memberof kVsLicense
* @param    license         kVsLicense object.
* @return                   OptionCode of target software package.
*/
kVsFx(k32u) kVsLicense_OptionCode(kVsLicense license);

/**
* Sets the option code for the current license.
*
* @public                   @memberof kVsLicense
* @param    license         kVsLicense object.
* @param    optionCode      OptionCode of target software package.
* @return                   Operation status.
*/
kVsFx(kStatus) kVsLicense_SetOptionCode(kVsLicense license, k32u optionCode);

/**
* Adds a feature to the current license.
*
* @public                   @memberof kVsLicense
* @param    license         kVsLicense object.
* @param    feature         Feature number.
* @return                   Operation status.
*/
kVsFx(kStatus) kVsLicense_AddFeature(kVsLicense license, k32s feature);

/**
* Removes a feature from the current license.
*
* @public                   @memberof kVsLicense
* @param    license         kVsLicense object.
* @param    feature         Feature number.
* @return                   Operation status.
*/
kVsFx(kStatus) kVsLicense_RemoveFeature(kVsLicense license, k32s feature);

/**
* Checks if the current license has a specific feature.
*
* @public                   @memberof kVsLicense
* @param    license         kVsLicense object.
* @param    feature         Feature number.
* @return                   Whether or not the license has the feature.
*/
kVsFx(kBool) kVsLicense_HasFeature(kVsLicense license, k32s feature);

/**
* Returns the number of features in the current license.
*
* @public                   @memberof kVsLicense
* @param    license         kVsLicense object.
* @return                   The number of features.
*/
kVsFx(kSize) kVsLicense_FeatureCount(kVsLicense license);

/**
* Returns the feature at a specific index of the current license.
*
* @public                   @memberof kVsLicense
* @param    license         kVsLicense object.
* @param    index           The feature index.
* @return                   The feature number.
*/
kVsFx(k32s) kVsLicense_FeatureAt(kVsLicense license, kSize index);

#include <kVision/Vs/kVsLicense.x.h>

#endif    /* K_VISION_LICENSE_H */
