/** 
 * @file    kVsPrepareOptions.h
 * @brief   Declares the kVsPrepareOptions class. 
 *
 * @internal
 * Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_VISION_PREPARE_OPTIONS_DEF_H
#define K_VISION_PREPARE_OPTIONS_DEF_H

#include <kVision/Common/kVision.h>
#include <kVision/Vs/kVsModelDef.h>

/**
 * @class       kVsPrepareOptions
 * @extends     kObject
 * @ingroup     kVision-Vs
 * @brief       
 */
typedef kObject kVsPrepareOptions;

/** 
 * Constructs a PrepareOptions object from an XML file. The xml will be parsed and any relevant values extracted during construction.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Destination for the constructed object handle.
 * @param   xml             Filtered xml object
 * @param   allocator       Memory allocator (or kNULL for default).
 * @return                  Operation status.
 */
kVsFx(kStatus) kVsPrepareOptions_Construct(kVsPrepareOptions* options, kXml xml, kMap filterMap, kAlloc allocator);

/** 
 * Returns the option name, or "" if it doesn't exist.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @return                  Option name string.
 */
kVsFx(const kChar*) kVsPrepareOptions_OptionName(kVsPrepareOptions options);

/** 
 * Returns the Upgrade Package path, or "" if it doesn't exist.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @return                  Upgrade Package path string.
 */
kVsFx(const kChar*) kVsPrepareOptions_UpgradePackagePath(kVsPrepareOptions options);

/** 
 * Returns the rescue firmware path, or "" if it doesn't exist.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @return                  Rescue Firmware path string.
 */
kVsFx(const kChar*) kVsPrepareOptions_RescueFirmwarePath(kVsPrepareOptions options);

/** 
 * Returns the firmware path, or "" if it doesn't exist.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @return                  Firmware path string.
 */
kVsFx(const kChar*) kVsPrepareOptions_FirmwarePath(kVsPrepareOptions options);

/** 
 * Returns the OS path, or "" if it doesn't exist.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @return                  OS path string.
 */
kVsFx(const kChar*) kVsPrepareOptions_OsPath(kVsPrepareOptions options);

/** 
 * Returns the rescue OS path, or "" if it doesn't exist.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @return                  Rescue OS path string.
 */
kVsFx(const kChar*) kVsPrepareOptions_RescueOsPath(kVsPrepareOptions options);

/** 
 * Returns the compatible firmware tag, or "" if it doesn't exist.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @return                  Compatible Firmware Tag string.
 */
kVsFx(const kChar*) kVsPrepareOptions_CompatibleFirmwareTag(kVsPrepareOptions options);

/** 
 * Returns the the path to avaliable upgrade packages.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @return                  Path string.
 */
kVsFx(const kChar*) kVsPrepareOptions_UpgradeRootPath(kVsPrepareOptions options);

/** 
 * Returns the version filter for upgrade package. N/A if not specified
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @return                  Version string.
 */
kVsFx(const kChar*) kVsPrepareOptions_UpgradeVersionMask(kVsPrepareOptions options); 

/** 
 * Returns best found version according to version filter.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @return                  Target Version string.
 */
kVsFx(const kChar*) kVsPrepareOptions_UpgradeTargetVersion(kVsPrepareOptions options);

/** 
 * Returns the prefix for upgrade package.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @return                  Prefix string.
 */
kVsFx(const kChar*) kVsPrepareOptions_UpgradePrefix(kVsPrepareOptions options);


/** 
 * Returns the number of config options.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @return                  The number of config options.
 */
kVsFx(kSize) kVsPrepareOptions_ConfigCount(kVsPrepareOptions options);

/** 
 * Returns the config name at a given index.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @param   index           Index of the config. Use ConfigCount to get count.
 * @return                  A config name.
 */
kVsFx(const kChar*) kVsPrepareOptions_ConfigNameAt(kVsPrepareOptions options, kSize index);

/** 
 * Returns the config path at a given index.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @param   index           Index of the config. Use ConfigCount to get count.
 * @return                  A config file path.
 */
kVsFx(const kChar*) kVsPrepareOptions_ConfigPathAt(kVsPrepareOptions options, kSize index);

/** 
 * Returns the number of file options.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @return                  The number of file options.
 */
kVsFx(kSize) kVsPrepareOptions_FileCount(kVsPrepareOptions options);

/** 
 * Returns the file name at a given index.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @param   index           Index of the file. Use FileCount to get count.
 * @return                  A file name.
 */
kVsFx(const kChar*) kVsPrepareOptions_FileNameAt(kVsPrepareOptions options, kSize index);

/** 
 * Returns the file path at a given index.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @param   index           Index of the file. Use FileCount to get count.
 * @return                  A file path.
 */
kVsFx(const kChar*) kVsPrepareOptions_FilePathAt(kVsPrepareOptions options, kSize index);

/** 
 * Returns the config xml index
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @param   index           Index of the config. Must path to a valid file.
 * @param   xml             Destination for the constructed object handle.
 * @return                  Operation status.
 */
kVsFx(kXml) kVsPrepareOptions_ConfigAt(kVsPrepareOptions options, kSize index);

/** 
 * Returns the file path to the Custom Tool List xml file
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   prepareOptions  Object handle.
 * @return                  File path.
 */
kVsFx(const kChar*) kVsPrepareOptions_CustomToolListPath(kVsPrepareOptions options);

/** 
 * Returns the number of license features.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   options         Object handle.
 * @return                  The number of license features.
 */
kVsFx(kSize) kVsPrepareOptions_LicenseFeatureCount(kVsPrepareOptions options);

/** 
 * Returns the license feature at a given index.
 *
 * @public                  @memberof kVsPrepareOptions
 * @param   options         Object handle.
 * @param   index           Index of the license feature.
 * @return                  A license feature.
 */
kVsFx(k32s) kVsPrepareOptions_LicenseFeatureAt(kVsPrepareOptions options, kSize index);

#include <kVision/Vs/kVsPrepareOptions.x.h>

#endif  /* K_VISION_PREPARE_OPTIONS_DEF_H */