/** 
 * @file    kVsXmlConfigParser.h
 * @brief   Declares the OptionList class. 
 *
 * @internal
 * Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_VISION_XML_CONFIG_PARSER_H
#define K_VISION_XML_CONFIG_PARSER_H

#include <kVision/Common/kVision.h>
#include <kVision/Vs/kVsModelDef.h>
#include <kFireSync/kNodeDef.h>

/**
 * @class       kVsXmlConfigParser
 * @extends     kObject
 * @ingroup     kVision-Vs
 * @brief       Loads, stores, and filters optionlist xml files.
 */
typedef kObject kVsXmlConfigParser;

/** 
 * Constructs an OptionList object from an XML file.
 *
 * This function will return kERROR_PARAMETER if the version attribute in the xml doesn't match the version defined in 
 * kVsXmlConfigParser.x.h
 *
 * @public              @memberof kVsXmlConfigParser
 * @param   optionList  Destination for the constructed object handle.
 * @param   path        Path to the local OptionList XML file.
 * @param   allocator   Memory allocator (or kNULL for default).
 * @return              Operation status. 
 */
kVsFx(kStatus) kVsXmlConfigParser_Construct(kVsXmlConfigParser* optionList, const kChar* path, kAlloc allocator);

/**
* Provided with a model definition - and optionally a task and nodeInfo (for controller type) - filter 
* the optionlist against those parameters. 
*
* Produces a new, filtered xml based on the parameters. This xml can be passed to kVsFactoryOptions or kVsPrepareOptions for easy use.
* When parsing xml, items with same tag name but different attribute names or value will be consider as unique. Order of attributes does not matter.
*
* @public               @memberof kVsXmlConfigParser
* @param    optionList  kVsXmlConfigParser object.
* @param    model       Target model definition as a kVsModelDef object.
* @param    task        Defined operation task (e.g. kVS_OPTION_LIST_TASK_INIT).
* @param    nodeInfo    Node info pointer (used for adding ControllerType to the filter, ie. M1,M2,M50)
* @param    xml         Handle for constructed filtered xml object.
* @return               Operation status.
*/
kVsFx(kStatus) kVsXmlConfigParser_ApplyFilter(kVsXmlConfigParser optionList, kMap filter, kXml *xml);


kVsFx(kStatus) kVsXmlConfigParser_AddToFilter(kMap filter, const kChar* key, const kChar* value);

#include <kVision/Vs/kVsXmlConfigParser.x.h>

#endif  /* K_VISION_XML_CONFIG_PARSER_H */
