var Express = require('express');
var BodyParser = require('body-parser');
var Busboy = require('connect-busboy');
var HttpStatus = require('http-status-codes');
var kStatus = require('../engine/kStatus.js');
var tmp = require('tmp');
var fs = require('fs');
var promise = require('bluebird');
var path = require('path');

var constructRouter = function(engine) {
    var router = Express.Router();
    var jsonParser = BodyParser.json();

    router.use(Busboy());

    router.get('/', function(req, res) {
        var records = [];
        var profiles = engine.repository.list();
        for (var i = 0; i < profiles.length; i++) {
            records.push(profileToRecord(profiles[i]));
        }
        res.json(records);
    });

    //TODO: improve temp file writing
    //ONLY ENABLES IMPORT/DELETE requests on desktop (ie: not the server)
    if (!engine._config.isServer){
        
        router.post('/', function(req, res) {
            
            if (!req.busboy) {
                res.status(HttpStatus.INTERNAL_SERVER_ERROR).type('text/plain').send({
                    status: kStatus.ERROR_PARAMETER
                });
                return;
            }

            var scenarioFileName = null;
            var scenarioFileOp = null;

            req.busboy.on('file', (fieldname, file, filename, encoding) => {
                if (fieldname == 'data') {
                    scenarioFileName = filename;
                    scenarioFileOp = writeTempFile(file);
                }
            });

            var savedFilePath = null;

            req.busboy.on('finish', () => {
                if (!scenarioFileName) {
                    res.status(HttpStatus.INTERNAL_SERVER_ERROR).type('text/plain').send({status: kStatus.ERROR_PARAMETER});
                    return;
                }

                scenarioFileOp.then((filePath) => {
                    savedFilePath = filePath;

                    var scenarioName = findAvailableScenarioName(engine, path.basename(scenarioFileName, '.gs'));

                    console.log("Importing support file <" + scenarioFileName + ">");
                    engine.repository.import(filePath, scenarioName);

                    //res.type('text/plain').send({status: kStatus.OK});//crummy workaround cause the embedded internet explorer is terrible.
                    //res.json({status: kStatus.OK});
                    var profiles = engine.repository.list();
                    var postedProfileFound = false;
                    for (var i = 0; i < profiles.length; i++) {
                        profileToRecord(profiles[i]);
                        if (profiles[i].name == scenarioName) {
                            res.type('text/plain').send({status: kStatus.OK, id: profiles[i].id});
                            postedProfileFound = true;
                            break;
                        }
                    }

                    // reach here means we didn't find the profile
                    if (!postedProfileFound) {
                        res.type('text/plain').send({status: kStatus.ERROR_PARAMETER});
                    }
                }).caught((err) => {
                    console.log("Failed to import support file <" + scenarioFileName + ">. File may be corrupted.");
                    console.log(err);
                    var status = ('status' in err) ? err.status : kStatus.ERROR;

                    res.status(HttpStatus.INTERNAL_SERVER_ERROR).type('text/plain').send({status: status});
                }).finally(() => {
                    if (savedFilePath) {
                        try {
                            os.unlink(savedFilePath);
                        } catch (err) {

                        }
                    }
                });
            });

            req.pipe(req.busboy);
        });

        router.delete('/:id', function (req, res) {
            try {
                engine.repository.delete(decodeURIComponent(req.params.id));
                res.type('text/plain').send({status: kStatus.OK});
            } catch (err) {
                res.status(HttpStatus.INTERNAL_SERVER_ERROR).type('text/plain').send({
                    status: err.status
                    });
                }
            });
        }
        else 
        {//Disable on these requests on server.
            router.post('/', function(req, res) {
                res.status(HttpStatus.METHOD_NOT_ALLOWED).type('text/plain').send({
                    status: kStatus.ERROR
                });
                return;
            });
               
            router.delete('/:id', function (req, res) {
                    res.status(HttpStatus.METHOD_NOT_ALLOWED).type('text/plain').send({
                    status: kStatus.ERROR
                     });
                    return;
            });
        }
    
    router.get('/:id', function(req, res) {
        var profile = engine.repository.find(decodeURIComponent(req.params.id));

        if (profile != null) {
            res.json(profileToRecord(profile));
        } else {
            res.status(HttpStatus.NOT_FOUND).type('text/plain').send({
                status: kStatus.ERROR_NOT_FOUND
            });
        }
    });

    router.put('/:id', jsonParser, function(req, res) {
        var profile = engine.repository.find(decodeURIComponent(req.params.id));

        if (profile == null) {
            res.status(HttpStatus.NOT_FOUND).type('text/plain').send({
                status: kStatus.ERROR_NOT_FOUND
            });
            return;
        }

        var kName = "name";
        var kDescription = "description";
        
        // Check for description changes first, because changing the name inherently changes the id.
        // This causes the reference to the old 'name' to be lost when trying to modify the description
        // Modifying the description first should solve this problem, as the name is independant of the description but not vice versa
        if ('description' in req.body) {
            try {
                engine.repository.updateInfo(decodeURIComponent(req.params.id), kDescription, req.body.description);
                res.type('text/plain').send({status: kStatus.OK});
            } catch (err) {
                res.status(HttpStatus.INTERNAL_SERVER_ERROR).type('text/plain').send({
                    status: err.status
                });
            }
        }
        
        if ('name' in req.body) {
            try {
                engine.repository.updateInfo(decodeURIComponent(req.params.id), kName, req.body.name);
                res.type('text/plain').send({status: kStatus.OK});
            } catch (err) {
                res.status(HttpStatus.INTERNAL_SERVER_ERROR).type('text/plain').send({
                    status: err.status
                });
            }
        }
    });

    return router;
};

function profileToRecord(profile) {
    return {
        id: profile.id,
        name: profile.name,
        model: profile.model,
        family: profile.family,
        series: profile.series,
        version: profile.version,
        description: profile.description,
        hasBuddy: profile.hasBuddy,
        modifiable: profile.modifiable,
        modelDisplayName: profile.modelDisplayName
    };
};

function writeTempFile(inStream) {
    var deferred = promise.pending();

    tmp.file(function(err, path, fd, cleanupCallback) {
        if (err) {
            deferred.reject(err);
        } else {
            // TODO: this is to work around a issue with opening the file in the addon.
            // Should look into better ways to create the file.
            cleanupCallback();
            var outfile = fs.createWriteStream(path);

            outfile.on('close', function() {
                deferred.resolve(path);
            });

            inStream.pipe(outfile);
        }
    });

    return deferred.promise;
}

function findAvailableScenarioName(engine, currentName){
    //check if name is already there & remove trailing spaces. Create a new name if already there.
    var profiles = engine.repository.list();
    var newName = currentName.trim();//remove whitespace from front, back.
    
    var nameAlreadyExists = false;
    for (var i = 0; i < profiles.length; i++)
    {
         if (profiles[i].name == newName)
         {
             nameAlreadyExists = true;
             break;
         }
    }
    
    if (nameAlreadyExists)
    {
        //it does exist and we need to find a replacement of the form newName(1)
        var suffixIndex = 0;
        var indexName = new String();
        while (nameAlreadyExists)
        {
            nameAlreadyExists = false;
            suffixIndex++;
            indexName = newName + "("+suffixIndex.toString()+")";
            for (i = 0; i < profiles.length; i++)
            {   
                 if (profiles[i].name == indexName)
                 {
                     nameAlreadyExists = true;
                     break;
                 }
            }
        }
        newName = indexName;
    }
    return newName;
}



module.exports = constructRouter;
